/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 
package org.eclipse.mtj.core.version;

import java.util.StringTokenizer;

/**
 * Represents the version of a component.
 * Understands a string of the form "major.minor.secondary_patch".
 * <p>
 * <b>Note:</b> This class/interface is part of an interim API that is still under development and expected to
 * change before reaching stability. It is being made available at this early stage to solicit feedback
 * from pioneering adopters on the understanding that any code that uses this API will almost 
 * certainly be broken as the API evolves.
 * </p>
 * Copyright (c) 2003-2005 Craig Setera<br>
 * All Rights Reserved.<br>
 * Licensed under the Eclipse Public License - v 1.0<p/>
 * <br>
 * $Revision: 1.1.1.1 $
 * <br>
 * $Date: 2006/08/30 18:50:26 $
 * <br>
 * @author Craig Setera
 */
public class Version implements Comparable {
	private String major;
	private String minor;
	private String secondary;
	private String patch;
	private String versionString;
	
	/**
	 * Construct a new version instance based
	 * on the specified version string.
	 */
	public Version(String versionString) {
		super();
		
		this.versionString = versionString;
		major = ""; //$NON-NLS-1$
		minor = ""; //$NON-NLS-1$
		secondary = ""; //$NON-NLS-1$
		patch = ""; //$NON-NLS-1$
		
		parseVersionString(versionString);
	}

	/**
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(Object o) {
		return compareTo((Version) o);
	}

	/**
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(Version version) {
		int value = major.compareTo(version.major);
		if (value != 0) return value;

		value = minor.compareTo(version.minor);
		if (value != 0) return value;
		
		value = secondary.compareTo(version.secondary);
		if (value != 0) return value;
		
		value = patch.compareTo(version.patch);
		return value;
	}
	
	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object object) {
		return 
			(object instanceof Version) &&
			equals((Version) object);
	}
	
	/**
	 * Return a boolean indicating the equality of the specified version
	 * to this version.
	 * 
	 * @param version
	 * @return
	 */
	public boolean equals(Version version) {
		return (compareTo(version) == 0);
	}
	
	/**
	 * Get the major version number.
	 * 
	 * @return the major version string
	 */
	public String getMajor() {
		return major;
	}

	/**
	 * Get the minor version number or <code>null</code>
	 * if not specified.
	 * 
	 * @return the minor version string
	 */
	public String getMinor() {
		return minor;
	}

	/**
	 * Get the patch version number or <code>null</code>
	 * if not specified.
	 * 
	 * @return the patch version string
	 */
	public String getPatch() {
		return patch;
	}

	/**
	 * Get the secondary version number or <code>null</code>
	 * if not specified.
	 * 
	 * @return the secondary version string.
	 */
	public String getSecondary() {
		return secondary;
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return versionString;
	}
	
	/**
	 * Parse the specified version string into the
	 * constituent parts.
	 * 
	 * @param versionString
	 */
	private void parseVersionString(String vString) {
		// Peel off the patch level first, if it exists
		int index = vString.indexOf('_');
		if (index != -1) {
			patch = vString.substring(index + 1);
			vString = vString.substring(0, index);
		}
		
		StringTokenizer st = new StringTokenizer(vString, "."); //$NON-NLS-1$
		if (st.hasMoreTokens()) major = st.nextToken();
		if (st.hasMoreTokens()) minor = st.nextToken();
		if (st.hasMoreTokens()) secondary = st.nextToken();
	}
}
