/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 

package org.eclipse.mtj.internal.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.core.Launch;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.debug.core.model.IStreamsProxy;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.ITypeHierarchy;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.core.BinaryType;
import org.eclipse.jdt.internal.core.JavaModel;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.mtj.core.IEclipseMtjCoreConstants;
import org.eclipse.mtj.core.Messages;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.osgi.service.datalocation.Location;

/**
 * Various utility functions.
 * <p />
 */
public class Utils {
	// TODO This should probably be handled better... 
	private static final String PRIVATE_CONFIGURATION =
		"org.eclipse.debug.ui.private"; //$NON-NLS-1$
	
	// Copy buffer
	private static final byte[] buffer = new byte[1024];
	
	// Track whether the file system is case sensitive
	private static boolean caseSensitivityChecked;
	private static boolean caseSensitiveFileSystem;
	
	/**
	 * Return a boolean indicating whether the specified paths
	 * point to the same file.
	 * 
	 * @param path1
	 * @param path2
	 * @return
	 */
	public static boolean arePathTargetsEqual(IPath path1, IPath path2)
	{
		boolean equal = false;
		
		File file1 = getPathTargetFile(path1);
		File file2 = getPathTargetFile(path2);
		
		if ((file1 != null) && (file2 != null)) {
			equal = file1.equals(file2);
		}
		
		return equal;
	}
	
	/**
	 * Copy the specified source file into the specified target file.
	 * 
	 * @param sourceFile
	 * @param targetFile
	 * @throws IOException
	 */
	public static void copyFile(IFile sourceFile, IFile targetFile)
		throws IOException
	{
		copyFile(
				sourceFile.getLocation().toFile(),
				targetFile.getLocation().toFile());
	}
	
	/**
	 * Copy the specified source file into the specified target file.
	 * 
	 * @param sourceFile
	 * @param targetFile
	 * @throws IOException
	 */
	public static void copyFile(File sourceFile, File targetFile)
		throws IOException
	{
		FileInputStream fis = new FileInputStream(sourceFile);
		FileOutputStream fos = new FileOutputStream(targetFile);
		copyInputToOutput(fis, fos);
		fis.close();
		fos.close();
	}
	
	/**
	 * Copy the contents of the input to the output stream.
	 * 
	 * @param input
	 * @param output
	 * @throws IOException
	 */
	public static void copyInputToOutput(InputStream input, OutputStream output)
		throws IOException
	{
		int bytesRead = 0;
		
		do {
			bytesRead = input.read(buffer, 0, buffer.length);
			if (bytesRead > 0) {
				output.write(buffer, 0, bytesRead);
			}
		} while (bytesRead != -1);
		
		output.flush();
	}

	/**
	 * If the system property "eclipseme.dump.launch" is set to "true",
	 * dump the command line that was used to launch the emulator.
	 */
	public static void dumpCommandLine(ILaunch launch) {
		// Search for the process command-line
		IProcess[] processes = launch.getProcesses();
		if ((processes != null) && (processes.length > 0)) {
			IProcess process = processes[0];
			if (process != null) {
				String commandLine = process.getAttribute(IProcess.ATTR_CMDLINE);
				dumpCommandLine(commandLine);
			}
		}
	}

	/**
	 * If the system property "eclipseme.dump.launch" is set to "true",
	 * dump the command line that was used to launch the emulator.
	 * 
	 * @param commandLine
	 */
	public static void dumpCommandLine(String commandLine) {
		// Pull the dump choice from system properties
		String propValue =
			System.getProperty(IEclipseMtjCoreConstants.PROP_DUMP_LAUNCH, "false"); //$NON-NLS-1$
		boolean doDump = propValue.equalsIgnoreCase("TRUE"); //$NON-NLS-1$
	
		// Only do this if requested
		if (doDump) {
			// Let the user know what happened via the log file
			String text = (commandLine == null) ?
				Messages.Utils_CommandLineNotFound :
				Messages.Utils_CommandLine + commandLine;
			MtjCorePlugin.log(IStatus.INFO, text);
		}		
	}
	
	/**
	 * If the system property "eclipseme.dump.launch" is set to "true",
	 * dump the command line that was used to launch the emulator.
	 * 
	 * @param commandLine
	 */
	public static void dumpCommandLine(String[] commandLine) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < commandLine.length; i++) {
			String string = commandLine[i];
			if (i != 0) sb.append(' ');
			sb.append(string);
		}

		dumpCommandLine(sb.toString());
	}

	/**
	 * Exec a new process via the debug plugin.
	 * 
	 * @param commandLine
	 * @param workingDirectory
	 * @param env The environment to be used or <code>null</code>
	 * to take the default.
	 * @return
	 * @throws CoreException
	 */
	public static Process exec(String[] commandLine, File workingDirectory, String[] env) 
		throws CoreException
	{
		return DebugPlugin.exec(commandLine, workingDirectory, env);
	}
	
	/**
	 * Test if the specified executable file exist.  Add a ".exe"
	 * if necessary to handle Windows systems.
	 * 
	 * @param executableFile
	 * @return
	 */
	public static boolean executableExists(File executableFile) {
		File winExecutable = new File(
			executableFile.getParentFile(), 
			executableFile.getName() + ".exe"); //$NON-NLS-1$
		
		return 
			executableFile.exists() ||
			winExecutable.exists();
	}
	
	/**
	 * Get a new launch instance with the specified configuration
	 * name.
	 * 
	 * @param configName
	 * @return
	 * @throws CoreException
	 */
	public static ILaunch getNewLaunch(String configName) 
		throws CoreException 
	{
		return new Launch(
			getNewLaunchConfiguration(configName),
			ILaunchManager.RUN_MODE,
			null);	
	}
	
	/**
	 * Get the launch configuration to be used in building
	 * the ILaunch instance.
	 */
	public static ILaunchConfiguration getNewLaunchConfiguration(String name) 
		throws CoreException 
	{
		ILaunchManager launchManager = 
			DebugPlugin.getDefault().getLaunchManager();
		ILaunchConfigurationType configType =
			launchManager.getLaunchConfigurationType(
				IJavaLaunchConfigurationConstants.ID_JAVA_APPLICATION);
				
		ILaunchConfigurationWorkingCopy config =
			configType.newInstance(null, name);
		config.setAttribute(PRIVATE_CONFIGURATION, true);
	
		return config;	
	}

	/**
	 * Return the resource that is resolved for the specified path.
	 * This will return one of three things:
	 * <ul>
	 * <li>An instance of org.eclipse.core.resources.IResource if the entry refers to a workspace resource.</li>
	 * <li>An instance of java.io.File if the entry refers to a non-workspace resource.</li>
	 * <li><code>null</code> if the entry points to a non-existent resource.</li>
	 * </ul>
	 * 
	 * @param path the path to look for an object
	 * @return the resolved object
	 */
	public static Object getPathTarget(IPath path) {
		IWorkspaceRoot root = MtjCorePlugin.getWorkspace().getRoot();
		return JavaModel.getTarget(root, path, true);
	}

	/**
	 * Return the java.io.File instance referenced by the specified path
	 * or <code>null</code> if no such file exists.
	 * 
	 * @param entry
	 * @return
	 */
	public static File getPathTargetFile(IPath path) {
		File entryFile = null;
		
		Object pathTarget = getPathTarget(path);
		
		if (pathTarget instanceof IResource) {
			entryFile = ((IResource) pathTarget).getLocation().toFile();
		} else if (pathTarget instanceof File) {
			entryFile = (File) pathTarget;
		}
		
		return entryFile;
	}
	
	/**
	 * Attempt to retrieve the fully-qualified class name.
	 * 
	 * @param type
	 * @return
	 */
	public static String getQualifiedClassName(IType type) {
		String classname = null;
		
		if (type instanceof BinaryType) {
			classname = getQualifiedClassName((BinaryType) type);
		} else {
			classname = type.getFullyQualifiedName();
		}
		
		return classname;
	}
	
	/**
	 * Return the resource that is resolved for the specified classpath
	 * entry.  This will return one of three things:
	 * <ul>
	 * <li>An instance of org.eclipse.core.resources.IResource if the entry refers to a workspace resource.</li>
	 * <li>An instance of java.io.File if the entry refers to a non-workspace resource.</li>
	 * <li><code>null</code> if the entry points to a non-existent resource.</li>
	 * </ul>
	 * 
	 * @param entry the entry to be resolved
	 * @return the resolved object
	 */
	public static Object getResolvedClasspathEntry(IClasspathEntry entry) {
		IClasspathEntry resolved = JavaCore.getResolvedClasspathEntry(entry);
		return getPathTarget(resolved.getPath());
	}

	/**
	 * Return the java.io.File instance referenced by the specified classpath
	 * entry or <code>null</code> if no such file exists.
	 * 
	 * @param entry
	 * @return
	 */
	public static final File getResolvedClasspathEntryFile(IClasspathEntry entry) {
		IClasspathEntry resolved = JavaCore.getResolvedClasspathEntry(entry);
		return getPathTargetFile(resolved.getPath());
	}
	
	/**
	 * Launch the specified command line and return the output from 
	 * the standard output.
	 * 
	 * @param name
	 * @param commandLine
	 * @return
	 * @throws CoreException
	 */
	public static String getStandardOutput(String name, String[] commandLine) 
		throws CoreException
	{
		IProcess process = 
			Utils.launchApplication(commandLine, null, null, name, name);
		
		// Listen on the process output streams
		IStreamsProxy proxy = process.getStreamsProxy();
		
		// Wait until completion
		while (!process.isTerminated()) {
			try { Thread.sleep(1000); } catch (InterruptedException e) {};
		}
		
		return proxy.getOutputStreamMonitor().getContents();
	}
	
	/**
	 * Return a boolean indicating whether the host file system
	 * appears to be case sensitive. 
	 * 
	 * @return case sensitivity of the host file system
	 */
	public static boolean isFileSystemCaseSensitive() {
		if (!caseSensitivityChecked) {
			caseSensitivityChecked = true;
			
			Location location = Platform.getInstallLocation();
			if (location != null) {
				URL url = location.getURL();
				if (url != null) {
					String urlString = url.toString();
					if (urlString.startsWith("file:/")) { //$NON-NLS-1$
						urlString = urlString.substring("file:/".length()).toUpperCase(); //$NON-NLS-1$
						caseSensitiveFileSystem = !(new File(urlString)).exists();
					}
				}
			}
		}
		
		return caseSensitiveFileSystem;
	}
	
	/**
	 * Return a boolean indicating whether the specified type is a Midlet
	 * subclass.
	 * 
	 * @param type
	 * @return
	 */
	public static boolean isMidlet(IType type, IProgressMonitor monitor)
		throws JavaModelException 
	{
		boolean isMidlet = false;
		
		if (type != null) {
			IJavaProject javaProject = type.getJavaProject();

			if (!type.exists() && type.isBinary()) {
				// A binary type, which won't help much... Attempt to convert
				// to a source type and use that to do the lookup
				String classname = getQualifiedClassName(type);
				if (classname != null) {
					IType sourceType = javaProject.findType(classname);
					isMidlet = isMidlet(sourceType, monitor);
				}
			} else {
				ITypeHierarchy typeHierarchy = type.newSupertypeHierarchy(monitor);
				IType midletType = 
					javaProject.findType(IEclipseMtjCoreConstants.MIDLET_SUPERCLASS);
					
				isMidlet = 
					(midletType != null) &&
					typeHierarchy.contains(midletType);
			}
		}
		
		return isMidlet;
	}

	/**
	 * Launch a new application and return the IProcess
	 * representing the application.
	 * 
	 * @param commandLine
	 * @param workingDirectory
	 * @param environment
	 * @param configName
	 * @param label
	 * @return
	 * @throws CoreException
	 */
	public static IProcess launchApplication(
		String[] commandLine, 
		File workingDirectory,
		String[] environment,
		String configName,
		String label) 
			throws CoreException 
	{
		// Execute the process.
		Process execProcess = exec(commandLine, workingDirectory, environment);
			
		// Wrap it up in a JDT IProcess instance
		Launch launch = new Launch(
			getNewLaunchConfiguration(configName),
			ILaunchManager.RUN_MODE,
			null);
			
		IProcess process = DebugPlugin.newProcess(launch, execProcess, label);
		Utils.dumpCommandLine(commandLine);
		
		return process;
	}
	
	/**
	 * Private constructor
	 */
	private Utils() {
		super();
	}

	/**
	 * Attempt to retrieve the fully-qualified class name.
	 * 
	 * @param type
	 * @return
	 */
	private static String getQualifiedClassName(BinaryType type) {
		IJavaElement javaElement = type.getPackageFragment();
		StringBuffer name = new StringBuffer(type.getElementName());
		
		while (javaElement.getElementType() != IJavaElement.JAVA_PROJECT) {
			String elementName = javaElement.getElementName();
			if ((elementName != null) && (elementName.length() > 0)) {
				name.insert(0, '.').insert(0, elementName);
			}
			
			javaElement = javaElement.getParent();
		}
		
		return name.toString();
	}
}
