/*******************************************************************************
* Copyright (c) 2003,2006 IBM Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
* Contributors:
*IBM - initial API and implementation
*******************************************************************************/
package org.eclipse.mtj.jad.util;


import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.mtj.jad.Messages;


public class SynchronizeManifestFileToJad implements IRunnableWithProgress {
	public static final IPath META_INFO_PATH = new Path("META-INF/MANIFEST.MF"); //$NON-NLS-1$

	private IFile fJadFile;
	private static final Set fRequiredManifestMidletKeys = new HashSet();
	
	static {
		fRequiredManifestMidletKeys.add(JadFileDescriptor.S_NAME);
		fRequiredManifestMidletKeys.add(JadFileDescriptor.S_VENDOR);
		fRequiredManifestMidletKeys.add(JadFileDescriptor.S_VERSION);			
		fRequiredManifestMidletKeys.add(JadFileDescriptor.S_PROFILE_NAME);			
		fRequiredManifestMidletKeys.add(JadFileDescriptor.S_CONFIGURATION_NAME);		
	}
	
	private static final Set fManifestPrefixes = new HashSet();
	static {
		fManifestPrefixes.add(JadFileDescriptor.MIDLET_PREFIX);
		fManifestPrefixes.add(JadFileDescriptor.MICROEDITION_PREFIX);		
	}	

	public SynchronizeManifestFileToJad(IFile jadFile) {
		fJadFile = jadFile;
	}


	/**
	 * This method is aggressive - it replaces files instead of intelligently 
	 * merging them, which is no doubt a headache to someone somewhere.
	 */
	public void run(IProgressMonitor monitor)
		throws InvocationTargetException, InterruptedException {
		IProgressMonitor pm = new SubProgressMonitor(monitor, 5);
		try {
			pm.beginTask(Messages.SynchronizeManifestFileToJad_UpdateInfoAction, 10); 
			JadFileDescriptor descriptor =
				JadFileDescriptor.getDescriptor(fJadFile);

			IProject infoDir = getMidletsInfoFolder(fJadFile);

			IFolder manifestDir = getManifestFileFolder();
			if (!manifestDir.exists()) {
				manifestDir.create(true, true, pm);
			}

			IFile manifestFile = getManifestFile();
			if (!manifestFileIsUpToDate(manifestFile, descriptor)) {
				
				PropertyFile manifestPropFile = new PropertyFile();
				manifestPropFile.setProperty("Manifest-Version", "1.0"); //$NON-NLS-1$ //$NON-NLS-2$					
								
				manifestPropFile.setProperty(JadFileDescriptor.S_PROFILE_NAME, descriptor.getProperty(JadFileDescriptor.S_PROFILE_NAME)); 
				if (descriptor.isMidp20()) {
					//   configuration can be CLDC 1.0 or 1.1 
					manifestPropFile.setProperty(JadFileDescriptor.S_CONFIGURATION_NAME, descriptor.getProperty(JadFileDescriptor.S_CONFIGURATION_NAME)); 
				}
				else {
					manifestPropFile.setProperty(JadFileDescriptor.S_CONFIGURATION_NAME, descriptor.CLDC10);
				}
					
				if (manifestFile.exists()) {
					//keep comments to preserve blank lines (ie: section breaks)
					manifestPropFile.loadInto(manifestFile, true);
				}
				Set propertySet = new HashSet();
				propertySet.addAll(fRequiredManifestMidletKeys);
				// also add all MIDlet-N keys
				for (int i = 1; i <= descriptor.getMIDletCount(); i++) { 
					propertySet.add(JadFileDescriptor.getMIDletKey(i));
				}
				//descriptor.mergeInto(propertySet, manifestPropFile).serialize(manifestFile, pm);
				descriptor.mergeInto(propertySet, manifestPropFile);
		  		// remove the MIDlet-N keys from the manifest file not in the propertySet
				descriptor.updateManifest(propertySet, manifestPropFile);
 				manifestPropFile.serialize(manifestFile, pm);
			}
//			WorkspaceUtil.refreshOutputFolder(manifestFile.getParent());  //TODO
			IPath destination =
				infoDir.getFullPath().append(fJadFile.getName());
			// Copy the jad file : see oti bug 92260
			IFile destFile =
				ResourcesPlugin.getWorkspace().getRoot().getFile(destination);
			if (destFile.exists()) {
				PropertyFile destPropFile = JadFileDescriptor.getDescriptor(destFile);
				PropertyFile origDestFile = (PropertyFile)destPropFile.clone(); // copy prior to merge
				if (!origDestFile.equals(descriptor.mergeInto(destPropFile))) { // only write it to disk if it's changed.
					//Destination file has the wrong information, the descriptor should be saved
					//destPropFile.serialize(destFile, pm);
					descriptor.serialize(destFile, pm);
				}																
			}
			else {
				fJadFile.copy(destination, true, pm);
			}
		}
		catch (CoreException e) {
			throw new InvocationTargetException(e);
		}
		catch (IOException e) {
			throw new InvocationTargetException(e);
		}
		finally {
			pm.done();
		}
	}

	private static final String getSuiteName(IFile jadFile) {
		IPath jadPath = jadFile.getFullPath();
		return jadPath
			.removeFirstSegments(jadPath.segmentCount() - 1)
			.removeFileExtension()
			.toString();
	}

	public static IProject getMidletsInfoFolder(IFile jadFile) {
		IProject project = jadFile.getProject();
		return project;
	}


	private IFolder getManifestFileFolder() {
		IFolder folder = getMidletsInfoFolder(fJadFile).getFolder(
				META_INFO_PATH.segment(0));
		return folder;
	}

	public IFile getManifestFile() {
		return getManifestFileFolder().getFile(META_INFO_PATH.segment(1));
	}

	private boolean manifestFileIsUpToDate(
		IFile manifestFile,
		JadFileDescriptor d)
		throws IOException, CoreException {
		if (!manifestFile.exists())
			return false;

		Map manifestMidletEntries = getMidletEntriesFromManifest(manifestFile);

		// Check if Manifest contains all required JAD keys, and values are equal
		Iterator reqKeyIter = fRequiredManifestMidletKeys.iterator();
		while (reqKeyIter.hasNext()) {
			String reqKey = (String) reqKeyIter.next();
			String entryInJad = d.getProperty(reqKey);
			if (entryInJad == null)
				continue; // missing required entry in JAD file

			String entryInManifest = (String) manifestMidletEntries.get(reqKey);

			if (!entryInJad.equals(entryInManifest))
				return false;
		}

		// Check if Manifest contains all MIDlets from the JAD
		for (int index = 1; index <= d.getMIDletCount(); index++) {
			String entryInJad = d.getMIDlet(index).getValue();
			String entryInManifest =
				(String) manifestMidletEntries.get(
					JadFileDescriptor.getMIDletKey(index));
			if (!entryInJad.equals(entryInManifest))
				return false;
		}

		// Check if Manifest does not contain more MIDlets than the JAD
		String forbiddenKey =
			JadFileDescriptor.getMIDletKey(d.getMIDletCount() + 1);
		if (manifestMidletEntries.containsKey(forbiddenKey))
			return false;

		return true;
	}

	private Map getMidletEntriesFromManifest(IFile manifestFile)
		throws IOException, CoreException {
		HashMap result = new HashMap();

		PropertyFile manifest = PropertyFile.load(manifestFile, false).prefixedSubset(fManifestPrefixes);
		for (Iterator i = manifest.keyIterator(); i.hasNext(); ) {
			String key = (String)i.next();
			if (fRequiredManifestMidletKeys.contains(key)|| JadFileDescriptor.isMidletNrKey(key)) {
				result.put(key, manifest.getProperty(key));
			}					
		}		
		return result;
	}
}


