/**
 * Copyright (c) 2003-2005 Craig Setera
 * All Rights Reserved.
 * Licensed under the Eclipse Public License - v 1.0
 * For more information see http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.mtj.extension.devide.jadeditor;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.internal.debug.ui.launcher.ComboFieldEditor;
import org.eclipse.jface.preference.FieldEditor;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.IntegerFieldEditor;
import org.eclipse.jface.preference.StringFieldEditor;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.widgets.FormToolkit;

/**
 * An editor part page that may be added to the JAD editor.
 * <p />
 * Copyright (c) 2003-2005 Craig Setera<br>
 * All Rights Reserved.<br>
 * Licensed under the Eclipse Public License - v 1.0<p/>
 * <br>
 * $Revision: 1.1.1.1 $
 * <br>
 * $Date: 2006/08/30 18:51:24 $
 * <br>
 * @author Craig Setera
 */
public abstract class JADPropertiesEditorPage extends AbstractJADEditorPage {
	// The descriptors being editted
	private DescriptorPropertyDescription[] descriptors;
	
	// The field editors in use
	private FieldEditor[] fieldEditors;
	
	/**
	 * Constructor
	 */
	public JADPropertiesEditorPage(
		JADEditor editor,
		String id,
		String title,
		DescriptorPropertyDescription[] descriptors) 
	{
		super(editor, id, title);
		this.descriptors = descriptors;
	}

	/**
	 * @see org.eclipse.ui.ISaveablePart#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void doSave(IProgressMonitor monitor) {
		monitor.setTaskName(getTitle());
		
		for (int i = 0; i < fieldEditors.length; i++) {
			fieldEditors[i].store();
		}
		setDirty(false);
	}

	/**
	 * @see org.eclipse.ui.forms.editor.FormPage#createFormContent(org.eclipse.ui.forms.IManagedForm)
	 */
	protected void createFormContent(IManagedForm managedForm) {
		FormToolkit toolkit = managedForm.getToolkit();
		
		Composite composite = createSectionComposite(managedForm); 
		composite.setLayout(new GridLayout(2, false));
		new Label(composite, SWT.NONE);
		new Label(composite, SWT.NONE);
		
		fieldEditors = new FieldEditor[descriptors.length];
		for (int i = 0; i < descriptors.length; i++) {
			switch (descriptors[i].getDataType()) {
				case DescriptorPropertyDescription.DATATYPE_INT:
					fieldEditors[i] = createIntegerFieldEditor(toolkit, composite, descriptors[i]);
				break;
				
				case DescriptorPropertyDescription.DATATYPE_LIST:
					fieldEditors[i] = createComboFieldEditor(toolkit, composite, descriptors[i]);
				break;
				
				case DescriptorPropertyDescription.DATATYPE_URL:
				case DescriptorPropertyDescription.DATATYPE_STRING:
				default:
					fieldEditors[i] = createStringFieldEditor(toolkit, composite, descriptors[i]);
				break;
			}

			Label label = fieldEditors[i].getLabelControl(composite);
			toolkit.adapt(label, false, false);
			
			// Listen for property change events on the editor
			fieldEditors[i].setPropertyChangeListener(new IPropertyChangeListener() {
				public void propertyChange(PropertyChangeEvent event) {
					if (event.getProperty().equals(FieldEditor.VALUE)) {
						setDirty(true);
					}
				}
			});
		}
		
		updateEditComponents();
		addContextHelp(composite);
	}

	protected void addContextHelp(Composite c)
	{
	}
	
	/**
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	public void setFocus() {
		if (fieldEditors.length > 0) {
			fieldEditors[0].setFocus();
		}
	}
	
	/**
	 * @see org.eclipse.ui.part.EditorPart#setInput(org.eclipse.ui.IEditorInput)
	 */
	protected void setInput(IEditorInput input) {
		super.setInput(input);		
		updateEditComponents();
		setDirty(false);
	}
	
	/**
	 * Create a new combo field editor.
	 * 
	 * @param toolkit
	 * @param composite
	 * @param description
	 * @return
	 */
	private FieldEditor createComboFieldEditor(
			FormToolkit toolkit, 
			Composite composite, 
			DescriptorPropertyDescription description) 
	{
		ListDescriptorPropertyDescription listDescription =
			(ListDescriptorPropertyDescription)	description;
		
		ComboFieldEditor editor = new ComboFieldEditor(
				listDescription.getPropertyName(),
				listDescription.getDisplayName(),
				listDescription.getNamesAndValues(),
				composite);
		
		toolkit.adapt(editor.getComboBoxControl(composite), true, true);
				
		return editor;
	}

	/**
	 * Create a new integer field editor.
	 * 
	 * @param toolkit
	 * @param parent
	 * @param descriptor
	 * @return
	 */
	private IntegerFieldEditor createIntegerFieldEditor(
			FormToolkit toolkit,
			Composite parent,
			DescriptorPropertyDescription descriptor)
	{
		IntegerFieldEditor integerEditor =
			new IntegerFieldEditor(
				descriptor.getPropertyName(),
				descriptor.getDisplayName(),
				parent);
		toolkit.adapt(integerEditor.getTextControl(parent), true, true);
		
		return integerEditor;
	}

	/**
	 * Create a new String field editor.
	 * 
	 * @param toolkit
	 * @param parent
	 * @param descriptor
	 * @return
	 */
	private StringFieldEditor createStringFieldEditor(
			FormToolkit toolkit,
			Composite parent,
			DescriptorPropertyDescription descriptor)
	{
		StringFieldEditor editor =
			new StringFieldEditor(
				descriptor.getPropertyName(),
				descriptor.getDisplayName(),
				parent);
		toolkit.adapt(editor.getTextControl(parent), true, true);
		
		return editor;
	}
	
	/**
	 * Update the application descriptor the components
	 * are handling
	 */
	private void updateEditComponents() {
		if (fieldEditors != null) {
			IPreferenceStore store = getPreferenceStore();
			for (int i = 0; i < fieldEditors.length; i++) {
				FieldEditor fieldEditor = fieldEditors[i];
				fieldEditor.setPreferenceStore(store);
				fieldEditor.load();
			}
		}
	}

	/**
	 * @see org.eclipse.mtj.extension.devide.jadeditor.AbstractJADEditorPage#getSectionDescription()
	 */
	protected String getSectionDescription() {
		return null;
	}

	/**
	 * @see org.eclipse.mtj.extension.devide.jadeditor.AbstractJADEditorPage#getSectionTitle()
	 */
	protected String getSectionTitle() {
		return null;
	}

	/**
	 * @see org.eclipse.mtj.extension.devide.jadeditor.AbstractJADEditorPage#editorInputChanged()
	 */
	void editorInputChanged() {
		updateEditComponents();
	}

	/**
	 * @see org.eclipse.mtj.extension.devide.jadeditor.AbstractJADEditorPage#isManagingProperty(java.lang.String)
	 */
	boolean isManagingProperty(String property) {
		boolean manages = false;

		for (int i = 0; i < descriptors.length; i++) {
			DescriptorPropertyDescription desc = descriptors[i];
			if (property.equals(desc.getPropertyName())) {
				manages = true;
				break;
			}
		}
		return manages;
	}
}
