/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.extension.preverify;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.extension.PreverificationProvider;
import org.eclipse.mtj.api.extension.impl.MtjExtensionImpl;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.api.model.preverifier.IPreverificationPolicy;
import org.eclipse.mtj.api.model.preverifier.PreverificationError;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.exception.MtjException;
import org.osgi.framework.Bundle;

public class PreverificationProviderImpl extends MtjExtensionImpl implements
		PreverificationProvider {

	public PreverificationProviderImpl() {
		super();
		
		setId(PreverifierPlugin.getDefault().getBundle().getSymbolicName());
		setDescription(Messages.PreverificationProviderImpl_PluginDescription);
		setVendor(Messages.PreverificationProviderImpl_PluginVendor);
		setVersion(Messages.PreverificationProviderImpl_PluginVersion);
		setType(ExtensionType.PREVERIFICATION_PROVIDER_LITERAL);
	}
	
	public PreverificationError[] preverify(IMtjProject project,
			RuntimePlatformDefinition targetPlatform,
			IResource[] resource, IFolder outputFolder, IProgressMonitor monitor)
			throws MtjException {
		List errorList = new ArrayList();
		
		try {
			for ( int i = 0; i < resource.length; i++ ) {
				if (resource[i] instanceof IContainer) {
					preverify(project, targetPlatform, (IContainer) resource[i], outputFolder, errorList, monitor);
				} else {
					preverify(project, targetPlatform, (IFile) resource[i], outputFolder, errorList, monitor);
				}
			}
			return (PreverificationError[]) errorList.toArray(new PreverificationError[errorList.size()]);
		}
		catch (Exception ex) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.PREVERIFICATION_ERROR),ex);
		}
	}

	public PreverificationError[] preverifyJarFile(String projectName,
			RuntimePlatformDefinition targetPlatform,
			File jarFile, IFolder outputFolder, IProgressMonitor monitor)
			throws MtjException {
		// TODO Auto-generated method stub
		return null;
	}

	private void preverify(
			IMtjProject project,
			RuntimePlatformDefinition targetPlatform,
			IContainer container, 
			IFolder outputFolder, 
			List errorList,
			IProgressMonitor monitor) 
				throws MtjException, CoreException, IOException 
		{
			IResource[] resources = null;
			try {
				resources = container.members();
			} catch (Exception e) {
				return ;
			}
			
			for (int i = 0; i < resources.length; i++) {
				IResource resource = resources[i];
				
				if (resource.getType() == IResource.FOLDER) {
					IFolder folder = (IFolder) resource;
					preverify(project, targetPlatform, folder, outputFolder, errorList, monitor);
				} else {
					IFile file = (IFile) resource;
					preverify(project, targetPlatform, file, outputFolder, errorList, monitor);
				}
			}
		}

	/**
	 * Preverify the specified file.
	 * 
	 * @param file
	 * @param outputFolder
	 * @param params
	 * @param monitor
	 * @throws CoreException
	 * @throws IOException
	 */
	private void preverify(
		IMtjProject project,
		RuntimePlatformDefinition targetPlatform,
		IFile file, 
		IFolder outputFolder, 
		List errorList,
		IProgressMonitor monitor) 
			throws MtjException, CoreException, IOException 
	{
		// Preverify the file
		ClassPreverifier classPreverifier = getClassPreverifier(project, targetPlatform);
		InputStream contentStream = file.getContents();
		URL[] classpath = getClasspath(project);
		PreverificationResults results = classPreverifier.preverify(contentStream, classpath);
		
		// Write the output if not an errorf
		if (results.isErrorResult()) {
			errorList.addAll(Arrays.asList(results.getErrors()));
		} else {
			InputStream is = new ByteArrayInputStream(results.getPreverifiedClassBytes());
			
			IFolder packageFolder = createPackageFolder(outputFolder, results, monitor);
			IFile outputFile = packageFolder.getFile(file.getName());
			if (!outputFile.exists()) {
				outputFile.create(is, true, monitor);
			} else {
				outputFile.setContents(is, true, false, monitor);
			}
		}
	}

	/**
	 * Create a folder matching the package name.
	 * 
	 * @param outputFolder
	 * @param results
	 * @return
	 * @throws CoreException 
	 */
	private IFolder createPackageFolder(
		IFolder outputFolder, 
		PreverificationResults results,
		IProgressMonitor monitor) 
			throws CoreException 
	{
		String typeName = results.getPreverifiedClassNode().name;
		String[] components = typeName.split("/"); //$NON-NLS-1$
		IFolder currentFolder = outputFolder;
		
		for (int i = 0; i < components.length - 1; i++) {
			String component = components[i];
			currentFolder = currentFolder.getFolder(component);
			if (!currentFolder.exists()) {
				currentFolder.create(true, true, monitor);
			}
		}

		return currentFolder;
	}

	private ClassPreverifier classPreverifier = null;
	/**
	 * @return Returns the classPreverifier.
	 */
	private ClassPreverifier getClassPreverifier(IMtjProject project, RuntimePlatformDefinition targetPlatform)  throws MtjException, CoreException {
		if (classPreverifier == null) {
			classPreverifier = new ClassPreverifier(getPreverificationPolicy(project, targetPlatform));
		}
		
		return classPreverifier;
	}
	
	private IPreverificationPolicy preverificationPolicy = null;
	/**
	 * @return Returns the preverificationPolicy.
	 */
	protected IPreverificationPolicy getPreverificationPolicy(IMtjProject project, RuntimePlatformDefinition targetPlatform) throws MtjException, CoreException {
		if (preverificationPolicy == null) {
			preverificationPolicy = project.getPreverificationPolicy(targetPlatform);
		}
		return preverificationPolicy;
	}

	/**
	 * Return the classpath for the specified midlet project.
	 * 
	 * @return
	 * @throws CoreException 
	 */
	protected URL[] getClasspath(IMtjProject project) 
		throws MtjException, CoreException, IOException 
	{
		IJavaProject javaProject = project.getJavaProject();
		String[] entries = JavaRuntime.computeDefaultRuntimeClassPath(javaProject);
		URL[] classpath = new URL[entries.length + 1];
		
		classpath[0] = getEmptyAPI();
		for (int i = 0; i < entries.length; i++) {
			File file = new File(entries[i]);
			classpath[i + 1] = file.toURL();
		}

		return classpath;
	}

	/**
	 * Return a reference to the empty API used during preverification.
	 * 
	 * @return
	 * @throws IOException
	 */
	protected URL getEmptyAPI() 
		throws IOException 
	{
		Bundle bundle = PreverifierPlugin.getDefault().getBundle();
		URL emptyapi = bundle.getEntry("lib/emptyapi.zip"); //$NON-NLS-1$
		return FileLocator.resolve(emptyapi);
	}

}
