/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.core.project.midp;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;

import org.eclipse.core.internal.resources.Workspace;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.search.IJavaSearchScope;
import org.eclipse.mtj.api.deployment.Deployment;
import org.eclipse.mtj.api.enumerations.DeploymentType;
import org.eclipse.mtj.api.enumerations.ProjectType;
import org.eclipse.mtj.api.extension.ProjectExtensionProvider;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.api.model.MidletDeployment;
import org.eclipse.mtj.api.model.preverifier.IPreverificationPolicy;
import org.eclipse.mtj.api.project.Project;
import org.eclipse.mtj.api.project.TargetDevice;
import org.eclipse.mtj.api.runtimes.DeviceConfiguration;
import org.eclipse.mtj.api.runtimes.DeviceProfile;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.core.IEclipseMtjCoreConstants;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.core.jad.IJADConstants;
import org.eclipse.mtj.core.project.midp.preverification.CLDC1_0PreverificationPolicy;
import org.eclipse.mtj.core.project.midp.preverification.CLDC1_1PreverificationPolicy;
import org.eclipse.mtj.core.ui.util.MidletSearchScope;
import org.eclipse.mtj.core.util.CoreUtil;
import org.eclipse.mtj.core.version.Version;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.devide.MtjDevIdePlugin;
import org.eclipse.mtj.extension.devide.config.LibrarySpecification;
import org.eclipse.mtj.internal.utils.ColonDelimitedProperties;
import org.eclipse.mtj.internal.utils.TemporaryFileManager;
import org.eclipse.mtj.internal.utils.Utils;

public class MidpProjectExtensionProvider implements ProjectExtensionProvider {

	private static final String DEPLOYMENT_DIRECTORY = "deployed"; //$NON-NLS-1$
	private static final String LAUNCHABLE_CLASSES_FOLDER = "verified/classes"; //$NON-NLS-1$
	private static final String LAUNCHABLE_LIBS_FOLDER = "verified/libs"; //$NON-NLS-1$
	
	private IPath jadFilePath = null;

	public ProjectType getProjectType() {
		return ProjectType.PROJECT_TYPE_MIDP_LITERAL;
	}

	public Deployment getDeployment(IMtjProject project, String launchableApplication) throws MtjException {
		File jar = findFile(project.getProject(), DEPLOYMENT_DIRECTORY, "jar"); //$NON-NLS-1$
		if (jar == null ) {
			throw new MtjException(Messages.MidpProjectExtensionProvider_JarNotFound);
		}
		
		File jad = findFile(project.getProject(), DEPLOYMENT_DIRECTORY, "jad"); //$NON-NLS-1$
		if (jad == null ) {
			throw new MtjException(Messages.MidpProjectExtensionProvider_JadNotFound);
		}

		File jad2;
		try {
			jad2 = calculateJADFile(jad, jar, launchableApplication);
		} catch (CoreException e) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.MIDP_JAD_CREATION_ERROR),e);
		}
		
		File[] files = { jar , jad };
		MidletDeployment midletDeployment = (MidletDeployment)
			CoreUtil.createDeployment(files, DeploymentType.DEPLOYMENT_TYPE_MIDLET_LITERAL);
		
		return midletDeployment;
	}

	public Deployment getDeployment(IMtjProject project) throws MtjException {
		
		File jar = findFile(project.getProject(), DEPLOYMENT_DIRECTORY, "jar"); //$NON-NLS-1$
		if (jar == null ) {
			throw new MtjException(Messages.MidpProjectExtensionProvider_JarNotFound);
		}
		
		File jad = findFile(project.getProject(), DEPLOYMENT_DIRECTORY, "jad"); //$NON-NLS-1$
		if (jad == null ) {
			throw new MtjException(Messages.MidpProjectExtensionProvider_JadNotFound);
		}
		
		File[] files = { jar , jad };
		MidletDeployment midletDeployment = (MidletDeployment)
			CoreUtil.createDeployment(files, DeploymentType.DEPLOYMENT_TYPE_MIDLET_LITERAL);

		return midletDeployment;
	}

	/**
	 * Calculate the return the appropriate JAD file to use 
	 * during launching.
	 * 
	 * @param configuration
	 * @return
	 * @throws CoreException
	 */
	protected File calculateJADFile(File jadFile, File srcJar, String midletClass) 
		throws CoreException
	{
		return createSingleClassJadFile( jadFile,  srcJar,  midletClass );
	}

	/**
	 * Create and return a new temporary JAD file using the
	 * specified JAD file with the selected class to be run
	 * as the single midlet listed in the temporary JAD file.
	 * 
	 * @param configuration
	 * @param jadFile
	 * @return
	 * @throws CoreException
	 */
	private File createSingleClassJadFile(File jadFile, File srcJar, String midletClass ) 
		throws CoreException 
	{
		File tempJadFile = null;

		if (jadFile.exists()) {
			try {
				// Read the current JAD file contents...
				ColonDelimitedProperties props = new ColonDelimitedProperties();
				InputStream contents = new FileInputStream(jadFile);
				props.load(contents);
				contents.close();
				
				// Convert the JAD properties to be a single class
				getSingleClassJADProperties(midletClass, props);
				
				// Write out to a temp directory
				File tempDirectory = 
					TemporaryFileManager.instance.createTempDirectory("launch_", ".mtj"); //$NON-NLS-1$ //$NON-NLS-2$

				// Copy the predeployed jar file
				File tgtJar = new File(tempDirectory, srcJar.getName());
				Utils.copyFile(srcJar, tgtJar);

				props.setProperty("MIDlet-Jar-URL", srcJar.getName()); //$NON-NLS-1$
				tempJadFile = new File(tempDirectory, jadFile.getName());
				FileOutputStream fos = new FileOutputStream(tempJadFile);
				props.store(fos, Messages.MidpProjectExtensionProvider_TemporaryManifestFile);
				fos.close();
				
			} catch (IOException e) {
				IStatus status = new Status(
						IStatus.ERROR,
						IEclipseMtjCoreConstants.PLUGIN_ID,
						MtjCoreErrors.MIDP_JAD_CREATION_ERROR,
						MtjCoreErrors.getErrorMessage(MtjCoreErrors.MIDP_JAD_CREATION_ERROR),
						e);
				throw new CoreException(status);		
			}
		}
		
		return tempJadFile;		
	}

	/**
	 * Get a JAD properties object set up for a single class.
	 * 
	 * @param configuration
	 * @param props
	 * @throws CoreException
	 */
	private void getSingleClassJADProperties( String midletClass, ColonDelimitedProperties props) 
			throws CoreException 
	{
		// Remove the midlets that are defined in the JAD file
		for (int i = 0; i < 100; i++) {
			String key = "MIDlet-" + i; //$NON-NLS-1$
			if (props.containsKey(key)) {
				props.remove(key);
			} else {
				break;
			}
		}
		
		// Add the midlet to the JAD
		String propValue = midletClass + ",," + midletClass; //$NON-NLS-1$
		props.setProperty("MIDlet-1", propValue); //$NON-NLS-1$
	}
	
	private File findFile(IProject project, String folder2, String fileType) {
		String folder = project.getLocation().toString();
		
		final String fType = fileType;
		File f = new File(folder + "/" + folder2); //$NON-NLS-1$
		if ( f.exists() ) {
			String[] files = f.list(new FilenameFilter() {
				public boolean accept(File dir, String name) {
					return name.endsWith("."+fType); //$NON-NLS-1$
				}
			});
			if ( files != null && files.length > 0 ) {
				String fol = folder + "/" + folder2 + "/" + files[0]; //$NON-NLS-1$ //$NON-NLS-2$
				return new File(fol);
			}
		}
		
		return null;
	}
	
	public String getLaunchableClassesFolder(IProgressMonitor monitor) throws CoreException {
		return LAUNCHABLE_CLASSES_FOLDER;
	}

	public String getDeploymentFolder(IProgressMonitor monitor) throws CoreException {
		return DEPLOYMENT_DIRECTORY;
	}

	public String getLaunchableLibsFolder(IProgressMonitor monitor) throws CoreException {
		return LAUNCHABLE_LIBS_FOLDER;
	}

	public class MtjFile extends org.eclipse.core.internal.resources.File {
		public MtjFile(IPath path, Workspace container) {
			super(path, container);
		}
		public IPath getLocation() {
			return getLocalManager().locationFor(this);
		}
	}
	
	/**
	 * Return DeploymentType that the ProjectExtensionProvider is supporting
	 * 
	 * @return
	 * @throws MtjException
	 */
	public DeploymentType getDeploymentType() throws MtjException {
		return DeploymentType.DEPLOYMENT_TYPE_MIDLET_LITERAL;
	}

	/**
	 * Return IJavaSearchScope object for searching classes that support the project's scope
	 * 
	 * @param javaProject
	 * @return
	 * @throws MtjException
	 */
	public IJavaSearchScope getJavaSearchScope(IJavaProject javaProject) throws MtjException {
		try {
			return new MidletSearchScope(javaProject);
		} catch (JavaModelException e) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_EXTENSION_DATA_ERROR), e);
		}
	}
	
	/**
	 * Returns PreverificationPolicy that the project supports
	 * 
	 * @param targetPlatform
	 * @return
	 */
	public IPreverificationPolicy getPreverificationPolicy(RuntimePlatformDefinition targetPlatform) throws MtjException {
		Version version = null;
		
		Iterator it = targetPlatform.getDeviceConfigurations().iterator();
		while ( it.hasNext() ) {
			DeviceConfiguration dc = (DeviceConfiguration)it.next();
			if ( dc.getVersion() != null ) {
				Version v = new Version(dc.getVersion());
				if (version == null || version.compareTo(v) > 0 ) {
					version = v;
				}
			}
		}
		if (version == null || version.getMinor().equals("0")) { //$NON-NLS-1$
			return new CLDC1_0PreverificationPolicy();
		}
		else {
			return new CLDC1_1PreverificationPolicy();
		}
	}

	/**
	 * Method initialized new Java Project project with the extension type -specific features
	 * 
	 * @param javaProject
	 * @throws MtjException
	 */
	public void initializeNewProject(IJavaProject javaProject, IProgressMonitor monitor) throws MtjException {
//		// Check the JAD file for existence
//		IFile jadFile = getJadFile(javaProject);
//		if (!jadFile.exists()) {
//			if (jadFilePath != null && jadFilePath.segmentCount() > 1) {
//				IContainer container = javaProject.getProject();
//				for (int i = 0; i < (jadFilePath.segmentCount() - 1); i++) {
//					IPath folderPath = new Path(jadFilePath.segment(i));
//					IFolder folder = container.getFolder(folderPath);
//					if (!folder.exists()) {
//						try {
//							folder.create(true, true, monitor);
//						} catch (CoreException e) {
//							throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_EXTENSION_DATA_ERROR), e);
//						}
//					}
//					
//					container = folder;
//				}
//			}
//			
//			InputStream is;
//			try {
//				is = getJADFileSource(javaProject);
//				jadFile.create(is, true, monitor);
//			} catch (MtjException e) {
//				throw e; 
//			} catch (Exception e) {
//				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_EXTENSION_DATA_ERROR), e);
//			}
//		}		
	}
	
//	/**
//	 * Get the bytes that make up the contents of the to-be created JAD
//	 * file.
//	 * 
//	 * @param midletSuite
//	 * @return
//	 * @throws IOException
//	 * @throws CoreException
//	 * @throws MtjException 
//	 */
//	private InputStream getJADFileSource(IJavaProject javaProject)
//		throws IOException, CoreException, MtjException
//	{
//		// The InputStream to be returned...
//		InputStream is = null;
//		
//		// Get the application descriptor and write it to 
//		// a byte array
//		ByteArrayOutputStream bos = new ByteArrayOutputStream();
//		ColonDelimitedProperties defaultProps = 
//			getDefaultApplicationDescriptorProperties(MtjProject.getMtjProject(javaProject));
//		defaultProps.store(bos, "");
//		
//		// Use the byte array as the source to create the
//		// new file
//		is = new ByteArrayInputStream(bos.toByteArray());
//		
//		return is;
//	}
//	
//	private IFile getJadFile(IJavaProject javaProject) {
//		IFile jadFile = null;
//		
//		IProject project = javaProject.getProject();
//		String filename = getProjectNameWithoutSpaces(javaProject) + ".jad";
//		jadFile = project.getFile(filename);
//		
//		return jadFile;
//	}
	
	private String getJarFileName(IJavaProject javaProject) {
		return getProjectNameWithoutSpaces(javaProject) + ".jar"; //$NON-NLS-1$
	}
	
	/**
	 * Get the project name, replacing spaces with underscores.
	 * 
	 * @return
	 */
	private String getProjectNameWithoutSpaces(IJavaProject javaProject)
	{
		String projectName = javaProject.getProject().getName();
		return projectName.replace(' ', '_');
	}

	private ColonDelimitedProperties getDefaultApplicationDescriptorProperties(IMtjProject mtjproject) throws CoreException, MtjException
	{
		ColonDelimitedProperties descriptor = new ColonDelimitedProperties();
		
		// Do the OTA URL
		descriptor.setProperty(
			IJADConstants.JAD_MIDLET_JAR_URL,
			getJarFileName(mtjproject.getJavaProject()));
	// KMH:
		// Couple of names...
		descriptor.setProperty(
			IJADConstants.JAD_MIDLET_NAME,
			mtjproject.getProject().getName() + " Midlet Suite"); //$NON-NLS-1$
		descriptor.setProperty(
			IJADConstants.JAD_MIDLET_VENDOR,
			Messages.MidpProjectExtensionProvider_MidletSuiteVendor);
		descriptor.setProperty(
			IJADConstants.JAD_MIDLET_VERSION,
			"1.0.0"); //$NON-NLS-1$
			
		DeviceConfiguration dc = null;
		DeviceProfile dp = null;
		Project project = mtjproject.getProjectData();
		TargetDevice targetDevice = project.getDefaultTargetDevice();
		if  (targetDevice != null) {
			dc = (DeviceConfiguration)
				targetDevice.getRuntimePlatform().getDeviceConfigurations().get(0);
			dp = (DeviceProfile)
				targetDevice.getRuntimePlatform().getDeviceProfiles().get(0);
		
			String configVersion = null;
			LibrarySpecification ls = getCorrConfiguration(dc);
			if ( ls != null ) {
				configVersion = ls.getIdentifier();
			}
			if (configVersion != null) {
				descriptor.setProperty(
						IJADConstants.JAD_MICROEDITION_CONFIG,
						configVersion);
			}
	
			
			String profileVersion = null;
			ls = getCorrProfile(dp);
			if ( ls != null ) {
				profileVersion = ls.getIdentifier();
			}
			if (profileVersion != null) {
				descriptor.setProperty(
						IJADConstants.JAD_MICROEDITION_PROFILE,
						profileVersion);
			}
		
		}
		
		return descriptor;
	}

	private LibrarySpecification getCorrConfiguration(DeviceConfiguration dc) throws CoreException {
		LibrarySpecification[] lss = MtjDevIdePlugin.getConfigurationSpecifications();
		for (int i = 0; i < lss.length; i++) {
			Version v = new Version(dc.getVersion());
			if ( lss[i].getVersion().equals(v) ) {
				return lss[i];
			}
		}
		return null;
	}
	
	private LibrarySpecification getCorrProfile(DeviceProfile dp) throws CoreException {
		LibrarySpecification[] lss = MtjDevIdePlugin.getProfileSpecifications();
		for (int i = 0; i < lss.length; i++) {
			Version v = new Version(dp.getVersion());
			if ( lss[i].getVersion().equals(v) ) {
				return lss[i];
			}
		}
		return null;
	}
	
}
