/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.core.util;

import java.io.File;

import org.eclipse.mtj.api.deployment.Deployment;
import org.eclipse.mtj.api.enumerations.DeploymentType;
import org.eclipse.mtj.api.model.CdcDeployment;
import org.eclipse.mtj.api.model.MidletDeployment;
import org.eclipse.mtj.api.model.OsgiAwareMidletDeployment;
import org.eclipse.mtj.api.model.ResourceDeployment;
import org.eclipse.mtj.core.Messages;
import org.eclipse.mtj.exception.MtjException;

public class CoreUtil {
	/**
	 * Method creates Deployment -instance from the resource -files which type is deploymentType.<br>
	 * Input resources of different deployment types are:
	 * <p>
	 * <h3>Midlet Deployment</h3><br>
	 * <table>
	 *  <tr>
	 *   <td>position</td><td>type</td>
	 *  </tr>
	 *  <tr>
	 *   <td>0</td><td>jar file</td>
	 *  </tr>
	 *  <tr>
	 *   <td>1</td><td>jad file</td>
	 *  </tr>
	 * </table>
	 * 
	 * <p>
	 * <h3>Osgi Aware Midlet Deployment</h3><br>
	 * One Bundle file in array position 0.
	 * 
	 * <p>
	 * <h3>CDC Deployment</h3><br>
	 * Jar files.
	 * 
	 * <p>
	 * <h3>Resource Deployment</h3><br>
	 * Array of files.
	 * 
	 * @param resources
	 * @param deploymentType
	 * @return
	 * @throws MtjException
	 */
	public static Deployment createDeployment(File[] resources, DeploymentType deploymentType) throws MtjException {
		
		if ( deploymentType == DeploymentType.DEPLOYMENT_TYPE_MIDLET_LITERAL ) {
			return createMidletDeployment(resources);
		}
		else if ( deploymentType == DeploymentType.DEPLOYMENT_TYPE_OSGI_MIDLET_LITERAL ) {
			return createOsgiDeployment(resources);
		}
		else if ( deploymentType == DeploymentType.DEPLOYMENT_TYPE_CDC_LITERAL ) {
			return createCdcDeployment(resources);
		}
		else if ( deploymentType == DeploymentType.DEPLOYMENT_TYPE_RESOURCE_LITERAL ) {
			return createResourceDeployment(resources);
		}
		
		return null;
	}
	
	private static Deployment createOsgiDeployment(File[] resources) throws MtjException {
		String msg = Messages.CoreUtil_FileArrayMustContainOnlyOneBundle;
		
		if (resources.length != 1) {
			throw new MtjException(msg);
		}
		OsgiAwareMidletDeployment o = new OsgiAwareMidletDeployment();
		o.setType(DeploymentType.DEPLOYMENT_TYPE_OSGI_MIDLET_LITERAL);
		o.setBundle(resources[0]);
		return o;
	}

	private static Deployment createCdcDeployment(File[] resources) throws MtjException {
		String msg = Messages.CoreUtil_FileArrayMustBeJarFIles;
		for (int i = 0; i < resources.length; i++) {
			if (!DirUtil.getFileExtension(resources[i].getName()).equalsIgnoreCase("jar")) { //$NON-NLS-1$
				throw new MtjException(msg);
			}
		}
		CdcDeployment c = new CdcDeployment();
		c.setType(DeploymentType.DEPLOYMENT_TYPE_CDC_LITERAL);
		c.setJars(resources);
		return c;
	}

	private static Deployment createResourceDeployment(File[] resources) throws MtjException {
		ResourceDeployment rd = new ResourceDeployment();
		rd.setType(DeploymentType.DEPLOYMENT_TYPE_RESOURCE_LITERAL);
		rd.setResources(resources);
		return rd;
	}

	private static Deployment createMidletDeployment(File[] resources) throws MtjException {
		String msg = Messages.CoreUtil_FileArrayMustContainJarAndJadLocations;
		
		if (resources.length != 2) {
			throw new MtjException(msg);
		}
		if (!DirUtil.getFileExtension(resources[0].getName()).equalsIgnoreCase("jar")) { //$NON-NLS-1$
			throw new MtjException(msg);
		}
		if (!DirUtil.getFileExtension(resources[1].getName()).equalsIgnoreCase("jad")) { //$NON-NLS-1$
			throw new MtjException(msg);
		}
		MidletDeployment md = new MidletDeployment();
		md.setType(DeploymentType.DEPLOYMENT_TYPE_MIDLET_LITERAL);
		md.setJar(resources[0]);
		md.setJad(resources[1]);
		return md;
	}

}
