/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 
package org.eclipse.mtj.extension.devide.launching;
        
import java.io.File;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.launching.AbstractJavaLaunchConfigurationDelegate;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.jdt.launching.VMRunnerConfiguration;
import org.eclipse.mtj.api.deployment.Deployment;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.extension.DeviceManagement;
import org.eclipse.mtj.api.extension.DevicePlatformProvider;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.core.IEclipseMtjCoreConstants;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.mtj.core.MtjServices;
import org.eclipse.mtj.core.launching.ILaunchConstants;
import org.eclipse.mtj.core.launching.LaunchHelper;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.devide.Messages;
import org.eclipse.mtj.extension.devide.action.ToggleProjectBuildAction;
import org.eclipse.mtj.extension.devide.nature.MtjNature;
import org.eclipse.mtj.extension.devide.project.MtjProject;
import org.eclipse.mtj.internal.utils.Utils;

/**
 * Launch configuration for launching a Wireless Toolkit emulator
 * that launches as an executable rather than a Java class.
 */
public class EmulatorLaunchConfigDelegate extends
	AbstractJavaLaunchConfigurationDelegate 
{
	/** Status code for which the no midlet status is registered. */
	private static final IStatus NO_MIDLET_STATUS = new Status(
			IStatus.ERROR, 
			IEclipseMtjCoreConstants.PLUGIN_ID, 
			IEclipseMtjCoreConstants.ERR_OTA_NO_MIDLETS, 
			"", null); //$NON-NLS-1$
	
	/**
	 * Construct a new Executable emulator launch config delegate. 
	 */
	public EmulatorLaunchConfigDelegate() {
		super();
	}

	/**
	 * @see eclipseme.core.launching.AbstractEmulatorLaunchConfigDelegate#getVMRunnerConfiguration(org.eclipse.debug.core.ILaunchConfiguration, java.lang.String)
	 */
	protected VMRunnerConfiguration getVMRunnerConfiguration(
		ILaunchConfiguration launchConfig, 
		DevicePlatform platformDef,
		String mode) 
			throws CoreException 
	{
		File workingDir = verifyWorkingDirectory(launchConfig);
		String workingDirName = null;
		if (workingDir != null) {
			workingDirName = workingDir.getAbsolutePath();
		}
		
		// Create VM config
		VMRunnerConfiguration runConfig = 
			new VMRunnerConfiguration(platformDef.getName(), new String[0]);
		runConfig.setVMArguments(new String[0]);
		runConfig.setWorkingDirectory(workingDirName);
		runConfig.setVMSpecificAttributesMap(new HashMap());

		return runConfig;
	}

	/**
	 * @see org.eclipse.debug.core.model.ILaunchConfigurationDelegate#launch(org.eclipse.debug.core.ILaunchConfiguration, java.lang.String, org.eclipse.debug.core.ILaunch, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void launch(
			ILaunchConfiguration launchConfig, 
			String mode, 
			ILaunch launch, 
			IProgressMonitor monitor) 
				throws CoreException 
	{
		// Make sure we have a progress monitor
		if (monitor == null) {
			monitor = new NullProgressMonitor();
		}
		
		// Get the platform definition and test that it isn't
		// the unspecified platform definition
		DevicePlatform platformDef = null;
		try {
			// perform MTJ build
			(new ToggleProjectBuildAction()).run(launchConfig);

			platformDef = getPlatformDefinition(launchConfig);
		} catch (Exception e) {
			IStatus status = new Status(
					IStatus.ERROR, 
					IEclipseMtjCoreConstants.PLUGIN_ID, 
					MtjCoreErrors.TARGET_PLATFORM_ACCESS_ERROR, 
					MtjCoreErrors.getErrorMessage(MtjCoreErrors.TARGET_PLATFORM_ACCESS_ERROR), e);
			throw new CoreException(status);
		}
		
		if (platformDef == null) {
			// The platform definition isn't available, so we can't
			// launch
			MtjCorePlugin.throwCoreException(
					IStatus.ERROR, -999, 
					Messages.EmulatorLaunchConfigDelegate_Platform_definition_unspecified); 
		}
		
		// In the case that a predeployment is necessary, go ahead
		// and do that now.
		Deployment deployment = getDeployment(launchConfig, isPredeployNecessary(launchConfig, platformDef));
		
		Object[] obj = new Object[] { launchConfig.getName() };
		monitor.beginTask(
			MessageFormat.format(
				Messages.EmulatorLaunchConfigDelegate_launching, 
				obj), 
			    3);
		
		// check for cancellation
		if (monitor.isCanceled()) {
			return;
		}
		
		// Set up the VM runner
		monitor.subTask(
				Messages.EmulatorLaunchConfigDelegate_verifying_launch_attributes);
		
		// Create VM config
		VMRunnerConfiguration runConfig = 
			getVMRunnerConfiguration(launchConfig, platformDef, mode);
		
		// check for cancellation
		if (monitor.isCanceled()) {
			return;
		}		
		
		// done the verification phase
		monitor.worked(1);
		
		// Handle the source locator support
		monitor.subTask(
				Messages.EmulatorLaunchConfigDelegate_creating_source_locator);
	
		setDefaultSourceLocator(launch, launchConfig);
		monitor.worked(1);		
		
		try {			
			Device device = getDefaultDevice(launchConfig, platformDef);
		
			// Launch the configuration - 1 unit of work
			DevicePlatformProvider dpp = getDevicePlatformProvider(platformDef);

			dpp.launch(
					 platformDef,
					 device,
					 deployment,
					 runConfig,
					 launchConfig,
					 launch,
					 mode,
					 MtjNature.getNatureID(),
					 monitor);
		} catch (Exception e) {
			IStatus status = new Status(
					IStatus.ERROR, 
					IEclipseMtjCoreConstants.PLUGIN_ID, 
					IJavaLaunchConfigurationConstants.ERR_INTERNAL_ERROR, 
					"", e); //$NON-NLS-1$
			throw new CoreException(status);
		}
		
		// Dump the command line if requested by system property
		Utils.dumpCommandLine(launch);
		
		// check for cancellation
		if (!monitor.isCanceled()) {
			monitor.done();
		}	
	}

	private DevicePlatformProvider getDevicePlatformProvider(DevicePlatform devicePlatform) throws MtjException {
		return getDeviceManagement().getDevicePlatformProvider(devicePlatform);
	}
	
	/**
	 * @see org.eclipse.debug.core.model.ILaunchConfigurationDelegate2#preLaunchCheck(org.eclipse.debug.core.ILaunchConfiguration, java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public boolean preLaunchCheck(
		ILaunchConfiguration configuration,
		String mode, 
		IProgressMonitor monitor) 
			throws CoreException 
	{
		boolean continueLaunch = super.preLaunchCheck(configuration, mode, monitor);
		if (continueLaunch) {
			continueLaunch = verifyEmulationSettings(configuration, monitor);
		}
		
		return continueLaunch;
	}

	/**
	 * Deploy the application so that it is available to the target platform.
	 * 
	 * @param launchConfig
	 * @param monitor
	 * @throws CoreException
	 */
	private Deployment getDeployment(ILaunchConfiguration launchConfig, boolean isPredeployNecessary) 
		throws CoreException 
	{
		IMtjProject suite = getProjectSuite(launchConfig);
		if (suite != null) {
			if (isPredeployNecessary) {
				return suite.getDeployment(LaunchHelper.getTargetApplication(launchConfig));
			}
			else {
				return suite.getDeployment();
			}
		}
		return null;
	}

	/**
	 * Return projects deployment
	 * 
	 * @param launchConfig
	 * @return
	 * @throws CoreException
	 */
	private Deployment getDeployment(ILaunchConfiguration launchConfig) 
	throws CoreException 
{
	IMtjProject suite = getProjectSuite(launchConfig);
	if (suite != null) {
		return suite.getDeployment();
	}
	return null;
}

	/**
	 * Return a boolean indicating whether the emulator requires the
	 * suite to be predeployed.
	 * 
	 * @param platformDef
	 * @param launchConfig
	 * @return
	 * @throws CoreException
	 */
	private boolean doesEmulatorRequirePredeploy(
		ILaunchConfiguration launchConfig, 
		DevicePlatform platformDef) 
			throws CoreException 
	{
		return platformDef.isPredeploymentRequired();
	}

	/**
	 * Return the midlet suite that is selected for this launch.
	 * 
	 * @param launchConfig
	 * @return
	 * @throws CoreException
	 */
	private IMtjProject getProjectSuite(ILaunchConfiguration launchConfig) 
		throws CoreException
	{
		IJavaProject javaProject = getJavaProject(launchConfig);
		try {
			return MtjProject.getMtjProject(javaProject);
		} catch (MtjException e) {
			// timeout, consult status handler if there is one
			IStatus status = new Status(
				IStatus.ERROR, 
				IEclipseMtjCoreConstants.PLUGIN_ID, 
				MtjCoreErrors.PROJECT_INSTANTIATION_ERROR, 
				MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_INSTANTIATION_ERROR), e);
			throw new CoreException(status);
		}
	}
	
	private Device getDefaultDevice(ILaunchConfiguration launchConfig, DevicePlatform platformDef) throws CoreException, MtjException  {
		boolean useProjectDef = 
			launchConfig.getAttribute(ILaunchConstants.USE_PROJECT_PLATFORM_DEF, true);

		if ( useProjectDef ) {
			IJavaProject javaProject = getJavaProject(launchConfig);
			IMtjProject mtjProject = 
				MtjProject.getMtjProject(javaProject);
			return mtjProject.getDefaultDevice();
		}
		else {
			String deviceName = launchConfig.getAttribute(ILaunchConstants.EMULATED_DEVICE, ""); //$NON-NLS-1$
			List list = platformDef.getDevices();
			Iterator it = list.iterator();
			while (it.hasNext()) {
				Device device = (Device)it.next();
				if ( device.getName().equals(deviceName)) {
					return device;
				}
			}
			throw new MtjException(Messages.EmulatorLaunchConfigDelegate_0 + deviceName); //$NON-NLS-2$
		}
		
	}
	
	private void getProjectExtension() {
		
	}
	
	/**
	 * Get the platform definition for the launch configuration.
	 * 
	 * @param launchConfig
	 * @return
	 * @throws CoreException
	 * @throws MtjException 
	 */
	private DevicePlatform getPlatformDefinition(ILaunchConfiguration launchConfig) 
		throws CoreException, MtjException 
	{
		DevicePlatform platformDef = null;
		
		boolean useProjectDef = 
			launchConfig.getAttribute(ILaunchConstants.USE_PROJECT_PLATFORM_DEF, true);
		
		if (useProjectDef) {
			// Set up the midlet project so it is available throughout
			IJavaProject javaProject = getJavaProject(launchConfig);
			IMtjProject mtjProject = 
				MtjProject.getMtjProject(javaProject);
			platformDef = mtjProject.getPlatformDefinition();
			
		} else {
				String defName = launchConfig.getAttribute(
						ILaunchConstants.EMULATION_PLATFORM_DEF, 
						""); //$NON-NLS-1$
				DeviceManagement dm = getDeviceManagement();
				platformDef = dm.getDevicePlatformByName(defName);
				
				if (platformDef == null) {
					MtjCorePlugin.throwCoreException(
						IStatus.ERROR, 
						-99999, 
						Messages.EmulatorLaunchConfigDelegate_Launch_platform_definition_invalid);
				}
		}
		
		return platformDef;
	}
	
	private DeviceManagement getDeviceManagement() {
		DeviceManagement dm = (DeviceManagement)
			MtjServices.getInstance().getImplementations(ExtensionType.DEVICE_MANAGEMENT_LITERAL,null,null)[0];
		return dm;
	}
	
	/**
	 * Verify the current emulation settings before launching.  Return a
	 * boolean indicating whether the specified settings are valid.
	 * 
	 * @param configuration
	 * @param monitor
	 * @return
	 * @throws CoreException
	 */
	private boolean verifyEmulationSettings(
		ILaunchConfiguration configuration, 
		IProgressMonitor monitor) 
			throws CoreException 
	{
		boolean valid = true;
		
		// If this is OTA, there should be at least one midlet
		// defined in the JAD file.
		// TODO
		
		return valid;
	}

	/**
	 * Return a boolean indicating whether the midlet suite needs to be deployed
	 * prior to emulation.
	 * 
	 * @param launchConfig
	 * @param platformDef
	 * @return
	 * @throws CoreException
	 */
	private boolean isPredeployNecessary(
		ILaunchConfiguration launchConfig, 
		DevicePlatform platformDef) 
			throws CoreException 
	{
		return 
			isDeploymentOutOfDate(launchConfig) &&
			doesEmulatorRequirePredeploy(launchConfig, platformDef);
	}
	
	/**
	 * Return a boolean indicating whether or not the currently deployed
	 * jar is out of date.
	 * 
	 * @param launchConfig
	 * @return
	 * @throws CoreException
	 */
	private boolean isDeploymentOutOfDate(ILaunchConfiguration launchConfig) 
		throws CoreException 
	{
		IMtjProject projectSuite = getProjectSuite(launchConfig);
		return !projectSuite.isDeployedJarUpToDate();
	}

	/**
	 * Return a boolean concerning whether the user wants to continue the launch.
	 * 
	 * @return
	 * @throws CoreException
	 */
	private boolean promptWhetherToContinue()
		throws CoreException
	{
		Boolean shouldContinue = 
			(Boolean) MtjCorePlugin.statusPrompt(NO_MIDLET_STATUS, this);
		return (shouldContinue != null) ? shouldContinue.booleanValue() : false;
	}
}
