/*******************************************************************************
* Copyright (c) 2005,2006 Nokia Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
 * Additional Contributors - 
 * 		Kevin Horowitz (IBM Corp.) - Make getFolder public.
 *******************************************************************************/
package org.eclipse.mtj.extension.devide.project;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.jdt.core.IClasspathContainer;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.search.IJavaSearchScope;
import org.eclipse.mtj.api.deployment.Deployment;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.enumerations.DeploymentType;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.enumerations.ProjectType;
import org.eclipse.mtj.api.extension.DeviceManagement;
import org.eclipse.mtj.api.extension.PersistentStoreProvider;
import org.eclipse.mtj.api.extension.ProjectExtensionProvider;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.api.model.preverifier.IPreverificationPolicy;
import org.eclipse.mtj.api.project.Project;
import org.eclipse.mtj.api.project.TargetDevice;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.core.IEclipseMtjCoreConstants;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.core.MtjServices;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.devide.Messages;
import org.eclipse.mtj.extension.devide.MtjDevIdePlugin;
import org.eclipse.mtj.extension.devide.core.J2MEClasspathContainer;
import org.eclipse.mtj.extension.devide.nature.MtjNature;

public class MtjProject implements IMtjProject {

	/** Project data is persisted to the file with the name */
	public static final String PROJECT_DATA_FILE_NAME = "mtj.project"; //$NON-NLS-1$

	/** The verified subdirectory for classes */
	public static final String CLASSES_DIRECTORY = "classes"; //$NON-NLS-1$

	/** The verified subdirectory for libraries */
	public static final String LIBS_DIRECTORY = "libs"; //$NON-NLS-1$

	// Storage of the previously created midlet suite projects
	private static final Map projectSuiteMap = new HashMap();

	// The up-to-date flag for the deployed jar file
	private static final QualifiedName DEPLOYMENT_UP_TO_DATE_PROP =
		new QualifiedName(IEclipseMtjCoreConstants.PLUGIN_ID, "deployed_up_to_date"); //$NON-NLS-1$
	
	// The java project on which this midlet suite is based
	private IJavaProject javaProject;
	private ProjectExtensionProvider projectExtension;
	
	/**
	 * Return the midlet suite project instance for the specified java project.
	 * 
	 * @param javaProject
	 *            the Java project to retrieve the midlet suite wrapper
	 * @return the midlet suite wrapper
	 */
	public static IMtjProject getMtjProject(IJavaProject javaProject) throws MtjException {
		try {
			if (javaProject == null)
				return null;
			if (!javaProject.exists())
				return null;
			if ( MtjNature.hasMtjNature(javaProject.getProject()) ) {
				return getMtjProject(javaProject, null);
			}
		} catch (CoreException e) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_INSTANTIATION_ERROR), e);
		} catch (MtjException e) {
			throw e;
		}
		return null;
	}
	
	/**
	 * Return the midlet suite project instance for the specified java project.
	 * 
	 * @param javaProject
	 *            the Java project to retrieve the midlet suite wrapper
	 * @return the midlet suite wrapper
	 */
	public static IMtjProject getMtjProject(IProject project) throws MtjException {
		IJavaProject javaProject = JavaCore.create(project);
		return MtjProject.getMtjProject(javaProject);
	}
	
	/**
	 * Return the midlet suite project instance for the specified java project.
	 * 
	 * @param javaProject
	 *            the Java project to retrieve the midlet suite wrapper
	 * @return the midlet suite wrapper
	 */
	public static IMtjProject getMtjProject(String projectName) throws MtjException {
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		return MtjProject.getMtjProject(project);
	}
	
	
	
	/**
	 * Return the midlet suite project instance for the specified java project.
	 * 
	 * @param javaProject
	 *            the Java project to retrieve the midlet suite wrapper
	 * @return the midlet suite wrapper
	 */
	public static IMtjProject getMtjProject(IJavaProject javaProject, ProjectExtensionProvider projectExtension) throws MtjException {
		IMtjProject suite = null;
		
		synchronized (projectSuiteMap) {
			suite = (IMtjProject) projectSuiteMap.get(javaProject);
			if (suite == null) {
				suite = new MtjProject(javaProject);
				projectSuiteMap.put(javaProject, suite);
			}
		}

		if ( projectExtension != null ) {
			suite.setProjectExtensionProvider(projectExtension);
		}
		else {
			Project project = suite.getProjectData();
			if (project == null) {
				throw new MtjException(Messages.MtjProject_Project_data_not_found);
			}
			ProjectType projectType = project.getType();
			if (projectType == null) {
				throw new MtjException(Messages.MtjProject_Project_type_not_defined);
			}
			
			ProjectExtensionProvider projectExtensionProvider = 
				MtjServices.getInstance().getProjectExtension(projectType);
			suite.setProjectExtensionProvider(projectExtensionProvider);
		}
		return suite;
	}

	/**
	 * Method sets projectExtension to the project
	 * 
	 * @param projectExtension
	 */
	public void setProjectExtensionProvider(ProjectExtensionProvider projectExtension) {
		this.projectExtension = projectExtension;
	}
	
	/**
	 * Private constructor for singleton instances of the class.
	 */
	private MtjProject(IJavaProject javaProject) {
		super();
		this.javaProject = javaProject;
	}

	public IFolder getVerifiedClassesOutputFolder(IProgressMonitor monitor)
			throws CoreException {
		String outputFolderName = projectExtension.getLaunchableClassesFolder(monitor);

		return getFolder(outputFolderName, monitor);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.mtj.api.model.IMtjProject#getFolder(java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IFolder getFolder(String folderName, IProgressMonitor monitor) throws CoreException {
		IFolder ret = null;
		String folder = folderName.replace('\\','/');
		StringTokenizer st = new StringTokenizer(folder, "/"); //$NON-NLS-1$
		while ( st.hasMoreTokens() ) {
			String fl = st.nextToken();
			if ( ret == null ) {
				ret = getProject().getFolder(fl);
			}
			else {
				ret = ret.getFolder(fl);
			}
			if (!ret.exists()) {
				ret.create(true, true, monitor);
			}
		}
		return ret;
	}
	
	public IProject getProject() {
		return javaProject.getProject();
	}

	public IJavaProject getJavaProject() {
		return javaProject;
	}

	public IFolder getDeploymentFolder(IProgressMonitor monitor)
			throws CoreException {
		String outputFolderName = projectExtension
				.getDeploymentFolder(monitor);

		IFolder outputFolder = getProject().getFolder(outputFolderName);

		if (!outputFolder.exists()) {
			outputFolder.create(true, true, monitor);
		}

		return outputFolder;
	}

	public IFolder getVerifiedLibrariesOutputFolder(IProgressMonitor monitor)
			throws CoreException {
		String outputFolderName = projectExtension.getLaunchableLibsFolder(monitor);

		return getFolder(outputFolderName, monitor);
	}
	
	public IFolder getClassesOutputFolder(IProgressMonitor monitor) throws CoreException {
		IPath outputLocation = getJavaProject().getOutputLocation();
		IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
		IFolder outputFolder;
		
		try {
			outputFolder = workspaceRoot.getFolder(outputLocation);
		} catch (IllegalArgumentException e) { // We didn't find a folder
			outputFolder = workspaceRoot.getFolder(outputLocation.append ("preverifyOutput")); //$NON-NLS-1$
		}
		
		return outputFolder;
	}
	
	/**
	 * Get the DevicePlatform in use for this project
	 * 
	 * @return the currently associated platform definition
	 */
	public DevicePlatform getPlatformDefinition() throws MtjException {
		
		try {
			Project project = getProjectData();

			if (project != null) {
				TargetDevice td = project.getDefaultTargetDevice();
				if ( td != null ) {
					String name = td.getDevicePlatformName();
					if ( name != null ) {
						DeviceManagement dm = (DeviceManagement)
							MtjServices.getInstance().getImplementations(ExtensionType.DEVICE_MANAGEMENT_LITERAL,null,null)[0];
						DevicePlatform dp = dm.getDevicePlatformByName(name);
						if ( dp == null ) {
							throw new MtjException(Messages.MtjProject_Device_platform_does_not_exist + name );
						}
						return dp;
					}
				}
			}
		}
		catch (Exception ex) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR),ex);
		}
		throw new MtjException(Messages.MtjProject_Device_platform_not_found);
	}

	public Device[] getDevices() throws MtjException {
		ArrayList list = new ArrayList();
		try {
			Project project = getProjectData();
			if (project != null) {
				Iterator it = project.getTargetDevices().iterator();
				while (it.hasNext()) {
					TargetDevice td = (TargetDevice)it.next();
					if ( td != null ) {
						String name = td.getDevicePlatformName();
						String deviceName = td.getDeviceName();
						Device device = getDevice( name,  deviceName);
						if ( device != null ) {
							list.add(device);
						}
					}
				}
			}
		}
		catch (Exception ex) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR),ex);
		}
		Device[] ds = new Device[list.size()];
		for (int i=0; i < list.size(); i++) {
			ds[i] = (Device)list.get(i);
		}
		
		return ds;
	}
	
	public Device getDefaultDevice() throws MtjException {
		try {
			Project project = getProjectData();

			if (project != null) {
				TargetDevice td = project.getDefaultTargetDevice();
				if ( td != null ) {
					String name = td.getDevicePlatformName();
					String deviceName = td.getDeviceName();
					Device device = getDevice( name,  deviceName);
					if ( device != null ) {
						return device;
					}
				}
			}
		}
		catch (Exception ex) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR),ex);
		}
		throw new MtjException(Messages.MtjProject_Device_platform_not_found_2);
	}
	
	
	/**
	 * Returns DevicePlatform's Device that corresponds the TargetDevice
	 * 
	 * @param targetDevice
	 * @return
	 * @throws MtjException
	 */
	public Device getDevice(TargetDevice targetDevice) throws MtjException {
		return getDevice(targetDevice.getDevicePlatformName(), targetDevice.getDeviceName());
	}
	
	private Device getDevice(String devicePlatformName, String deviceName) throws MtjException {
		if ( devicePlatformName != null && deviceName != null ) {
			DeviceManagement dm = (DeviceManagement)
				MtjServices.getInstance().getImplementations(ExtensionType.DEVICE_MANAGEMENT_LITERAL,null,null)[0];
			DevicePlatform platform = dm.getDevicePlatformByName(devicePlatformName);
			if ( platform != null ) {
				Iterator it = platform.getDevices().iterator();
				while ( it.hasNext() ) {
					Device device = (Device)it.next();
					if (deviceName.equals(device.getName())) {
						return device;
					}
				}
			}
		}
		return null;
	}
	
	public Project getProjectData() throws MtjException {
		
		try {
			PersistentStoreProvider pstore = (PersistentStoreProvider)
				MtjServices.getInstance().getImplementations(ExtensionType.PERSISTENT_STORE_PROVIDER_LITERAL,null,null)[0];
					
			IWorkspace workspace = ResourcesPlugin.getWorkspace();
			IWorkspaceRoot root = workspace.getRoot();
			IProject project2 = root.getProject(javaProject.getProject()
					.getName());
			IFile file = project2.getFile(PROJECT_DATA_FILE_NAME);
			if (!project2.exists()) {
				throw new MtjException(Messages.MtjProject_Project_does_not_exist
						+ javaProject.getProject().getName());
			}
			if (!project2.isOpen())
				project2.open(null);
			String fileName;
			if (!file.exists()) {
				return null;
			} else {
				fileName =file.getLocation().toFile().toURI().toString();
			}

			EList list = (EList)
			  pstore.get(PersistentStoreProvider.DATA_TYPE_EMF, 
					"project", fileName); //$NON-NLS-1$
				
			return (Project)list.get(0);
		}
		catch (Exception ex) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR),ex);
		}
	}
	
	public static Project getTemporaryProjectData(IJavaProject javaProject) throws MtjException {
	
		String projectName = javaProject.getProject().getName();
		return getTemporaryProjectData (projectName);

	}
	
	public static boolean isTemporaryProjectDataExisting (String projectName)
	{
		IPath path = getMetadataTemporaryStorage(projectName);
		if (path.toFile().exists())
			return true;
		else
			return false;
	}
	public static Project getTemporaryProjectData(String projectName) throws MtjException {
		
		try {
			PersistentStoreProvider pstore = (PersistentStoreProvider)
				MtjServices.getInstance().getImplementations(ExtensionType.PERSISTENT_STORE_PROVIDER_LITERAL,null,null)[0];
				
			IPath path = getMetadataTemporaryStorage(projectName);
			IPath filePath = path.append(PROJECT_DATA_FILE_NAME);


			String fileName;
			if (!filePath.toFile().exists()) {
				return null;
			} else {
				fileName = filePath.toFile().toURI().toString();
			}

			EList list = (EList)
			  pstore.get(PersistentStoreProvider.DATA_TYPE_EMF, 
					"project", fileName); //$NON-NLS-1$
				
			return (Project)list.get(0);
		}
		catch (Exception ex) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR),ex);
		}
	}
	/**
	 * This method returns the location of the metadata for this plugin plus the project name for a unique location
	 * @return
	 */
	public static IPath getMetadataTemporaryStorage (String projectName)
	{
		IPath metaDataArea = MtjDevIdePlugin.getDefault().getStateLocation();
		IPath metadataPath = metaDataArea.append(projectName);
		return metadataPath;
	}

	public static void setProjectData(IJavaProject javaProject, Project project) throws MtjException {
		try {
			if ( MtjNature.hasMtjNature(javaProject.getProject()) ) {
				IMtjProject suite = null;
				
				synchronized (projectSuiteMap) {
					suite = (IMtjProject) projectSuiteMap.get(javaProject);
					if (suite == null) {
						suite = new MtjProject(javaProject);
						projectSuiteMap.put(javaProject, suite);
					}
					
					suite.setProjectData(project);
				}
			}
		} catch (CoreException e) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_INSTANTIATION_ERROR), e);
		} catch (MtjException e) {
			throw e;
		}
	}
	
	public void setProjectData(Project project) throws MtjException {

		try {
			IWorkspace workspace = ResourcesPlugin.getWorkspace();
			IWorkspaceRoot root = workspace.getRoot();
			IProject project2 = root.getProject(javaProject.getProject()
					.getName());
			IFile file = project2.getFile(PROJECT_DATA_FILE_NAME);
			if (!project2.exists()) {
				throw new MtjException(Messages.MtjProject_Project_does_not_exist + 
						javaProject.getProject().getName());
			}
			if (!project2.isOpen())
				project2.open(null);
			
			String fileName = file.getLocation().toFile().toURI().toString();
			
			PersistentStoreProvider pstore = (PersistentStoreProvider) MtjServices
					.getInstance().getImplementations(
							ExtensionType.PERSISTENT_STORE_PROVIDER_LITERAL,
							null, null)[0];

			pstore.store(project,
					PersistentStoreProvider.DATA_TYPE_EMF, "project", //$NON-NLS-1$
					fileName);

			file.refreshLocal(IResource.DEPTH_ZERO, new NullProgressMonitor());
			
			updateProjectClasspath(new NullProgressMonitor());
			
		} catch (Exception ex) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR), ex);
		}
	}
	
	/**
	 * This is used during project creation to store project data when the project has not been created yet.
	 * When the project is created on the file system, this data will be copied, and the temprorary data
	 * will be deleted.
	 * 
	 * @param project
	 * @param path
	 * @throws MtjException
	 */
	public static void setProjectDataTemporaryLocation(Project project, IPath path) throws MtjException {
		try {

			IPath filePath = path.append(PROJECT_DATA_FILE_NAME);
			path.toFile().mkdirs();
			File outputFile = filePath.toFile();
			if (outputFile.exists())
			{
				boolean success = outputFile.delete();
				if (!success)
					throw new MtjException (Messages.MtjProject_Error_deleting_temporary_file + outputFile.toURI());
			}
			outputFile.createNewFile();

			String fileName = outputFile.toURI().toString();
			
			PersistentStoreProvider pstore = (PersistentStoreProvider) MtjServices
					.getInstance().getImplementations(
							ExtensionType.PERSISTENT_STORE_PROVIDER_LITERAL,
							null, null)[0];

			pstore.store(project,
					PersistentStoreProvider.DATA_TYPE_EMF, "project", //$NON-NLS-1$
					fileName);
			
		} catch (Exception ex) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR), ex);
		}
	}
	
	/**
	 * Since we might create the project file as a temporary file, here is a place to delete it knowing the directory
	 * @param projectName
	 */
	public static void deleteProjectData (String projectName)
	{
		IPath metadataPath = MtjProject.getMetadataTemporaryStorage(projectName);
		IPath filePath = metadataPath.append (PROJECT_DATA_FILE_NAME);
		filePath.toFile().delete();
		metadataPath.toFile().delete();
	}
	
	/**
	 * Returns Project's Deployment
	 * 
	 * @return
	 */
	public Deployment getDeployment() throws CoreException {
		if (projectExtension == null) {
			return null;
		}
		
		try {
			return projectExtension.getDeployment(this);
		} catch (MtjException e) {
			// timeout, consult status handler if there is one
			IStatus status = new Status(
				IStatus.ERROR, 
				IEclipseMtjCoreConstants.PLUGIN_ID, 
				MtjCoreErrors.PROJECT_DEPLOYMENT_ACCESS_ERROR, 
				MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_DEPLOYMENT_ACCESS_ERROR), e);
			throw new CoreException(status);
		}
	}
	
	/**
	 * Returns Project's Deployment to the target application
	 * 
	 * @param launchableApplication
	 * @return
	 * @throws CoreException
	 */
	public Deployment getDeployment(String launchableApplication) throws CoreException {
		if (projectExtension == null) {
			return null;
		}
		
		try {
			return projectExtension.getDeployment(this, launchableApplication);
		} catch (MtjException e) {
			// timeout, consult status handler if there is one
			IStatus status = new Status(
				IStatus.ERROR, 
				IEclipseMtjCoreConstants.PLUGIN_ID, 
				MtjCoreErrors.PROJECT_DEPLOYMENT_ACCESS_ERROR, 
				MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_DEPLOYMENT_ACCESS_ERROR), e);
			throw new CoreException(status);
		}
	}

	public boolean isDeployedJarUpToDate() throws CoreException {
		String upToDateString = getProject().getPersistentProperty(DEPLOYMENT_UP_TO_DATE_PROP);
		return (upToDateString == null) ? false : upToDateString.equals("true"); //$NON-NLS-1$
	}

	/**
	 * Return DeploymentType that the Project is supporting
	 * 
	 * @return
	 * @throws MtjException
	 */
	public DeploymentType getDeploymentType() throws CoreException {
		if (projectExtension == null) {
			return null;
		}
		
		try {
			return projectExtension.getDeploymentType();
		} catch (MtjException e) {
			// timeout, consult status handler if there is one
			IStatus status = new Status(
				IStatus.ERROR, 
				IEclipseMtjCoreConstants.PLUGIN_ID, 
				MtjCoreErrors.PROJECT_INSTANTIATION_ERROR, 
				MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_INSTANTIATION_ERROR), e);
			throw new CoreException(status);
		}
	
	}

	/**
	 * Return IJavaSearchScope object for searching classes that support the project's scope
	 * 
	 * @param javaProject
	 * @return
	 * @throws CoreException
	 */
	public IJavaSearchScope getJavaSearchScope(IJavaProject javaProject) throws CoreException {
		if (projectExtension == null) {
			return null;
		}
		
		try {
			return projectExtension.getJavaSearchScope(javaProject);
		} catch (MtjException e) {
			IStatus status = new Status(
				IStatus.ERROR, 
				IEclipseMtjCoreConstants.PLUGIN_ID, 
				MtjCoreErrors.PROJECT_EXTENSION_DATA_ERROR, 
				MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_EXTENSION_DATA_ERROR), e);
			throw new CoreException(status);
		}
	}

	/**
	 * Returns PreverificationPolicy that the project supports
	 * 
	 * @param targetPlatform
	 * @return
	 */
	public IPreverificationPolicy getPreverificationPolicy(RuntimePlatformDefinition targetPlatform) throws CoreException {
		if (projectExtension == null) {
			return null;
		}
		
		try {
			return projectExtension.getPreverificationPolicy(targetPlatform);
		} catch (MtjException e) {
			// timeout, consult status handler if there is one
			IStatus status = new Status(
				IStatus.ERROR, 
				IEclipseMtjCoreConstants.PLUGIN_ID, 
				MtjCoreErrors.PROJECT_INSTANTIATION_ERROR, 
				MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_EXTENSION_DATA_ERROR), e);
			throw new CoreException(status);
		}
	}

	/**
	 * Method updates new MTJ Projects classpath
	 * 
	 * @param monitor
	 * @throws CoreException
	 */
	public void updateProjectClasspath(IProgressMonitor monitor) throws CoreException {

		JavaCore.setClasspathContainer(
			new Path(J2MEClasspathContainer.J2ME_CONTAINER),
			new IJavaProject[] { getJavaProject() },
			new IClasspathContainer[] { null },
			monitor);
		
		// Force a rebuild
//		((org.eclipse.core.internal.resources.Project)getProject()).getNature("org.eclipse.mtj.extension.devide.nature");
		getProject().build(IncrementalProjectBuilder.FULL_BUILD, monitor);
	}
	
	/**
	 * Method initialized new Java Project
	 * 
	 * @param monitor
	 * @throws CoreException
	 */
	public void initializeNewProject(IProgressMonitor monitor) throws CoreException {
		if (projectExtension == null) {
			return;
		}
		
		try {
			projectExtension.initializeNewProject(getJavaProject(), monitor);
		} catch (MtjException e) {
			// timeout, consult status handler if there is one
			IStatus status = new Status(
				IStatus.ERROR, 
				IEclipseMtjCoreConstants.PLUGIN_ID, 
				MtjCoreErrors.PROJECT_INSTANTIATION_ERROR, 
				MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_EXTENSION_DATA_ERROR), e);
			throw new CoreException(status);
		}
		
		updateProjectClasspath(monitor);
	}
	
	/**
	 * Add the J2ME classpath container to the java project
	 * we wrap if it is currently missing.  
	 * 
	 * @param monitor
	 * @throws JavaModelException
	 */
	public void addClasspathContainerIfMissing(IProgressMonitor monitor) 
		throws CoreException 
	{
		if (!containsJ2MEClasspathContainer(getJavaProject())) {
			// Create a new classpath entry for the classpath
			// container
			IPath entryPath =
				new Path(J2MEClasspathContainer.J2ME_CONTAINER);
			IClasspathEntry newEntry =
				JavaCore.newContainerEntry(entryPath);
			
			// Get the current classpath entries
			IClasspathEntry[] rawClasspath = javaProject.getRawClasspath();
			Set currentClasspath = new LinkedHashSet(rawClasspath.length);
			for (int i = 0; i < rawClasspath.length; i++) {
				IClasspathEntry entry = rawClasspath[i];
				if (entry.getEntryKind() == IClasspathEntry.CPE_VARIABLE) {
					entry = JavaCore.getResolvedClasspathEntry(entry);
				}
				
				currentClasspath.add(entry);
			}
			
			// The classpath entries that are provided by
			// the platform definition currently
			IClasspathEntry[] platformEntries = 
				J2MEClasspathContainer.getClasspathEntries(javaProject);
	
			// Remove the classpath entries from the project
			// if they are provided by the platform definition
			for (int i = 0; i < platformEntries.length; i++) {
				IClasspathEntry entry = platformEntries[i];
				if (currentClasspath.contains(entry)) {
					currentClasspath.remove(entry);
				}
			}
			
			// Set the updated classpath
			currentClasspath.add(newEntry);
			IClasspathEntry[] newClasspath =
				(IClasspathEntry[]) currentClasspath.toArray(
						new IClasspathEntry[currentClasspath.size()]);
			javaProject.setRawClasspath(newClasspath, monitor);
		}
	}
	
	/**
	 * Return a boolean indicating whether the project contains the
	 * J2ME classpath container.
	 * 
	 * @param javaProject the project to be tested
	 * @return whether the project has the J2ME classpath container
	 * @throws JavaModelException
	 */
	public static boolean containsJ2MEClasspathContainer(IJavaProject javaProject) 
		throws JavaModelException 
	{
		boolean contains = false;
		
		IClasspathEntry[] classpath = javaProject.getRawClasspath();
		for (int i = 0; i < classpath.length; i++) {
			IClasspathEntry entry = classpath[i];
			if ((entry.getEntryKind() == IClasspathEntry.CPE_CONTAINER) &&
				(entry.getPath().segment(0).equals(J2MEClasspathContainer.J2ME_CONTAINER)))
			{
				contains = true;
				break;
			}
		}
		
		return contains;
	}
	
/*
 * Example of getting java source file from active project
 * 	
	private String getActiveProjectMidlet(){
		
		org.eclipse.core.resources.IResource[] iResourceChildren = null;
		
		IProject iPrjSource = null;
		IPath srcLocation = null;
		
		
		try {
			iPrjSource = getSelectedProject();
			srcLocation = iPrjSource.getLocation();
			iResourceChildren = iPrjSource.members();
			for (int i = 0; i < iResourceChildren.length; i++) {
				java.io.File sourceFile = iResourceChildren[i].getLocation().toFile();
				if (sourceFile.getAbsolutePath().endsWith(".java")){
					if (sourceFile.canRead()){
						//Read file content and analyse that does the file extend the MIDlet
						if (true) {
							//return the file name
							return sourceFile.getAbsolutePath();
						}
					}
				}
			}
		} catch (CoreException ce) {
			//Show error
			System.out.println("Error: "+ce.getMessage());
		}
		
		return "";
	}
	public static IProject getSelectedProject()
    {
         IProject iProject = null;
         org.eclipse.core.resources.IResource iResource = null;
         iResource = DebugUITools.getSelectedResource();
         if (iResource != null)
           iProject = iResource.getProject();
        return iProject;
   } 
*/   
}

