/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.extension.dpp.nokia;

import java.util.StringTokenizer;

import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DeviceDescription;
import org.eclipse.mtj.api.devices.DevicesFactory;
import org.eclipse.mtj.executable.uei.AbstractUeiDevicePlatformImporter;

public class NokiaSdkDevicePlatformImporter extends AbstractUeiDevicePlatformImporter {
	
	private final String UEI_PROP_DEVICE_DESCRIPTION = ".description";
	private final String UEI_PROP_DEVICE_MODEL = ".model";
	
	private final String vendor = "Nokia";
	
	private static NokiaSdkDevicePlatformImporter instance;
	
	private NokiaSdkDevicePlatformImporter(){
		super(false);
	}
	
	public static NokiaSdkDevicePlatformImporter getDefault() {
		
		if(instance == null){
			instance = new NokiaSdkDevicePlatformImporter();
		}
		
		return instance;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.mtj.executable.uei.AbstractUeiDevicePlatformImporter#getVendor()
	 */
	@Override
	public String getVendor() {
		
		return vendor;
	}
	

	/* 
	 * Set Device name, vendor, version and description.
	 * The description is taken from the list of following properties until one found:
	 * description, model, "Generic UEI device"
	 * 
	 * (non-Javadoc)
	 * @see org.eclipse.mtj.executable.uei.AbstractUeiDevicePlatformImporter#createDeviceDetails(org.eclipse.mtj.api.devices.Device)
	 */
	@Override
	public void createDeviceDetails(Device device) {
		
		String origName = device.getName();
		
		//device.setName(getStyledDeviceName(origName));
		device.setVendor(getVendor());
		device.setVersion(getDeviceVersion(origName));
		String deviceDescrKey = origName + UEI_PROP_DEVICE_DESCRIPTION;
		String description = getUeiProperties().getProperty(deviceDescrKey);
		if(description == null){
			deviceDescrKey = origName + UEI_PROP_DEVICE_MODEL;
			description = getUeiProperties().getProperty(deviceDescrKey);
			if(description == null){
				description = "Generic UEI device";
			}
		}
		device.setDescription(description);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.mtj.executable.uei.AbstractUeiDevicePlatformImporter#getRuntimePlatformDefinitionName(java.lang.String)
	 */
	public String getRuntimePlatformDefinitionName(String deviceName) {
		
		StringBuffer sb = new StringBuffer(getVendor());
		sb.append(" ");
		sb.append(getStyledDeviceName(deviceName));
		sb.append(" Runtime Platform");
		
		return sb.toString();
	}
	


	/* (non-Javadoc)
	 * @see org.eclipse.mtj.executable.uei.AbstractUeiDevicePlatformImporter#createDeviceDescription(java.lang.String)
	 */
	@Override
	public DeviceDescription createDeviceDescription(String deviceName) {
		
		DeviceDescription dd = DevicesFactory.eINSTANCE.createDeviceDescription();
		
		dd.setIdentifier(getVendor()+" " + getUeiProperties().getProperty(UEI_PROP_DEVICE_MODEL));
		
		//TODO Create device capabilities
		//DevicesFactory.eINSTANCE.createDeviceCapability();
		
		//TODO Create device features
		//DevicesFactory.eINSTANCE.createDeviceFeature()
		
		return dd;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.mtj.executable.uei.AbstractUeiDevicePlatformImporter#hasDebugServer()
	 */
	@Override
	public boolean hasDebugServer() {
		return true;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.mtj.executable.uei.AbstractUeiDevicePlatformImporter#isPredeployRequired()
	 */
	@Override
	public boolean isPredeployRequired() {
		return true;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.mtj.executable.uei.AbstractUeiDevicePlatformImporter#getLaunchCommand()
	 */
	@Override
	public String getLaunchCommand() {
		return "%executable% " +
			   "[%debugPort%|-Xdebug -Xrunjdwp:transport=dt_socket,suspend=y,server=y,address=%debugPort%] " +
		   	   "-classpath \"%classpath%\" -Xdevice:%device% [%jadfile%|\"-Xdescriptor:%jadfile%\"] [%verbose%|-Xverbose:%verbose%] " +
		       "[%heapsize%|-Xheapsize:%heapsize%] [%securityDomain%|-Xdomain:%securityDomain%] %userSpecifiedArguments% " +
		       " [%otaurl%|-Xjam:transient=%otaurl%]";
	}
	
	
	/** Returns device name with spaces as delimeter characters.*/
	private String getStyledDeviceName(String deviceName){
		
		StringBuffer styledDeviceName = new StringBuffer();
		
		StringTokenizer st = new StringTokenizer(deviceName, "_");
		
		while (st.hasMoreTokens()) {
			styledDeviceName.append(st.nextToken().trim());
			styledDeviceName.append(" ");
		}
		
		return styledDeviceName.toString().trim();
	}
	
	/** Get device version from Nokia SDK */
	private String getDeviceVersion(String deviceName){
		return "1.0.3";
	}

}
