/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation and others
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.core.ui;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.preference.IPersistentPreferenceStore;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.mtj.core.ui.constants.CoreConstants;
import org.eclipse.mtj.core.ui.preferences.PluginPreferenceStore;
import org.eclipse.mtj.core.util.DirUtil;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;


/**
 * The main plugin class to be used in the desktop.
 */
public class MtjCoreUiPlugin extends AbstractUIPlugin {

	//The shared instance.
	private static MtjCoreUiPlugin plugin;
	
	// A preference store wrapper around the core plugin plugin preferences
	private IPreferenceStore corePreferenceStore;

	
	/**
	 * The constructor.
	 */
	public MtjCoreUiPlugin() {
		plugin = this;
	}

	/**
	 * This method is called upon plug-in activation
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
		
		startTests();

	}

	/**
	 * This method is called when the plug-in is stopped
	 */
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
		plugin = null;
	}

	/**
	 * Returns the shared instance.
	 */
	public static MtjCoreUiPlugin getDefault() {
		return plugin;
	}

	/**
	 * Returns an image descriptor for the image file at the given
	 * plug-in relative path.
	 *
	 * @param path the path
	 * @return the image descriptor
	 */
	public static ImageDescriptor getImageDescriptor(String path) {
		return AbstractUIPlugin.imageDescriptorFromPlugin("org.eclipse.mtj.core.ui", path); //$NON-NLS-1$
	}
	/**
	 * Get an ImageDescriptor for the for specified plugin
	 * image.
	 */
	public static ImageDescriptor getIconImage(String imageName) {
		ImageDescriptor img = null;
		
		try {
			URL pluginURL = getDefault().getBundle().getEntry("/"); //$NON-NLS-1$
			URL imgURL = new URL(pluginURL, "icons/" + imageName); //$NON-NLS-1$
			
			img = ImageDescriptor.createFromURL(imgURL);
		} catch (MalformedURLException e) {
			MtjCorePlugin.log(IStatus.INFO, "getIconImage", e); //$NON-NLS-1$
		}
		
		return img;
	}
	/**
	 * Display an error to the user given the specified information.
	 * 
	 * @param shell
	 * @param severity
	 * @param code
	 * @param title
	 * @param message
	 * @param exception
	 */
	public static void displayError(
			Shell shell,
			int severity, 
			int code,
			String title,
			String message, 
			Throwable exception) 
	{
		String id = getDefault().getBundle().getSymbolicName();
		Status status = new Status(severity, id, code, message, exception);
		getDefault().getLog().log(status);
		ErrorDialog.openError(shell, title, message, status);
	}
	/**
	 * Return the active window's shell.
	 * 
	 * @return the active window shell
	 */
	public static Shell getActiveWindowShell() {
		MtjCoreUiPlugin plugin = getDefault();
		IWorkbenchWindow activeWindow = plugin.getWorkbench().getActiveWorkbenchWindow();
		return activeWindow.getShell();
	}

	/**
	 * Return a preference store that sits on top of the EclipseME core preferences
	 * for the specified project.
	 * 
	 * @param context
	 * @return
	 */
	public static IPersistentPreferenceStore getCoreProjectPreferenceStore(IProject context) {
		ProjectScope projectScope = new ProjectScope(context);
		return new ScopedPreferenceStore(projectScope, CoreConstants.PLUGIN_ID);
	}
	
	
	/**
	 * Return an IPreferenceStore wrapper around the EclipseME core
	 * plugin preferences.
	 * 
	 * @return
	 */
	public IPreferenceStore getCorePreferenceStore() {
		if (corePreferenceStore == null) {
			MtjCoreUiPlugin plugin = MtjCoreUiPlugin.getDefault();
			corePreferenceStore = 
				new PluginPreferenceStore(plugin, plugin.getPluginPreferences());
		}
		
		return corePreferenceStore;
	}
	
	public static IWorkbenchWindow getActiveWorkbenchWindow() {
		return PlatformUI.getWorkbench().getActiveWorkbenchWindow();
	}	
	
	public static Shell getActiveWorkbenchShell() {
		return getActiveWorkbenchWindow().getShell();
	}

	private void startTests() {

	}

	/**
	 * This method returns the plugin's icons directory location in the file system.
	 * @return plugin's icons directory path in the file system
	 */
	public static String getPluginIconsDir()
	{
		return getInstallLocation().append("/icons").toString(); //$NON-NLS-1$
	}
	
	/**
	 * This method populates the plugin's imageRegistry with icons found in the directory.
	 * @return plugin's icons directory path
	 */
	protected void initializeImageRegistry (ImageRegistry imageRegistry)
	{
		super.initializeImageRegistry(imageRegistry);
		try
		{
			ArrayList images = DirUtil.getAllFilesInDirectory (getPluginIconsDir(), true, true, "*.gif"); //$NON-NLS-1$
			String fileName;
			String imageKey;
			int pos;
			for (Iterator iter = images.iterator(); iter.hasNext();)
			{
				fileName = (String) iter.next();
				pos = fileName.indexOf("/icons/"); //$NON-NLS-1$
				if (pos != -1)
				{
					pos += 7;
					imageKey = fileName.substring(pos);
					imageRegistry.put(imageKey, ImageDescriptor.createFromURL(new URL("file:" + fileName))); //$NON-NLS-1$
				}
			}
		}
		catch (Exception e)
		{
			e.getMessage();
		}
	}

	public static ImageRegistry getMtjImageRegistry ()
	{
		return getDefault().getImageRegistry();
	}
	
	
	public static String getPluginId ()
	{
		return getDefault().getBundle().getSymbolicName();
	}

	/**
	 * This method returns the plugin's installation directory.
	 * @return plugin's installation directory path
	 */
	public static Path getInstallLocation()
	{
		try
		{
			Bundle bundle = getDefault().getBundle();
			URL url = FileLocator.find(bundle, new Path(""), null);  //$NON-NLS-1$
			String s1 = FileLocator.resolve(url).getFile();
			if(s1.startsWith("/")) //$NON-NLS-1$
				s1 = s1.substring(1);
			s1 = (new Path(s1)).toOSString();
			String s;
			if(s1.endsWith(File.separator))
				s = s1;
			else
				s = s1 + File.separator;
			return new Path(s);
		}
		catch(Exception exception)
		{
			return null;
		}
	}

}
