/*******************************************************************************
 * Copyright (c) 2005 Nokia Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 *******************************************************************************/
package org.eclipse.mtj.extension.dpp.uei;

import java.util.Iterator;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.jdt.launching.VMRunnerConfiguration;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.mtj.api.deployment.Deployment;
import org.eclipse.mtj.api.devices.ConfigurationItem;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.devices.DevicesFactory;
import org.eclipse.mtj.api.enumerations.DeviceStatus;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.extension.DevicePlatformProvider;
import org.eclipse.mtj.api.extension.PersistentStoreProvider;
import org.eclipse.mtj.api.extension.impl.MtjExtensionImpl;
import org.eclipse.mtj.api.model.IExecutablePlatform;
import org.eclipse.mtj.api.runtimes.DeviceConfiguration;
import org.eclipse.mtj.api.runtimes.DeviceProfile;
import org.eclipse.mtj.api.runtimes.ImplementationRef;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.api.runtimes.ServiceApi;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.core.MtjServices;
import org.eclipse.mtj.core.launching.EmulatorRunner;
import org.eclipse.mtj.core.toolkit.DevicePlatformImporter;
import org.eclipse.mtj.core.toolkit.ToolkitSupport;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.exception.MtjNotFoundException;
import org.eclipse.mtj.executable.uei.UeiExecutable;
import org.eclipse.mtj.extension.dm.preferences.DmPreferencePage;

/** Generic implementation of a UEI specific Device Platform Provider. */
public abstract class AbstractUeiDevicePlatformProvider extends MtjExtensionImpl
		implements DevicePlatformProvider, ToolkitSupport {
	public abstract DevicePlatformImporter getDevicePlatformImporter();

	/**
	 * Method returns Plugin name that owns the DevicePlatform data.
	 */
	public abstract String getPluginName();

	public DevicePlatform[] getDevicePlatforms() throws MtjException {
		try {
			PersistentStoreProvider pstore = (PersistentStoreProvider) MtjServices
					.getInstance().getImplementations(
							ExtensionType.PERSISTENT_STORE_PROVIDER_LITERAL,
							null, null)[0];

			EList list = null;
			try {
				list = (EList) pstore.get(
					PersistentStoreProvider.DATA_TYPE_EMF, "devices", //$NON-NLS-1$
					getPluginName(), "DevicePlatform"); //$NON-NLS-1$
			}
			catch (MtjNotFoundException ex) {
				setDevicePlatforms(new DevicePlatform[0]);
				return new DevicePlatform[0];
			}
			DevicePlatform[] devicePlatforms = new DevicePlatform[list.size()];
			for (int i = 0; i < list.size(); i++) {
				devicePlatforms[i] = (DevicePlatform) list.get(i);
				setRelativeFilePaths(devicePlatforms[i]);
				devicePlatforms[i].setDevicePlatformProvider(this);
			}

			return devicePlatforms;
		} catch (MtjNotFoundException ex) {
			return new DevicePlatform[0];
		} catch (Exception ex) {
			throw new MtjException(MtjCoreErrors
					.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR), ex);
		}
	}

	public void verify(IFolder sourceFolder, IProgressMonitor monitor)
			throws MtjException {
		// TODO
	}

	public String getRootDirectory(DevicePlatform dp) throws MtjException {
		Iterator it = dp.getConfiguration().getItems().iterator();
		while (it.hasNext()) {
			ConfigurationItem item = (ConfigurationItem) it.next();
			if ("directory".equals(item.getName())) { //$NON-NLS-1$
				String d = item.getValue();
				if (d != null) {
					d = d.replace('\\', '/');
					if (d.endsWith("/")) { //$NON-NLS-1$
						d = d.substring(0, d.length() - 1);
					}
					return d;
				}
			}
		}

		throw new MtjException(Messages.AbstractUeiDevicePlatformProvider_RootDirectoryNotDefined);
	}

	public void setRootDirectory(DevicePlatform dp, String directory) {
		Iterator it = dp.getConfiguration().getItems().iterator();
		while (it.hasNext()) {
			ConfigurationItem item = (ConfigurationItem) it.next();
			if ("directory".equals(item.getName())) { //$NON-NLS-1$
				item.setValue(directory);
				return;
			}
		}

		// root directory not yet defined
		ConfigurationItem item = DevicesFactory.eINSTANCE.createConfigurationItem();
		item.setName("directory"); //$NON-NLS-1$
		item.setValue(directory);
		dp.getConfiguration().getItems().add(item);
	}

	/**
	 * Method returns Device's status
	 * @param device
	 * @return
	 * @throws MtjException
	 */
	public DeviceStatus getStatus(Device device) throws MtjException {
		Iterator it = getDevicePlatforms()[0].getDevices().iterator();
		while (it.hasNext()) {
			Device d = (Device)it.next();
			if (d.getName() != null && d.getName().equals(device.getName())) {
				return DeviceStatus.DEVICE_STATUS_OK_LITERAL;
			}
		}
		
		return DeviceStatus.DEVICE_STATUS_ERROR_LITERAL;
	}	

	private void setRelativeFilePaths(DevicePlatform dp) throws MtjException {
		String root = getRootDirectory(dp);
		Iterator it = dp.getRuntimePlatformDefinitions().iterator();
		while (it.hasNext()) {
			RuntimePlatformDefinition rpd = (RuntimePlatformDefinition) it
					.next();
			Iterator it2 = rpd.getDeviceProfiles().iterator();
			while (it2.hasNext()) {
				DeviceProfile dep = (DeviceProfile) it2.next();
				Iterator it3 = dep.getServiceApis().iterator();
				while (it3.hasNext()) {
					ServiceApi sa = (ServiceApi) it3.next();
					ImplementationRef ir = sa.getImplementation();
					setRelativeFilePaths(ir, root);
				}
			}
			it2 = rpd.getServiceApis().iterator();
			while (it2.hasNext()) {
				ServiceApi sa = (ServiceApi) it2.next();
				ImplementationRef ir = sa.getImplementation();
				setRelativeFilePaths(ir, root);
			}
			it2 = rpd.getDeviceConfigurations().iterator();
			while (it2.hasNext()) {
				DeviceConfiguration dc = (DeviceConfiguration) it2.next();
				ImplementationRef ir = dc.getImplementation();
				setRelativeFilePaths(ir, root);
			}
		}
	}

	private void setRelativeFilePaths(ImplementationRef ir, String root)
			throws MtjException {
		ir.setFileRef(setRelativeFilePaths(ir.getFileRef(), root));
		ir.setJavadocRef(setRelativeFilePaths(ir.getJavadocRef(), root));
		ir.setSourceRef(setRelativeFilePaths(ir.getSourceRef(), root));
	}

	private String setRelativeFilePaths(String path, String root)
			throws MtjException {
		if (path == null) {
			return path;
		}
		String r = path.replace('\\', '/');
		if (r.startsWith("/")) { //$NON-NLS-1$
			r = root + r;
		}
		return r;
	}

	private void setAbsoluteFilePaths(DevicePlatform dp) throws MtjException {
		String root = getRootDirectory(dp);
		Iterator it = dp.getRuntimePlatformDefinitions().iterator();
		while (it.hasNext()) {
			RuntimePlatformDefinition rpd = (RuntimePlatformDefinition) it
					.next();
			Iterator it2 = rpd.getDeviceProfiles().iterator();
			while (it2.hasNext()) {
				DeviceProfile dep = (DeviceProfile) it2.next();
				Iterator it3 = dep.getServiceApis().iterator();
				while (it3.hasNext()) {
					ServiceApi sa = (ServiceApi) it3.next();
					ImplementationRef ir = sa.getImplementation();
					setAbsoluteFilePaths(ir, root);
				}
			}
			it2 = rpd.getServiceApis().iterator();
			while (it2.hasNext()) {
				ServiceApi sa = (ServiceApi) it2.next();
				ImplementationRef ir = sa.getImplementation();
				setAbsoluteFilePaths(ir, root);
			}
			it2 = rpd.getDeviceConfigurations().iterator();
			while (it2.hasNext()) {
				DeviceConfiguration dc = (DeviceConfiguration) it2.next();
				ImplementationRef ir = dc.getImplementation();
				setAbsoluteFilePaths(ir, root);
			}
		}
	}

	private void setAbsoluteFilePaths(ImplementationRef ir, String root)
			throws MtjException {
		ir.setFileRef(setAbsoluteFilePaths(ir.getFileRef(), root));
		ir.setJavadocRef(setAbsoluteFilePaths(ir.getJavadocRef(), root));
		ir.setSourceRef(setAbsoluteFilePaths(ir.getSourceRef(), root));
	}

	private String setAbsoluteFilePaths(String path, String root)
			throws MtjException {
		if (path == null) {
			return path;
		}
		String r = path.replace('\\', '/');
		if (r.startsWith(root)) {
			r = r.substring(root.length());
		}
		return r;
	}

	public void launch(DevicePlatform dp, Device device, Deployment deployment,
			VMRunnerConfiguration vmRunnerConfig,
			ILaunchConfiguration launchConfig, ILaunch launch, String runMode,
			String projectNatureId, IProgressMonitor monitor)
			throws MtjException {
 
		IExecutablePlatform executablePlatform = new UeiExecutable(dp, device);
		EmulatorRunner emulatorRunner = new EmulatorRunner(dp, executablePlatform, runMode);
		try {
			emulatorRunner.run(vmRunnerConfig, launchConfig, deployment,
					launch, projectNatureId, device, monitor);
		} catch (CoreException e) {
			throw new MtjException(e);
		} catch (MtjException e) {
			throw e;
		}
	}

	public void setDevicePlatforms(DevicePlatform[] dps) throws MtjException {
		EList list = new BasicEList();
		try {
			for (int i = 0; i < dps.length; i++) {
				
				setAbsoluteFilePaths(dps[i]);
				dps[i].setDevicePlatformProvider(null);
				list.add(dps[i]);
			}
			PersistentStoreProvider pstore = (PersistentStoreProvider) MtjServices
				.getInstance()
				.getImplementations(
					ExtensionType.PERSISTENT_STORE_PROVIDER_LITERAL,
					null, null)[0];

			pstore.store(list, PersistentStoreProvider.DATA_TYPE_EMF,
					"devices", getPluginName(), "DevicePlatform"); //$NON-NLS-1$ //$NON-NLS-2$
		
			for (int i = 0; i < dps.length; i++) {
				setRelativeFilePaths(dps[i]);
				dps[i].setDevicePlatformProvider(this);
			}
			
		} catch (Exception ex) {
			throw new MtjException(MtjCoreErrors
					.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR), ex);
		}
	}

	public PreferencePage getPreferencePage() {
		return new DmPreferencePage();
	}
}
