/*******************************************************************************
* Copyright  2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/

package org.eclipse.mtj.extension.dm.preferences;

import java.util.logging.Logger;

import org.eclipse.emf.common.util.EList;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.extension.DeviceManagement;
import org.eclipse.mtj.core.MtjServices;
import org.eclipse.mtj.core.ui.util.CompositeDialog;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.dm.Messages;
import org.eclipse.mtj.extension.dm.model.MtjServiceHandler;
import org.eclipse.mtj.extension.dm.ui.preferences.AddDevicePlatformDialog2;
import org.eclipse.mtj.extension.dm.ui.preferences.DeviceInfoComposite;
import org.eclipse.mtj.extension.dm.ui.preferences.DevicePlatformInfoComposite;
import org.eclipse.mtj.extension.dm.ui.preferences.DevicePlatformListComposite;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

public class DevicePlatformListController implements ISelectionChangedListener, SelectionListener {
	private static Logger log = Logger.getLogger(DevicePlatformListController.class.getName());
	private DevicePlatformListComposite composite;
	private MtjServiceHandler mtjServiceHandler = new MtjServiceHandler();
	private DevicePlatform selectedDevicePlatform;
	private Device selectedDevice;
	
	private static Shell containerShell = new Shell();	
	private Button addButton;
	private Button editRuntimeButton;
	private Button editDeviceButton;
	private Button removeButton;
	private TableViewer devicePlatformTableViewer;		
	private TableViewer deviceTableViewer;

	
	public DevicePlatformListController(DevicePlatformListComposite composite) {
		this.composite = composite;
		
		addButton = composite.getAddButton();
		editRuntimeButton = composite.getEditRuntimeButton();
		editDeviceButton = composite.getEditDeviceButton();
		removeButton = composite.getRemoveButton();
		devicePlatformTableViewer = composite.getDevicePlatformTableViewer();
		deviceTableViewer = composite.getDeviceTableViewer();
		
		setData();
		addListener();
		
		editRuntimeButton.setText(Messages.DevicePlatformListController_Info);
	}
	
	private void addListener() {
		this.devicePlatformTableViewer.addSelectionChangedListener(this);
		this.deviceTableViewer.addSelectionChangedListener(this);

		this.addButton.addSelectionListener(this);
		this.editDeviceButton.addSelectionListener(this);
		this.editRuntimeButton.addSelectionListener(this);
		this.removeButton.addSelectionListener(this);
	}
	
	private static TableColumn createTableColumn(String title, Table table, int index, int widthHint) {
		TableColumn tableColumn = new TableColumn(table, SWT.NONE, index);
		tableColumn.setResizable(true);
		tableColumn.setWidth(widthHint);
		tableColumn.setText(title);
		return tableColumn;
	}

	private void setData() {
		class ArrayContentProvider implements IStructuredContentProvider 
		{
			public Object[] getElements(Object o) {
				if(o instanceof DevicePlatform[]) {
					return (Object[])o;
				} else return null;
			}
			
			// public RuntimePlatform[] removeVariable(Object oldInputElement, RuntimePlatform selectedRp) {}				
			public void dispose() {}		
			public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {}
		}

		class dtlp extends LabelProvider implements ITableLabelProvider {
			public Image getColumnImage(Object element, int columnIndex) {
				  return null;
			}
			
			public String getColumnText(Object element, int columnIndex) {
				if(element instanceof DevicePlatform) {
					DevicePlatform dp = (DevicePlatform) element;
					if(columnIndex == 0) { return dp.getName(); }
					else if(columnIndex == 1) { return dp.getVersion(); }
					else if(columnIndex == 2) { return dp.getVendor(); }
				}
				return null;
			}			
		}

		devicePlatformTableViewer.setContentProvider(new ArrayContentProvider());
		devicePlatformTableViewer.setLabelProvider(new dtlp());

		createTableColumn(Messages.DevicePlatformListController_Name, devicePlatformTableViewer.getTable(), 0, 220);
		createTableColumn(Messages.DevicePlatformListController_Version, devicePlatformTableViewer.getTable(), 1, 40);
		createTableColumn(Messages.DevicePlatformListController_Vendor, devicePlatformTableViewer.getTable(), 2, 100);

		class tlp extends LabelProvider implements ITableLabelProvider {
			public Image getColumnImage(Object element, int columnIndex) {
				  return null;
			}
			
			public String getColumnText(Object element, int columnIndex) {
				if(element instanceof Device) {
					Device device = (Device) element;
					if(columnIndex == 0) { return device.getName(); }
					else if(columnIndex == 1) { return device.getDescription(); }
				}
				return null;
			}			
		}
		
		class EListContentProvider implements IStructuredContentProvider 
		{
			public Object[] getElements(Object o) {
				if(o instanceof EList) {
					return ((EList)o).toArray();
				}
				return null;
			}
			
			public void dispose() {}		
			public void inputChanged(Viewer viewer,Object oldInput,Object newInput) {}
		}

		deviceTableViewer.setContentProvider(new EListContentProvider());
		deviceTableViewer.setLabelProvider(new tlp());
		
		createTableColumn(Messages.DevicePlatformListController_Name,deviceTableViewer.getTable(),0,160);
		createTableColumn(Messages.DevicePlatformListController_Description,deviceTableViewer.getTable(),1,160);
		
		// Data
		devicePlatformTableViewer.setInput(mtjServiceHandler.getDevicePlatforms());		
		
		editRuntimeButton.setEnabled(false);
		editDeviceButton.setEnabled(false);
		removeButton.setEnabled(false);
	}
	
	private void devicePlatformTableEvent() {
		ISelection selection = devicePlatformTableViewer.getSelection();
		if (!selection.isEmpty()) {
			IStructuredSelection structured = (IStructuredSelection) selection;
			Object element = structured.getFirstElement();
			if(element instanceof DevicePlatform){
				DevicePlatform dp = (DevicePlatform) element;
				selectedDevicePlatform = dp;
				EList dl = dp.getDevices();
				this.deviceTableViewer.setInput(dl);
				editRuntimeButton.setEnabled(true);
				removeButton.setEnabled(true);				
				editDeviceButton.setEnabled(false);
			}
		}		
	}	

	public DevicePlatform[] getDevicePlatforms() {
		return (DevicePlatform[]) this.devicePlatformTableViewer.getInput();
	}
	
	private void deviceTableEvent() {
		ISelection selection = deviceTableViewer.getSelection();
		if (!selection.isEmpty()) {
			IStructuredSelection structured = (IStructuredSelection) selection;
			Object element = structured.getFirstElement();
			if(element instanceof Device){
				Device d = (Device) element;
				this.selectedDevice = d;
				this.editDeviceButton.setEnabled(true);
			} else
				this.editDeviceButton.setEnabled(false);
		}				
	}	
	
	public void selectionChanged(SelectionChangedEvent event) {
		if ( event.getSource() == devicePlatformTableViewer )
			devicePlatformTableEvent();
		if ( event.getSource() == deviceTableViewer )
			deviceTableEvent();
	}
	
	public void widgetDefaultSelected(SelectionEvent e) {}
	public void widgetSelected(SelectionEvent e) {
		if(e.widget == addButton) {
			addDevicePlatform();
		} else if(e.widget == editRuntimeButton) {
			editDevicePlatform();
		} else if(e.widget == editDeviceButton) {
			editDevice();
		} else if(e.widget == removeButton) {
			removeDevicePlatform();
		}
	}

	private void addDevicePlatform() {
		AddDevicePlatformDialog2 dialog = new AddDevicePlatformDialog2(composite.getShell(), Messages.DevicePlatformListController_AddMobileDevicePlatform);
		dialog.setBlockOnOpen(true);

		if (dialog.open() == Dialog.OK) {			
			DevicePlatform[] devicePlatforms = this.mtjServiceHandler.getDevicePlatforms();
			// boolean okToStore = MtjServiceHandler.setDevicePlatforms(devicePlatforms);
			this.devicePlatformTableViewer.setInput(devicePlatforms);
			if(devicePlatforms != null)
			this.devicePlatformTableViewer.refresh();
		}
	}
		
	private void removeDevicePlatform() {	
		IStructuredSelection ss = (IStructuredSelection) this.devicePlatformTableViewer.getSelection();
		Object o = ss.getFirstElement();
		DevicePlatform dp = (DevicePlatform)o;

		if ( dp != null ) {
			DeviceManagement dm = (DeviceManagement)
				MtjServices.getInstance().getImplementations(ExtensionType.DEVICE_MANAGEMENT_LITERAL, null, null, true)[0];
			try {
				dm.removeDevicePlatform(dp);
			} catch (MtjException e) {
				// TODO Show error to user
				e.printStackTrace();
				return;
			}
		}

		DevicePlatform[] dpa = (DevicePlatform[])this.devicePlatformTableViewer.getInput();
		DevicePlatform[] newdpa = new DevicePlatform[dpa.length-1];
		
		for(int i = 0,j = 0; i < dpa.length && j < newdpa.length; i++) {
			if(dp != dpa[i]) {
				newdpa[j] = dpa[i];
				j++;
			} 				
		}
		// boolean okToStore = MtjServiceHandler.setDevicePlatforms(newdpa);

		this.devicePlatformTableViewer.remove(dp);
		this.devicePlatformTableViewer.setInput(newdpa);
		
	}

	public void editDevice() {
		DeviceInfoComposite dic = new DeviceInfoComposite(containerShell);
		CompositeDialog d = CompositeDialog.createDialog(dic,new DeviceInfoController(this.selectedDevice, this.mtjServiceHandler, dic),Messages.DevicePlatformListController_Device,Messages.DevicePlatformListController_DeviceSettingsInfo);
		d.open();
	}

	private void editDevicePlatform() {
		DevicePlatformInfoComposite dpic = new DevicePlatformInfoComposite(containerShell);
		CompositeDialog d = CompositeDialog.createDialog(dpic,new DevicePlatformInfoController(dpic,this.selectedDevicePlatform),Messages.DevicePlatformListController_DevicePlatform,Messages.DevicePlatformListController_DevicePlatformInfo);
		d.open();
	}
}

