/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
 * Copyright (c) 2003-2005 Craig Setera
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: 
 * Nokia - Initial API and implementation 
 * Craig Setera - partial implementation
*******************************************************************************/
package org.eclipse.mtj.executable.uei;

import java.io.File;
import java.io.FileFilter;
import java.util.List;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.common.util.EList;
import org.eclipse.mtj.api.devices.Configuration;
import org.eclipse.mtj.api.devices.ConfigurationItem;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DeviceDescription;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.devices.DevicesFactory;
import org.eclipse.mtj.api.devices.ProtectionDomain;
import org.eclipse.mtj.api.enumerations.PlatformType;
import org.eclipse.mtj.api.runtimes.DeviceConfiguration;
import org.eclipse.mtj.api.runtimes.DeviceProfile;
import org.eclipse.mtj.api.runtimes.ImplementationRef;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.api.runtimes.RuntimesFactory;
import org.eclipse.mtj.api.runtimes.ServiceApi;
import org.eclipse.mtj.core.toolkit.DevicePlatformImporter;

/** 
 * Abstract UEI specific Device Platform importer. 
 * Enables the import of device platforms that conform to the UEI specification.
 * Vendor specific implemenations should extend this class.*/
public abstract class AbstractUeiDevicePlatformImporter implements DevicePlatformImporter, UEIConstants {
	private Properties ueiProperties;
	private String useApiString;	
		
	public AbstractUeiDevicePlatformImporter() {
		this(true);
	}

	public AbstractUeiDevicePlatformImporter(boolean useApis) {
		this.useApiString = useApis ? UEI_PROP_DEVICE_APIS: UEI_PROP_DEVICE_BOOTCLASSPATH;
	}
	
	public abstract boolean hasDebugServer();
	public abstract boolean isPredeployRequired();
	public abstract String getLaunchCommand();
	
	public Properties getUeiProperties(){
		return ueiProperties;
	}
	
	/** 
	 * Returns the vendor name that will be used in the UI.
	 * Because accessing the vendor name is not specified in the UEI specification, 
	 * vendors should provide own implementation on how to get this value.*/
	public abstract String getVendor();
	
	/**
	 * Return the runttime platform definition name that will be used for the device.
	 * The vendors should provide own implementation on how to get this value.
	 * Default value will be the "vendorName deviceName Runtime Platform"
	 */
	public abstract String getRuntimePlatformDefinitionName(String deviceName);
	
	public DevicePlatform[] getDevicePlatforms(File rootDirectory) {
		DevicePlatform[] dps = null;
		Vector foundDps = new Vector();
		
		//TODO Add progress monitor
		
		//Search root directory
		findPlatforms(rootDirectory, foundDps);
		
		//Search subdirectories
		File[] subdirectories = rootDirectory.listFiles(new FileFilter() {
			public boolean accept(File pathname) {
				return pathname.isDirectory();
			}
		});
		
		for (int i = 0; i < subdirectories.length; i++) {			
			findPlatforms(subdirectories[i], foundDps);
		}
		
		if(foundDps.size() == 0){
			return null;
		}
		
		dps = new DevicePlatform[foundDps.size()];		
		return (DevicePlatform[]) foundDps.toArray(dps);
	}
	
	private Properties parseUEIPropertiesString(String output) {
		return null;
	}
	
	/** Find platforms from directories that comply to UEI spec */
	private void findPlatforms(File directory, Vector dps){		
		//Create emulator file from provided location
		File emulatorExecutable = UeiExecutable.getEmulatorExecutable(directory);
		
		try {
			ueiProperties = UeiPropertiesCache.instance.getUEIProperties(emulatorExecutable);
		} catch (CoreException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			return;
		}
		
		if(ueiProperties.isEmpty()){
			return;
		}
		
		DevicePlatform dp = DevicesFactory.eINSTANCE.createDevicePlatform();
		
		//Create Platform details ie name, vendor, version and type
		//setPlatformDetails(dp);
		createPlatformDetails(dp);
		
		//Create Devices
		createDevices(dp);
		
		//Create configuration
		Configuration conf = createDirConfiguration(directory.getAbsolutePath());
		dp.setConfiguration(conf);
		//TODO What other is needed/wanted?
		
		dps.add(dp);
	}
	
	public Configuration createDirConfiguration(String rootDirectory) {	
		Configuration conf = DevicesFactory.eINSTANCE.createConfiguration();
		
		ConfigurationItem item = DevicesFactory.eINSTANCE.createConfigurationItem();
		item.setName("directory"); //$NON-NLS-1$
		item.setLabel("Directory"); //$NON-NLS-1$
		item.setDescription("Emulator installation directory"); //$NON-NLS-1$
		item.setValue(rootDirectory);
		
		conf.getItems().add(item);
		
		return conf;
	}

	/** Create general parameters for the device platform */
	public void createPlatformDetails(DevicePlatform dp){		
		dp.setName(ueiProperties.getProperty(UeiPropertiesCache.PROP_TOOLKIT_NAME));
		dp.setVendor(getVendor());
		dp.setVersion("1.0"); //$NON-NLS-1$
		dp.setType(PlatformType.EXECUTABLE_EMULATOR_LITERAL);
	}
	
	/** Create Device and RuntimePlatformDefinition contents for the platform */
	public void createDevices(DevicePlatform dp){		
		//Empty device list holder for adding devices
		EList deviceList = dp.getDevices();
		
		//Get list of devices and create Device instance of each
		List deviceNames = getMultipleValues(ueiProperties.getProperty(UEI_PROP_DEVICE_LIST,""), ","); //$NON-NLS-1$ //$NON-NLS-2$
		
		Device device = null;
		for(int i = 0; i < deviceNames.size(); i++){
			device = DevicesFactory.eINSTANCE.createDevice();
			
			//Name
			String deviceName = (String)deviceNames.get(i);
			device.setName(deviceName);
			
			createDeviceDetails(device);
			//TODO check is this needed here
			//setDeviceDetails(device);
			
			//Protection Domain
			createSecurityDomains(device);
			
			//RuntimePlatformDefinitions
			RuntimePlatformDefinition rpd = createRuntimePlatformDefinition(deviceName);
			device.getRuntimePlatformDefinitions().add(rpd);
			dp.getRuntimePlatformDefinitions().add(rpd);
			
			//DeviceDescription
			device.setDeviceDescription(createDeviceDescription(deviceName));
			
			//TODO Configuration
			//Configuration conf = createDirConfiguration();
			//device.setConfiguration(conf);
			
			//Add to list of devices
			deviceList.add(device);
		}
	}
	
	public abstract DeviceDescription createDeviceDescription(String deviceName);
	public abstract void createDeviceDetails(Device device);

	private void createSecurityDomains(Device device){
		String securityDomains = ueiProperties.getProperty(device.getName() + UEI_PROP_DEVICE_SECURITY_DOMAINS);
		
		if(securityDomains != null){
			EList securityDomainList = device.getProtectionDomains();
			List domains = getMultipleValues(securityDomains, ","); //$NON-NLS-1$
			ProtectionDomain pd = null;
			for(int j = 0; j < domains.size(); j++){
				pd = DevicesFactory.eINSTANCE.createProtectionDomain();
				pd.setName((String)domains.get(j));
				securityDomainList.add(pd);
			}
		}
	}
	
	public RuntimePlatformDefinition createRuntimePlatformDefinition(String deviceName){
		RuntimePlatformDefinition rpd = RuntimesFactory.eINSTANCE.createRuntimePlatformDefinition();
		rpd.setName(getRuntimePlatformDefinitionName(deviceName));
		
		//Device configurations
		createDeviceConfigurations(rpd, deviceName);
		
		//Service Apis
		createServiceApis(rpd, deviceName);
		
		//Device profiles
		createDeviceProfiles(rpd, deviceName);
		
//		Create ServiceApis for the DeviceProfile
		//setServiceApisForProfile(rpd, dp, deviceName);
		
		return rpd;
	}
	
	private String getDeviceDescriptionIdentifier(String deviceName) {
		return getVendor()+" "+deviceName; //$NON-NLS-1$
	}

	private void createDeviceProfiles(RuntimePlatformDefinition rpd, String deviceName){
		EList deviceProfiles = rpd.getDeviceProfiles();
		
		String apiFiles = ueiProperties.getProperty(deviceName + useApiString, ""); //$NON-NLS-1$
		
		List l = getLibFiles(apiFiles);
		
		IServiceApiImporter[] importers = AbstractServiceApiImporter.SERVICE_API_IMPORTERS;
		File apiFile = null;
		DeviceProfile[] importedApis = null;
		
		for(int i = 0; i < l.size(); i++){
			apiFile = (File) l.get(i);
		
			importedApis = null;
			for(int j = 0; (importedApis == null) && (j < importers.length); j++){
				importedApis = importers[j].getProfileApis(apiFile);
			}
			
			for(int j = 0; j < importedApis.length; j++){
				deviceProfiles.add(importedApis[j]);
			}
		}
	}
	
	private void createDeviceConfigurations(RuntimePlatformDefinition rpd, String deviceName){
		EList dconfigs = rpd.getDeviceConfigurations();
		
		String apiFiles = ueiProperties.getProperty(deviceName + useApiString, ""); //$NON-NLS-1$
		
		List l = getLibFiles(apiFiles);
		
		IServiceApiImporter[] importers = AbstractServiceApiImporter.SERVICE_API_IMPORTERS;
		File apiFile = null;
		DeviceConfiguration[] importedApis = null;
		
		for(int i = 0; i < l.size(); i++){			
			apiFile = (File) l.get(i);
		
			importedApis = null;
			for(int j = 0; (importedApis == null) && (j < importers.length); j++){
				importedApis = importers[j].getConfigurationApis(apiFile);
			}
			
			for(int j = 0; j < importedApis.length; j++){
				dconfigs.add(importedApis[j]);
			}
		}
	}

	private void createServiceApis(RuntimePlatformDefinition rpd, String deviceName){		
		String apis = ueiProperties.getProperty(deviceName + useApiString, ""); //$NON-NLS-1$
		
		createServiceApisForList(rpd, null, apis);
	}
	
	/** Locate profile specific service apis and add to the deviceProfile argument.*/
	private void setServiceApisForProfile(RuntimePlatformDefinition rpd, DeviceProfile deviceProfile, String deviceName){		
		//TODO locate ServiceApis and implement this -> only profile specific serviceApis
		String apiFiles = ueiProperties.getProperty(deviceName + useApiString, ""); //$NON-NLS-1$
		
		//Get list of serviceApis for profile
		EList rpdSApiList = rpd.getServiceApis();
		
		EList dpSApiList = deviceProfile.getServiceApis();
		
		//
		
		//createServiceApisForList(rpd, deviceProfile, apiFiles);
	}
	
	private void createServiceApisForList(RuntimePlatformDefinition rpd, DeviceProfile deviceProfile, String apiFileNames){
		EList sapis = null;
		if ( deviceProfile != null ) {
			sapis = deviceProfile.getServiceApis();
		} else {
			sapis = rpd.getServiceApis();
		}
		
		List l = getMultipleValues(apiFileNames, ","); //$NON-NLS-1$
		
		IServiceApiImporter[] importers = AbstractServiceApiImporter.SERVICE_API_IMPORTERS;
		ServiceApi[] importedApis = null;
		ImplementationRef implRef = null;
		File sapiFile = null;
		Vector v = new Vector();
		
		for(int i = 0; i < l.size(); i++){			
			sapiFile = new File((String)l.get(i));
		
			importedApis = null;
			for(int j = 0; (importedApis == null) && (j < importers.length); j++){
				importedApis = importers[j].getServiceApis(sapiFile);
			}
			
			for(int j = 0; j < importedApis.length; j++){				
				String tmpName = importedApis[j].getName() + importedApis[j].getVersion();
				
				if(!v.contains(tmpName)){					
					sapis.add(importedApis[j]);
					v.add(tmpName);
				}
			}
		}
	}	
	
	/** Returns list of String elements */
	private List getMultipleValues(String s, String delim){		
		List l = new Vector();
		StringTokenizer st = new StringTokenizer(s, delim);
		while (st.hasMoreTokens()) {
			l.add(st.nextToken().trim());	
		}
		return l;
	}
	
	private List getLibFiles(String classpath){		
		List l = new Vector();
		StringTokenizer st = new StringTokenizer(classpath, ","); //$NON-NLS-1$
		File file = null;
		
		while (st.hasMoreTokens()) {
			file = new File(st.nextToken().trim());
			if(file.getParentFile().getName().equals(UEI_LIB_DIR)){
				l.add(file);	
			}
		}
		return l;		
	}
	
	/** 
	 * Creates name and version strings from "Name-version" string 
	 * 
	 * @param nameAndVersion String of type String1-String2
	 * @return String[], where String[0] equals String1, and String[1] equals String2
	 */
	private String[] getNameAndVersion(String nameAndVersion){		
		String name = nameAndVersion.substring(0, nameAndVersion.indexOf("-")); //$NON-NLS-1$
		String version = nameAndVersion.substring(nameAndVersion.indexOf("-") + 1); //$NON-NLS-1$		
		String[] s = {name, version};		
		return s;
	}
	
	/** Returns the file name of without the ending after "." character */
	private String getServiceApiName(File file){		
		String fileName = file.getName();
		int end = fileName.lastIndexOf("."); //$NON-NLS-1$
		
		return fileName.substring(0, end);
	}
}

