/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 
package org.eclipse.mtj.executable.uei;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.mtj.api.deployment.Deployment;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.extension.DevicePlatformProvider;
import org.eclipse.mtj.api.model.IExecutablePlatform;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.core.IEclipseMtjCoreConstants;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.core.launching.LaunchHelper;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.devide.executable.AbstractExecutable;
import org.eclipse.mtj.extension.devide.project.MtjProject;

public class UeiExecutable extends AbstractExecutable implements IExecutablePlatform {
	/** The property defining the UEI classpath */
	public static final String PROP_CLASSPATH = "bootclasspath"; //$NON-NLS-1$

	/** The property defining the known UEI arguments */
	public static final String PROP_KNOWN_ARGUMENTS = "uei.arguments"; //$NON-NLS-1$
	
	/** The property defining the devices provided by this emulator */
	public static final String PROP_DEVICE_LIST = "device.list"; //$NON-NLS-1$
	
	/** The executable to be run **/
	public static final String EXECUTABLE_PATH = "bin/emulator"; //$NON-NLS-1$

	// The File instance representing the emulator being executed
	private File emulatorExecutable;
	
	// The UEI properties
	private Properties ueiProperties;

	// The UEI known arguments
	private Set knownArguments;
	
	private DevicePlatform devicePlatform;
	private Device device;
	
	public UeiExecutable(DevicePlatform devicePlatform, Device device) {
		this.devicePlatform = devicePlatform;
		this.device = device;
	}
	
	/**
	 * Return the qualified executable to be run for UEI emulation.
	 * 
	 * @param directory
	 * @return
	 */
	public static File getEmulatorExecutable(File directory) {
		return new File(directory, EXECUTABLE_PATH);
	}
	
	public File getExecutable() throws CoreException {
		if (emulatorExecutable == null) {
			DevicePlatformProvider devicePlatformProvider = 
				(DevicePlatformProvider)devicePlatform.getDevicePlatformProvider();
			if ( devicePlatformProvider == null ) {
				IStatus status = new Status(
						IStatus.ERROR, 
						IEclipseMtjCoreConstants.PLUGIN_ID, 
						MtjCoreErrors.DEVICE_PLATFORM_PROVIDER_NOT_DEFINED_ERROR, 
						MtjCoreErrors.getErrorMessage(MtjCoreErrors.DEVICE_PLATFORM_PROVIDER_NOT_DEFINED_ERROR), null);
				throw new CoreException(status);
			}
			String rootDir;
			try {
				rootDir = devicePlatformProvider.getRootDirectory(devicePlatform);
			} catch (MtjException e) {
				IStatus status = new Status(
						IStatus.ERROR, 
						IEclipseMtjCoreConstants.PLUGIN_ID, 
						MtjCoreErrors.DEVICE_PLATFORM_ROOT_ERROR, 
						MtjCoreErrors.getErrorMessage(MtjCoreErrors.DEVICE_PLATFORM_ROOT_ERROR), e);
				throw new CoreException(status);
			}
			emulatorExecutable = getEmulatorExecutable(new File(rootDir));
		}
		
		return emulatorExecutable;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.mtj.api.model.IExecutablePlatform#getArguments(org.eclipse.debug.core.ILaunchConfiguration, boolean, int)
	 */
	public String getArguments(ILaunchConfiguration configuration, Deployment deployment, 
			boolean debugMode, int debugPort, String natureId, Device device) throws CoreException {
		// TODO Fix progress monitoring
		IProgressMonitor monitor = new NullProgressMonitor();
		
		// Build up the program arguments based on the launch config
		// selections
		ArrayList arguments = new ArrayList();
		// Add all of the command line arguments
		if (debugMode) addDebugArguments(arguments, debugPort, true /*devicePlatform.isDebugEnabled()*/ );
		
		if (!isPredeploymentRequired(configuration)) {
			// We assume that predeployment is only required if we
			// are running with a JAD/JAR combination.  This may need
			// to be changed later.
			addClasspath(arguments, configuration, natureId, monitor);
		}
		
		addDeviceArgument(arguments, configuration, device);
		addDescriptorArgument(arguments, deployment, configuration, device);
		addMiscellaneousArguments(arguments, configuration, device);
		addSecurityDomain(arguments, configuration);
		addExtraLaunchArguments(arguments, configuration);

		// TODO should be resolved from configuration
		if (/*isExecutableTargetRequired(configuration)*/ true) {
			addEmulationTarget(arguments, configuration, deployment, device);
		}
		
		// The keep alive port *must* be last
		if (isKeepAlivePortRequired(configuration)) {
			addKeepAlivePort(arguments, configuration);
		}
		
		// Convert the arguments list into a string...
		return convertArgsToString(arguments);
	}
	
	private boolean isPredeploymentRequired(ILaunchConfiguration configuration) throws CoreException {
		DevicePlatform devicePlatform = LaunchHelper.getTargetLaunchDevicePlatform(configuration);
		return devicePlatform.isPredeploymentRequired();
	}
	
	protected void addDescriptorArgument(ArrayList arguments, ILaunchConfiguration configuration, Deployment deployment) 
	throws CoreException 
	{
		if (isKnownArgument("Xdescriptor")) { //$NON-NLS-1$
			super.addDescriptorArgument(arguments, configuration, deployment);
		}
	}
	
	/**
	 * Add the device argument to the list of arguments.
	 * 
	 * @param arguments
	 * @param configuration
	 * @throws CoreException
	 */
	protected void addDeviceArgument(ArrayList arguments, ILaunchConfiguration configuration, Device device) 
		throws CoreException 
	{
		if (isKnownArgument("Xdevice")) { //$NON-NLS-1$
			addLaunchConfigurationArgument(arguments, configuration, 
					device, "-Xdevice:", false); //$NON-NLS-1$
		}
	}
	
	/**
	 * Return a boolean indicating whether the specified argument 
	 * is a known argument.
	 * 
	 * @param string
	 * @return
	 * @throws CoreException
	 */
	protected boolean isKnownArgument(String argument) 
		throws CoreException 
	{
		Set knownArgs = getKnownUEIArguments(device);
		return 
			(knownArgs.size() == 0) ||
			knownArgs.contains(argument);
	}

	/**
	 * Return the command-line arguments known by the UEI emulator.
	 * 
	 * @return
	 * @throws CoreException
	 */
	protected Set getKnownUEIArguments(Device device) 
		throws CoreException 
	{
		if (knownArguments == null) {
			knownArguments = new HashSet();
			
			// Search for the known UEI arguments 
			String commaSeparatedArgs = getUEIProperties().getProperty(PROP_KNOWN_ARGUMENTS);
			if (commaSeparatedArgs == null) {
				// See if there is a device-specific reference
				String propName = device.getName() + "." + PROP_KNOWN_ARGUMENTS; //$NON-NLS-1$

				commaSeparatedArgs = getUEIProperties().getProperty(propName, ""); //$NON-NLS-1$
			}
			
			// Break up the comma-separated list
			StringTokenizer st = new StringTokenizer(commaSeparatedArgs, ","); //$NON-NLS-1$
			while (st.hasMoreTokens()) {
				knownArguments.add(st.nextToken().trim());
			}
		}
		
		return knownArguments;
	}

	/**
	 * Return the UEI emulator properties.
	 * 
	 * @return the UEI properties.
	 * @throws CoreException
	 */
	public Properties getUEIProperties() 
		throws CoreException 
	{
		if (ueiProperties == null) {
			ueiProperties = UeiPropertiesCache.instance.getUEIProperties(getExecutable());
		}
		
		return ueiProperties;
	}

	/**
	 * Return a boolean indicating whether the emulation target
	 * (midlet class) should be passed to the emulator.
	 * 
	 * @return
	 */
	protected boolean isExecutableTargetRequired(ILaunchConfiguration configuration) throws CoreException {
		DevicePlatform devicePlatform = LaunchHelper.getTargetLaunchDevicePlatform(configuration);
		return devicePlatform.isExecutableTargetRequired();
	}
	
	/**
	 * Return a boolean indicating whether the keep alive port
	 * should be passed to the emulator.
	 * 
	 * @return
	 */
	protected boolean isKeepAlivePortRequired(ILaunchConfiguration configuration) throws CoreException {
		DevicePlatform devicePlatform = LaunchHelper.getTargetLaunchDevicePlatform(configuration);
		return devicePlatform.isKeepAlivePortRequired();
	}
	
	protected IMtjProject getMtjProject(IJavaProject javaProject) throws MtjException {
		return MtjProject.getMtjProject(javaProject);
	}
}
