/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 

package org.eclipse.mtj.executable.uei;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.mtj.internal.utils.ColonDelimitedProperties;
import org.eclipse.mtj.internal.utils.Utils;

/**
 * A cache holding UEI properties such that they don't need
 * to be read/parsed from the emulators each and every time.
 */
public class UeiPropertiesCache {
	/** Singleton instance of this cache **/
	public static final UeiPropertiesCache instance = new UeiPropertiesCache();

	/** A non-standard property in the properties to hold the emulator version */
	public static final String PROP_TOOLKIT_NAME = "eclipse.mtj.toolkit.name"; //$NON-NLS-1$
	
	// Cache information
	private static final String CACHE_DIRECTORY = "ueicache"; //$NON-NLS-1$
	private static final String CACHE_FILE = "cache.properties"; //$NON-NLS-1$

	// Parameter used to query UEI properties
	private static final String PARM_XQUERY = "-Xquery"; //$NON-NLS-1$
	
	// A map of emulator path to cached properties files
	private Properties cacheMap;
	
	// A map of emulator path to read properties file instances
	private Map propertiesMap;
	
	// The cache paths
	private IPath cachePath;
	private File cachePropertiesFile;
	
	// The cache 
	/**
	 * Private constructor for singleton access
	 */
	private UeiPropertiesCache() {
		super();
		propertiesMap = new HashMap();
	}

	/**
	 * Clear out the UEI properties cache.
	 * 
	 * @param monitor progress monitor
	 * @throws CoreException
	 */
	public void clearCache(IProgressMonitor monitor)
		throws CoreException
	{
		if (monitor == null) monitor = new NullProgressMonitor();
		
		File cacheDirFile = getCachePath().toFile();
		if (cacheDirFile.exists()) {
			File[] files = cacheDirFile.listFiles();
			for (int i = 0; i < files.length; i++) {
				File file = files[i];
				if (file.isFile() && file.getName().endsWith(".properties")) { //$NON-NLS-1$
					file.delete();
				}
			}
		}
	}
	
	/**
	 * Return the UEI properties associated with the
	 * specified emulator.
	 * 
	 * @param emulatorExecutable the file system file that represents the 
	 * emulator executable.
	 * 
	 * @return the properties that define this UEI emulator.
	 * @throws CoreException
	 */
	public Properties getUEIProperties(File emulatorExecutable) 
		throws CoreException
	{
		Properties properties = null;
		
		try {
			//properties = getCachedProperties(emulatorExecutable);
			if (properties == null) {
				// Not in the cache... Need to get it from the emulator
				properties = Utils.executableExists(emulatorExecutable) ? 
						getUEIProperties(getEmulatorOutput(emulatorExecutable),getToolkitName(emulatorExecutable),getToolkitProfileAndConfiguration(emulatorExecutable)) : 
						new Properties();		 				
				addEmulatorPropertiesToCache(emulatorExecutable, properties);
			}
		} catch (IOException e) {
			MtjCorePlugin.throwCoreException(IStatus.ERROR, -9999, e);
		}
		
		return properties;
	}
	
	/**
	 * Return the UEI emulator properties as specified by the emulator output.
	 * 
	 * @return the UEI properties
	 * @throws CoreException
	 * @throws IOException
	 */
	public Properties getUEIProperties(String output, String toolkitName, Properties profileAndConf) 
		throws CoreException, IOException 
	{
		Properties ueiProperties = parseUEIPropertiesString(output);

		// Add in a non-standard property to hold the name
		if ((toolkitName != null) && (toolkitName.length() > 0)) {
			ueiProperties.setProperty(PROP_TOOLKIT_NAME, toolkitName);
		}
			
		String tmp = profileAndConf.getProperty("Profile"); //$NON-NLS-1$
		if(tmp != null){
			ueiProperties.setProperty("Profile", tmp); //$NON-NLS-1$
		}
		tmp = profileAndConf.getProperty("Configuration"); //$NON-NLS-1$
		if(tmp != null){
			ueiProperties.setProperty("Configuration", tmp); //$NON-NLS-1$
		}
		
		return ueiProperties;
	}
	
	/**
	 * Add the emulator properties to the cache.
	 * 
	 * @param emulatorExecutable
	 * @param properties
	 * @throws IOException
	 */
	private void addEmulatorPropertiesToCache(File emulatorExecutable, Properties properties) 
		throws IOException 
	{
		String path = emulatorExecutable.toString();
		
		// Figure a place to stash the properties
		File cacheDirectory = getCachePath().toFile();
		File cacheFile = File.createTempFile("emulator_", ".properties", cacheDirectory); //$NON-NLS-1$ //$NON-NLS-2$
		
		// Store the properties
		FileOutputStream fos = new FileOutputStream(cacheFile);
		properties.store(fos, "Emulator properties for " + path); //$NON-NLS-1$
		fos.close();
		
		// Update and store the cache map
		Properties cache = getCacheMap();
		cache.put(path, cacheFile.getName());
		writeCache(cache);
	}

	/**
	 * Get the properties that were previously cached for this
	 * executable or <code>null</code> if no properties were cached
	 * or they are out of date.
	 * 
	 * @param emulatorExecutable
	 * @return
	 * @throws IOException
	 */
	private Properties getCachedProperties(File emulatorExecutable) 
		throws IOException 
	{
		String path = emulatorExecutable.getAbsolutePath();
		
		// Try and pull it from the properties map
		Properties properties = (Properties) propertiesMap.get(path);
		if (properties == null) {
			// We haven't read the properties yet.  Let's see if 
			// we have read from the emulator yet
			String propsPath = getCacheMap().getProperty(path);
			if (propsPath != null) {
				// We have a copy in the cache directory
				properties = readCachedProperties(emulatorExecutable, propsPath);
				propertiesMap.put(path, properties);
			}
		}
		
		return properties;
	}

	/**
	 * Get the map that describes the cache.
	 * 
	 * @return
	 * @throws IOException
	 */
	private Properties getCacheMap() throws IOException {
		if (cacheMap == null) {
			cacheMap = readCache();
		}
		
		return cacheMap;
	}
	
	/**
	 * Return the path to the UEI properties cache directory.
	 * 
	 * @return the IPath instance representing the UEI properties cache directory
	 */
	private IPath getCachePath() {
		if (cachePath == null) {
			IPath pluginStatePath = MtjCorePlugin.getDefault().getStateLocation();
			cachePath = pluginStatePath.append(CACHE_DIRECTORY);
			
			File cacheDirectory = cachePath.toFile();
			if (!cacheDirectory.exists()) {
				cacheDirectory.mkdir();
			}
		}

		return cachePath;
	}
	
	/**
	 * Return the properties cache file.
	 * 
	 * @return the UEI properties cache file.
	 */
	private File getCachePropertiesFile() {
		if (cachePropertiesFile == null) {
			cachePropertiesFile = getCachePath().append(CACHE_FILE).toFile();
			System.out.println("[INFO] MTJ: Reading emulator properties from cache file: " + getCachePath().append(CACHE_FILE).toString()); //$NON-NLS-1$
		}
		
		return cachePropertiesFile;
	}

	/**
	 * Return the toolkit name from the specified executable.
	 * 
	 * @param executable
	 * @return
	 */
	private static String getToolkitName(File executable) {
		String toolkitName = null;
		
		try {
			// Capture and parse the output
			String[] commandLine = new String[] { executable.toString(), "-version" };  //$NON-NLS-1$
			String output = Utils.getStandardOutput("UEI Emulator Version", commandLine); //$NON-NLS-1$
			BufferedReader reader = new BufferedReader(new StringReader(output));
			toolkitName = reader.readLine();
		} catch (CoreException e) {
		} catch (IOException e) {
		}
		
		if (toolkitName == null) toolkitName = "Generic UEI Toolkit"; //$NON-NLS-1$
		
		return toolkitName;
	}
		
	private static Properties getToolkitProfileAndConfiguration(File executable){
			Properties tmpProps = null;
			try {
				// Capture and parse the output
				tmpProps = parseUEIPropertiesString(getEmulatorVersionOutput(executable));	
			} catch (CoreException e) {
				e.printStackTrace();
			} catch (IOException e) {
				e.printStackTrace();
			}
	
			if(tmpProps == null){
				tmpProps = new Properties();
			}
			
			return tmpProps;
	}

	private static String getEmulatorVersionOutput (File emulatorExecutable) throws CoreException {
		String[] commandLine = new String[] { emulatorExecutable.toString(), "-version"  }; //$NON-NLS-1$
		return Utils.getStandardOutput("UEI Emulator version", commandLine); //$NON-NLS-1$
	}

	private static String getEmulatorOutput (File emulatorExecutable) throws CoreException {
		String[] commandLine = new String[] { emulatorExecutable.toString(), PARM_XQUERY };
		return Utils.getStandardOutput("UEI Emulator Query", commandLine); //$NON-NLS-1$
	}
	
	/**
	 * Parse the properties String containing the UEI properties.
	 * 
	 * @param contents the string containing the properties
	 * @return the newly parsed properties object
	 * 
	 * @throws IOException
	 */
	public static Properties parseUEIPropertiesString(String contents) 
		throws IOException 
	{
		ColonDelimitedProperties props = new ColonDelimitedProperties();
		
		StringReader reader = new StringReader(contents);
		try {
			props.load(reader);
		} catch (Exception e) {
			MtjCorePlugin.log(IStatus.WARNING, "Error parsing UEI properties", e); //$NON-NLS-1$
			MtjCorePlugin.log(IStatus.WARNING, contents);
		}
		
		return props;
	}
	
	/**
	 * Read the cached properties information.
	 * 
	 * @return the properties that were read
	 * @throws IOException
	 */
	private Properties readCache() 
		throws IOException 
	{
		Properties properties = new Properties();
		
		File cacheFile = getCachePropertiesFile();
		if (cacheFile.exists()) {
			FileInputStream fis = new FileInputStream(cacheFile);
			properties.load(fis);
			fis.close();
		}
		
		return properties;
	}

	/**
	 * Read the properties found in the specified properties path.
	 * 
	 * @param propsPath
	 * @return
	 * @throws IOException
	 */
	private Properties readCachedProperties(File emulatorExecutable, String propsPath) 
		throws IOException 
	{
		ColonDelimitedProperties properties = null;

		File file = getCachePath().append(propsPath).toFile(); 
		if (file.exists()) {
			if (file.lastModified() < emulatorExecutable.lastModified()) {
				// The cached file is out of date with respect to the
				// emulator.  Go ahead and delete it.
				file.delete();
			} else {
				// The cache file exists and is up to date.  Go ahead and read it in.
				properties = new ColonDelimitedProperties();
				FileInputStream fis = new FileInputStream(file);
				properties.load(fis);
				fis.close();
			}
		}
		
		return properties;
	}
	
	/**
	 * Write out the cache properties.
	 * 
	 * @param properties the properties to be written
	 * @throws IOException
	 */
	private void writeCache(Properties properties) 
		throws IOException 
	{
		if (properties != null) {
			File cacheFile = getCachePropertiesFile();
			FileOutputStream fos = new FileOutputStream(cacheFile);
			properties.store(fos, "UEI Properties Cache"); //$NON-NLS-1$
			fos.close();
		}
	}
}
