/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.extension.psp.impl;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Collections;
import java.util.Iterator;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.common.util.WrappedException;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceFactoryImpl;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.extension.PersistentStoreProvider;
import org.eclipse.mtj.api.extension.impl.MtjExtensionImpl;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.exception.MtjNotFoundException;
import org.eclipse.mtj.extension.psp.Messages;
import org.eclipse.mtj.extension.psp.PersistentStoreProviderPlugin;
import org.eclipse.osgi.service.datalocation.Location;

public class PersistentStoreProviderImpl extends MtjExtensionImpl implements PersistentStoreProvider {
	private boolean storeToHome = false;
	
	public PersistentStoreProviderImpl() {
		super();		
		setId(PersistentStoreProviderPlugin.getDefault().getBundle().getSymbolicName());
		setDescription(Messages.PersistentStoreProviderImpl_PluginDescription);
		setVendor(Messages.PersistentStoreProviderImpl_PluginVendor);
		setVersion(Messages.PersistentStoreProviderImpl_PluginVersion);
		setType(ExtensionType.PERSISTENT_STORE_PROVIDER_LITERAL);
	}
	
	public void store(Object data, int dataType, String dataExtension, String ownerPluginName,String dataKey) throws MtjException {
		try {
			String filename = getFileName(dataExtension, ownerPluginName, dataKey);
			store( data,  dataType,  dataExtension,  filename);
		} catch (Exception ex) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR),ex);
		}			
	}
	
	public void store(Object data, int dataType, String dataExtension, String filename) throws MtjException {
		try {
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap()
				.put(dataExtension, new XMIResourceFactoryImpl());
			
			ResourceSet resSet = new ResourceSetImpl();

			Resource res = resSet.createResource(URI.createURI(filename));

			if (data instanceof EList) {
				Iterator it = ((EList)data).iterator();
				while (it.hasNext()) {
					res.getContents().add(it.next());
				}
			} else {
				res.getContents().add(data);
			}
			
			res.save(Collections.EMPTY_MAP);
		} catch (Exception ex) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR),ex);
		}
	}

	public Object get(int dataType, String dataExtension, String ownerPluginName,
		String dataKey) throws MtjException {
		
		try {
			String filename = getFileName(dataExtension, ownerPluginName, dataKey);
			return get( dataType,  dataExtension,  filename);
		} catch (MtjException ex) {
			throw ex;
		} catch (Exception ex) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR),ex);
		}			
	}

	public Object get(int dataType, String dataExtension, String filename) throws MtjException {
		try {
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap()
				.put(dataExtension, new XMIResourceFactoryImpl());
			
			ResourceSet resSet = new ResourceSetImpl();
			Resource res = resSet.getResource(URI.createURI(filename),true);
	
			return res.getContents();
		} catch (WrappedException ex) {
			if ( ex.exception() != null && ex.exception() instanceof FileNotFoundException ) {
				throw new MtjNotFoundException(ex.getLocalizedMessage());
			} else {
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR),ex);
			}
		} catch (Exception ex) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.EMF_DATA_MODEL_ERROR),ex);
		}		
	}
	
	private String getFileName(String dataExtension, String ownerPluginName, String dataKey) throws IOException {
		String url = ""; //$NON-NLS-1$
		StringBuffer sb = new StringBuffer();		
		IPath path = new Path(""); //$NON-NLS-1$

		/*
		 * storing persistent data to user home 
		 */
		if(storeToHome) {
			Location location = Platform.getUserLocation();
			url = location.getURL()+".mtj/data/"; //$NON-NLS-1$
		
			// making sure that storaging folder exists
			File file = new File(url);
			if(!file.exists())
			file.mkdirs();

			sb.append(url);
		/*
		 * storing persistent data to psp plugins data folder
		 */			
		} else {
			url = FileLocator.toFileURL(FileLocator.find(
					PersistentStoreProviderPlugin.getDefault().getBundle(), path, null)).toString();
			sb.append(url);
			sb.append("data/"); //$NON-NLS-1$
		}		
				
		sb.append(ownerPluginName);
		sb.append("/"); //$NON-NLS-1$
		sb.append(dataKey);
		sb.append("."); //$NON-NLS-1$
		sb.append(dataExtension);
		
		return sb.toString();
	}
}
