/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM - initial API and implementation
 *******************************************************************************/

package org.eclipse.mtj.extension.smgmt.impl.IBM;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.extension.SecurityManagement;
import org.eclipse.mtj.api.extension.impl.MtjExtensionImpl;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.smgmt.IBM.SecurityManagementPlugin;

public class SecurityManagementImpl extends MtjExtensionImpl implements
		SecurityManagement {

	// private variables
	private String storeType = "JKS"; //$NON-NLS-1$
	private String aliaskey = null;
	private String passwrd = null;
	private String keyStoreNameLoc = ""; //$NON-NLS-1$
	private String validity = "365"; //$NON-NLS-1$
	
	private IPreferenceStore securityProviderPrefStore;
	
	public SecurityManagementImpl() {
		super();
		setId(SecurityManagementPlugin.getDefault().getBundle().getSymbolicName());
		setType(ExtensionType.PERSISTENT_STORE_PROVIDER_LITERAL);
		setVendor(Messages.SecurityManagementImpl_PluginVendor); 
		setVersion(Messages.SecurityManagementImpl_PluginVersion); 
		setDescription(Messages.SecurityManagementImpl_Security_manager); 
		setType(ExtensionType.SECURITY_MANAGEMENT_LITERAL);
		securityProviderPrefStore = SecurityManagementPlugin.getDefault().getPreferenceStore();
	}

	public String [] openKeyStore(String keyStore, String storePass, IProgressMonitor monitor) throws MtjException {
		monitor.beginTask(Messages.SecurityManagementImpl_Opening_key_store, 100); 
		monitor.worked(10);
		BufferedReader cmdOutputStream;
		String[] cmdArgs = generateOpenKeyStoreCmd(keyStore,storePass);
		Process p = runSecurityCmd(cmdArgs);
		if (p != null)
		{
			cmdOutputStream = new BufferedReader(new InputStreamReader(p.getInputStream()));
		}
		else 
		{
			StringBuffer str = new StringBuffer(""); //$NON-NLS-1$
			for (int i = 0; i < cmdArgs.length; i++)
			{
				str.append(" " + cmdArgs[i]); //$NON-NLS-1$
			}
			
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) + Messages.SecurityManagementImpl_Could_not_execute + " [" + str +"]");   //$NON-NLS-1$ //$NON-NLS-2$
		}
		
		String cmdOutput;
		//ArrayList aliases = new ArrayList();
		List<String> aliases = new ArrayList<String>();



		try {
			while ((cmdOutput=cmdOutputStream.readLine()) != null) {
				monitor.subTask(cmdOutput);
				monitor.worked(25);	
				if (cmdOutput.toLowerCase().indexOf("error")>=0) {  //$NON-NLS-1$
					monitor.done();
					if (cmdOutput.toLowerCase().indexOf("invalid keystore format")>=0) {  //$NON-NLS-1$
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SECURITY_BAD_KEY_TYPE) + " " + cmdOutput); //$NON-NLS-1$
					} else if (cmdOutput.toLowerCase().indexOf("password was incorrect")>=0) {  //$NON-NLS-1$
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SECURITY_BAD_KEYSTORE_OR_PASSWORD) + " " + cmdOutput); //$NON-NLS-1$
					} else {
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) + " " + cmdOutput); //$NON-NLS-1$	
					}
				} else if (cmdOutput.indexOf(",")>=0) { //$NON-NLS-1$
					StringTokenizer strtok = new StringTokenizer(cmdOutput, ".,"); //$NON-NLS-1$
					aliases.add(strtok.nextToken());
					//aliases.add(strtok.nextToken());
				}
			}
		} catch (IOException ee) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY), ee);
		}
		monitor.worked(100);
		monitor.done();
		//return (String[])aliases.toArray(new String[aliases.size()]);
		return (String[])aliases.toArray(new String[aliases.size()]);
	}

	public String getStoreType() throws MtjException{
		return storeType;
	}

	public String getAliaskey() throws MtjException{
		return aliaskey;
	}

	public String getPassWrd() throws MtjException{
		return passwrd;
	}

	public String getKeyStoreNameLoc() throws MtjException{
		return keyStoreNameLoc;
	}

	public String getValidity()throws MtjException {
		return validity;
	}

	public void setStoreType(String storeType) throws MtjException{
		this.storeType = storeType;
	}

	public void setAliaskey(String aliasKey) throws MtjException{
		this.aliaskey = aliasKey;
	}

	public void setPassWrd(String passWrd) throws MtjException{
		this.passwrd = passWrd;
	}

	public void setKeyStoreNameLoc(String keyStoreNameLoc) throws MtjException{
		this.keyStoreNameLoc = keyStoreNameLoc;
	}

	public void setValidity(String validity) throws MtjException{
		this.validity = validity;
	}

	public void setValues(String loc, String alias, String psswd, String strtype) throws MtjException {
		
		storeType 			= strtype;
		aliaskey 			= alias;
		passwrd 			= psswd;
	    keyStoreNameLoc 	= loc;	
		
	}

	
	/*
	 * Generating the command to Open a Key Store and display its contents.
	 * 
	 */
	private String[] generateOpenKeyStoreCmd( String keyStore, String storePasswd) throws MtjException {
		
		String[] openKeyStoreCmdArgs = {getSecurityManagementTool(),
										getConsoleEncoding(),
										SecurityManagementImplConstants.LIST,
										SecurityManagementImplConstants.STORETYPE,storeType,
										SecurityManagementImplConstants.KEYSTORE,keyStore,
										SecurityManagementImplConstants.STOREPASS,storePasswd};
		return openKeyStoreCmdArgs;
	}
    /*
	 * Get the Security Management tool from pref store location.
	 * 
	 */
	private final String getSecurityManagementTool() throws MtjException {	
		String securityToolLocation = getToolLocation(null);
		
		if (securityToolLocation == null ||
			securityToolLocation.length() <= 0 ||
			securityToolLocation.equals(Messages.SecurityManagementImpl_Specify_home_directory)	 
			) {
				String message = MessageFormat.format (Messages.SecurityManagementImpl_GetSecurityManagmentException, 
						new Object[] {getId(), Messages.SecurityManagementImpl_Security_tool_not_configured_correctly, Messages.SecurityManagementImpl_6});
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SECURITY_MANAGER_NOT_CONFIGURED) + "\n" + message);     //$NON-NLS-1$
			}
		
		StringBuffer buffer = new StringBuffer();
		buffer.append(securityToolLocation)
		.append(File.separator)
		.append("bin") //$NON-NLS-1$						
		.append(File.separator)
		.append("keytool.exe"); //$NON-NLS-1$
			
		return buffer.toString();
	}
	private static String getConsoleEncoding ()
	{
		return  "-J-Dconsole.encoding=" + System.getProperty("file.encoding"); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	private Process runSecurityCmd (String[] cmd) throws MtjException {
		
		Process p = null;

		try {p =	Runtime.getRuntime().exec(cmd);} 
		catch (IOException e){
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY));	
		}
		
		if (p == null)
		 {
			StringBuffer str = new StringBuffer(""); //$NON-NLS-1$
			
			for (int i = 0; i < cmd.length; i++)
			{
				str.append(" " + cmd[i]); //$NON-NLS-1$
			}
		
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) + Messages.SecurityManagementImpl_Could_not_execute + " [" + str +"]");    //$NON-NLS-1$ //$NON-NLS-2$
		}
		
		return p;	
	}

	/**
	 * isKeyStoreSelected - user of this class will specify the keystore name/location 
	 * to manage.
	 * @return true if a keystore name and location was set during this session.
	 */
	public boolean isKeyStoreSelected () throws MtjException {
		
		if ((keyStoreNameLoc == null) || (keyStoreNameLoc.length()<= 0))
			return false;
		
		return true;
	}
	
	/*
	 * 
	 * 
	 */
	public boolean importSignedCert( String certFile, IProgressMonitor monitor) throws MtjException {
		
		boolean cmdSuccessful = true;
		String[] cmdArgs = generateImportSignedCertCmd( certFile );
		Process p = runSecurityCmd(cmdArgs);
		
		BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getInputStream()));
		
		String cmdOutput;
		try {
			while ((cmdOutput=cmdOutputStream.readLine()) != null) {
						
				if (cmdOutput.indexOf("error")>=0) {  //$NON-NLS-1$
					//MessageDialog.openError(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), "Error", cmdOutput);  //$NON-NLS-1$
					//cmdSuccessful = false;
					throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) + cmdOutput);
				}
			}
		} catch (IOException ee) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY), ee);
		}
		return cmdSuccessful;
	}
	
	/*
	 * 
	 * 
	 */
	private String[] generateImportSignedCertCmd( String certFile ) throws MtjException {

		String[] importSignedCertCmdArgs = {getSecurityManagementTool(),
					 getConsoleEncoding(),
					 SecurityManagementImplConstants.IMPORT_CERT,
					 SecurityManagementImplConstants.NOPROMPT,
					 SecurityManagementImplConstants.ALIAS,aliaskey,
					 SecurityManagementImplConstants.KEYPASS,passwrd,
					 SecurityManagementImplConstants.FILE,certFile,
					 SecurityManagementImplConstants.STORETYPE,storeType,
					 SecurityManagementImplConstants.KEYSTORE,keyStoreNameLoc,
					 SecurityManagementImplConstants.STOREPASS,passwrd};
					
		return importSignedCertCmdArgs;
	}

	public boolean generateCSR( String certFile, IProgressMonitor monitor) throws MtjException{
		
		boolean cmdSuccessful = true;
		String[] cmdArgs = generateGenerateCSRCmd( certFile);
		Process p = runSecurityCmd(cmdArgs);
		
		BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getInputStream()));
				
		String cmdOutput;
		try {
			while ((cmdOutput=cmdOutputStream.readLine()) != null) {
						
				if (cmdOutput.toLowerCase().indexOf("error")>=0) {  //$NON-NLS-1$
					//MessageDialog.openError(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), "Error", cmdOutput); //$NON-NLS-1$
					//cmdSuccessful = false;
					throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) + " " + cmdOutput); //$NON-NLS-1$
				}
			}
		} catch (IOException ee) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY), ee);
		}
		return cmdSuccessful;

	}
	
	/*
	 * 
	 * 
	 */
	private String[] generateGenerateCSRCmd( String certFile ) throws MtjException {

		String[] generateCSRCmdArgs = {getSecurityManagementTool(),
					 getConsoleEncoding(),
					 SecurityManagementImplConstants.GENERATE_CSR,
					 SecurityManagementImplConstants.ALIAS,aliaskey,
					 SecurityManagementImplConstants.FILE,certFile,
					 SecurityManagementImplConstants.STORETYPE,storeType,
					 SecurityManagementImplConstants.KEYSTORE,keyStoreNameLoc,
					 SecurityManagementImplConstants.STOREPASS, passwrd};
					
		return generateCSRCmdArgs;
	}
	
	/*
	 * 
	 * 
	 */
	public boolean deleteKey(IProgressMonitor monitor ) throws MtjException{
		
		boolean cmdSuccessful = true;
		String[] cmdArgs = generateDeleteKeyCmd( );
		Process p = runSecurityCmd(cmdArgs);
		
		BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getInputStream()));
		
		String cmdOutput;
		try {
			while ((cmdOutput=cmdOutputStream.readLine()) != null) {
						
				if (cmdOutput.toLowerCase().indexOf("error")>=0) {  //$NON-NLS-1$
					//MessageDialog.openError(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), "Error", cmdOutput); //$NON-NLS-1$
					//cmdSuccessful = false;
					throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) + " " + cmdOutput); //$NON-NLS-1$
					
				}
			}
		} catch (IOException ee) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) , ee);
		}
		return cmdSuccessful;

	}
	
	/*
	 * 
	 * 
	 */
	private String[] generateDeleteKeyCmd( ) throws MtjException {

		String[] deleteKeyCmdArgs = {getSecurityManagementTool(),
									 getConsoleEncoding(),
									 SecurityManagementImplConstants.DELETE_KEY,
									 SecurityManagementImplConstants.ALIAS,aliaskey,
									 SecurityManagementImplConstants.STORETYPE,storeType,
									 SecurityManagementImplConstants.KEYSTORE,keyStoreNameLoc,
									 SecurityManagementImplConstants.STOREPASS,passwrd};
					
		return deleteKeyCmdArgs;
	}
	
	public boolean changeStorePassword(String newStorePass, String storePass, IProgressMonitor monitor ) throws MtjException{
		
		boolean cmdSuccessful = true;
		String[] cmdArgs = generateChangeStorePasswordCmd(newStorePass, storePass);
		Process p = runSecurityCmd(cmdArgs);
		
		BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getInputStream()));
		
			
		String cmdOutput;
		try {
			while ((cmdOutput=cmdOutputStream.readLine()) != null) {
						
				if (cmdOutput.toLowerCase().indexOf("error")>=0) {  //$NON-NLS-1$
					//MessageDialog.openError(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(),"Error", cmdOutput);  //$NON-NLS-1$
					//cmdSuccessful = false;
					throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) + " " + cmdOutput); //$NON-NLS-1$
				}
			}
		} catch (IOException ee) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY), ee);
		}
		return cmdSuccessful;
	}
	
	/*
	 * Generating the command to change the key store password.
	 * 
	 */
	private String[] generateChangeStorePasswordCmd(String newStorePass, String storePasswd) throws MtjException {
		
		String[] changeStorePasswordCmdArgs = { getSecurityManagementTool(),
												getConsoleEncoding(),
												SecurityManagementImplConstants.CHANGE_STORE_PASSWD,
												SecurityManagementImplConstants.NEWSTOREPASS,newStorePass,
												SecurityManagementImplConstants.STORETYPE,storeType,
												SecurityManagementImplConstants.KEYSTORE,keyStoreNameLoc,
												SecurityManagementImplConstants.STOREPASS,storePasswd};		
		
		return changeStorePasswordCmdArgs;
	}

	public boolean createNewKey(String alias, String commonName, String orgUnit, String orgName, String localityName, String stateName, String country, IProgressMonitor monitor) throws MtjException{
		monitor.beginTask(Messages.SecurityManagementImpl_Creating_key_alias, 100); 
		

		boolean cmdSuccessful = true;
		String Dname = generateDname(commonName, orgUnit, orgName, localityName, stateName, country);
		String[] cmdArgs = generateNewKeyCmd(alias, Dname, "RSA", "SHA1withRSA" ); //$NON-NLS-1$ //$NON-NLS-2$
		Process p = runSecurityCmd(cmdArgs);
		monitor.worked(30);
		
		BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getInputStream()));

		String cmdOutput;
		try {
			while ((cmdOutput=cmdOutputStream.readLine()) != null) {
				monitor.worked(40);		
				if (cmdOutput.toLowerCase().indexOf("error")>=0) {  //$NON-NLS-1$
					//MessageDialog.openError(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), "Error", cmdOutput); //$NON-NLS-1$
					//cmdSuccessful = false;
					monitor.done();
					if (cmdOutput.toLowerCase().indexOf("alias <" + alias + "> already exists")>=0) {  //$NON-NLS-1$ //$NON-NLS-2$
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SECURITY_ALIAS_DUPLICATE) + " " + cmdOutput); //$NON-NLS-1$
					}else {
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) + " " + cmdOutput);	 //$NON-NLS-1$
					}
				}
			}
		} catch (IOException ee) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY), ee);
		}
		monitor.worked(100);
		monitor.done();
		return cmdSuccessful;

	}
	
	/*
	 * 
	 * 
	 */
	private String[] generateNewKeyCmd(String alias, String dname, String keyAlg, String sigAlg) throws MtjException {
		
		String[] newKeyCmdArgs = {
				getSecurityManagementTool(), 
							getConsoleEncoding(),
							SecurityManagementImplConstants.GENERATE_KEY,
							SecurityManagementImplConstants.ALIAS,alias,
							SecurityManagementImplConstants.DNAME,dname,
							SecurityManagementImplConstants.KEYPASS,passwrd,
							SecurityManagementImplConstants.STORETYPE,storeType,
							SecurityManagementImplConstants.KEYALG,keyAlg,
							SecurityManagementImplConstants.SIGALG,sigAlg,
							SecurityManagementImplConstants.KEYSTORE,keyStoreNameLoc,
							SecurityManagementImplConstants.STOREPASS,passwrd,
							SecurityManagementImplConstants.VALIDITY,validity};
			
		return newKeyCmdArgs;
	}
	
	/*
	 * Generating the Distinguished Name (dname) string using the given user input. 
	 * 
	 */
	public String generateDname(String commonName, String orgUnit, String orgName, String localityName, String stateName, String country) {
		String Dname = SecurityManagementImplConstants.QUOTE + SecurityManagementImplConstants.COMMON_NAME_PREFIX + commonName + SecurityManagementImplConstants.COMMA_AND_SPACE
						+ SecurityManagementImplConstants.ORGANIZATION_UNIT_PREFIX + orgUnit + SecurityManagementImplConstants.COMMA_AND_SPACE
						+ SecurityManagementImplConstants.ORGANIZATION_NAME_PREFIX + orgName + SecurityManagementImplConstants.COMMA_AND_SPACE
						+ SecurityManagementImplConstants.LOCALITY_NAME_PREFIX + localityName + SecurityManagementImplConstants.COMMA_AND_SPACE
						+ SecurityManagementImplConstants.STATE_NAME_PREFIX + stateName + SecurityManagementImplConstants.COMMA_AND_SPACE
						+ SecurityManagementImplConstants.COUNTRY_PREFIX + country + SecurityManagementImplConstants.QUOTE;
		return Dname;
	}
	/**
	 * getCertificateInfo - Get the certificates associated with the alias key/keystore
	 * @param alias - alias key
	 * @param storePass
	 * @return
	 */ 
	public String getCertificateInfo( IProgressMonitor monitor ) throws MtjException{
		
		String certInfo = ""; //$NON-NLS-1$
		
		if ((aliaskey != null) && (aliaskey.length() > 0)) {
			
			try {
				String[] cmdArgs = generateDisplayCertifcates( );		
				Process p = runSecurityCmd(cmdArgs);
				BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getInputStream()));
				monitor.worked(20);
								
				String cmdOutput;
			
				while ((cmdOutput=cmdOutputStream.readLine()) != null) {
						
					if (cmdOutput.toLowerCase().indexOf("error")>=0) {  //$NON-NLS-1$
					 //MessageDialog.openError(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), "Error", cmdOutput);  //$NON-NLS-1$
					 //return ""; //$NON-NLS-1$
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) + " " + cmdOutput); //$NON-NLS-1$
					} else if (cmdOutput.length() >= 0) {
						certInfo = certInfo + cmdOutput;					
					  }
				}
						
			} catch (IOException ee) {
				certInfo = ""; //$NON-NLS-1$
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY), ee);
			}
			 catch(Exception e){
			 	certInfo = ""; //$NON-NLS-1$
			 	throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY), e);
			 }
		}//if aliaskey
		
		return certInfo;
	}
	
	/**
	 * generateDisplayCertifcates - Command to display all certificates for a given key (alias)
	 * @param alias - alias key for which the certificate info is being requested.
	 * @param storePasswd - password used to open keystore
	 * @return
	 * @throws MtjException 
	 */
	private String[] generateDisplayCertifcates( ) throws MtjException {
		
		String[] listCertificateCmdArgs = 
					{	getSecurityManagementTool(),
						getConsoleEncoding(),
						SecurityManagementImplConstants.LIST,
						SecurityManagementImplConstants.ALIAS,aliaskey,									
						SecurityManagementImplConstants.STORETYPE,storeType,					
						SecurityManagementImplConstants.KEYSTORE,keyStoreNameLoc,
						SecurityManagementImplConstants.STOREPASS,passwrd};
			
		return listCertificateCmdArgs;
	}

	public String getToolLocation(IProgressMonitor monitor) throws MtjException {
		return securityProviderPrefStore.getString(SecurityManagementImplConstants.SECURITY_TOOL_LOCATION);
	}

	public void storeToolLocation(String loc, IProgressMonitor monitor) throws MtjException {
		securityProviderPrefStore.setValue(SecurityManagementImplConstants.SECURITY_TOOL_LOCATION, loc);
		
	}

	public void resetValues() {
		// TODO Auto-generated method stub
		
	}
}