/*******************************************************************************
* Copyright (c) 2006 IBM.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/

package org.eclipse.mtj.extension.bp.eclipse.midp;

import org.apache.tools.ant.BuildException;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.common.util.EList;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.mtj.api.enumerations.DeploymentType;
import org.eclipse.mtj.api.extension.SigningProvider;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.api.project.Project;
import org.eclipse.mtj.api.project.SigningDetail;
import org.eclipse.mtj.api.project.TargetDevice;
import org.eclipse.mtj.api.signings.SigningCertificate;
import org.eclipse.mtj.api.signings.impl.SigningsFactoryImpl;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.bp.eclipse.BuildProviderUtil;
import org.eclipse.mtj.extension.bp.eclipse.Messages;
import org.eclipse.mtj.extension.devide.utils.InputValidationDialog;
import org.eclipse.mtj.extension.devide.utils.SmgmtCache;
import org.eclipse.swt.widgets.Shell;

public class MidpSigning {

	static final SmgmtCache keystoreCache = SmgmtCache.getSmgmtCache();
	boolean canSign = false;
	// We need to know if we are running in the ant process without a UI.
	boolean hasGuiUI = true;
	IMtjProject mtjProject;
	IProgressMonitor monitor;
	private IFolder deploymentFolder = null;

	
	/**
	 * 
	 */
	public MidpSigning(IMtjProject mtjProject, IProgressMonitor monitor) {
		super();
		this.mtjProject = mtjProject;
		this.monitor = monitor;
	}
	/**
	 * 
	 */
	public MidpSigning(IMtjProject mtjProject, IProgressMonitor monitor, boolean callerHasUI) {
		super();
		hasGuiUI = callerHasUI;
		this.mtjProject = mtjProject;
		this.monitor = monitor;

	}

	/**
	 * If returns null, should not try to sign
	 * @param mtjProject
	 * @param monitor
	 * @return
	 * @throws CoreException
	 * @throws MtjException
	 */
	public SigningDetail getSigningInfo(IMtjProject mtjProject, IProgressMonitor monitor){
		SigningDetail sd = null;
		DeploymentType dtype = null;
		TargetDevice targetDevice = null;
		EList signingDetails = null;
		SigningProvider signprov = null;

		try {
			dtype = mtjProject.getDeploymentType();
		} catch (CoreException e2) {
			System.err.println(Messages.BuildProviderImpl_16); 
			System.err.println(Messages.BuildProviderImpl_17 + Messages.BuildProviderImpl_18); 
			return null;
		}
		
		try {
			signprov = BuildProviderUtil.getSigningProvider(mtjProject.getProject(), dtype);
		} catch (MtjException e1) {}
		
		if ( signprov == null ) {
			System.err.println(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_PROVIDER_NOT_CONFIGURED));
			return null;
		}
	
		Project data = null;
		try {
			data = mtjProject.getProjectData();
		} catch (MtjException e) {
			System.err.println(Messages.BuildProviderImpl_19); 
			System.err.println(Messages.BuildProviderImpl_20 + Messages.BuildProviderImpl_21); 
		}

		//Look for signing information...
		if (data != null){						
			//get signing information from target device
			targetDevice = data.getDefaultTargetDevice();
			if ( targetDevice != null){
				signingDetails = targetDevice.getSigningDetails();
				//no signing information, skip signing process
				if (signingDetails != null){
					if (signingDetails.size() > 0){
						//	use the first signing certificate
						sd = ((SigningDetail)signingDetails.get(0));
					}
				} else {
					System.err.println(Messages.BuildProviderImpl_22); 
					System.err.println(Messages.BuildProviderImpl_23 + Messages.BuildProviderImpl_2); 
					//throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.getString("BuildProviderImpl.2")); //$NON-NLS-1$
				}
			} else {
				System.err.println(Messages.BuildProviderImpl_25); 
				System.err.println(Messages.BuildProviderImpl_26 + Messages.BuildProviderImpl_3); 
				//throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.getString("BuildProviderImpl.3")); //$NON-NLS-1$
			}
		} else {
			System.err.println(Messages.BuildProviderImpl_28); 
			System.err.println(Messages.BuildProviderImpl_29 + Messages.BuildProviderImpl_4); 
			//throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.getString("BuildProviderImpl.4")); //$NON-NLS-1$
		}
		
		return sd;
	}

	private boolean createSigningCertificate (SigningDetail sd, String password)
	{
		//create keystore, based on signing detail, and user input
		SigningCertificate sc = (new SigningsFactoryImpl()).createSigningCertificate();

		sc.setAliasName(sd.getAlias());
		sc.setKeystore(sd.getKeystore());
		sc.setCertificatePassword(password); 
		sc.setKeystorePassword(password);	
		// if caching succeeds, go ahead w/ signing step
		return keystoreCache.cacheSigningCertificate(sc);
	}
	
	public boolean preemtSignNoUI (String password){
		
		boolean shouldSign = false;

		//get singing detail in targetDevice for this project
		SigningDetail sd = getSigningInfo(mtjProject, monitor);
		if (sd == null) // no signing information so don't bother signing
			return false;
		
		shouldSign = createSigningCertificate (sd, password);
		
		canSign = shouldSign;
		return shouldSign;
		
	}
	
	public boolean preemtSign(){
		
		boolean shouldSign = false;

		//get singing detail in targetDevice for this project
		SigningDetail sd = getSigningInfo(mtjProject, monitor);
				
		//if no signing detail, don't need to try to sign
		if (sd != null){
			//look for corresponding signing certificate in cache based on this signing detail
			if (!keystoreCache.lookupSigningCertificate(sd.getKeystore())){
				//Not cached, must prompt user for it
				Shell shell = MtjCorePlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell();
						
				InputValidationDialog promptPassword = new InputValidationDialog(shell,Messages.BuildProviderImpl_1,Messages.BuildProviderImpl_0 + sd.getKeystore() + Messages.BuildProviderImpl_10 + sd.getAlias(), null, null, new oldPasswordLengthValidator(), null, null); 
				promptPassword.open();
				if (promptPassword.getReturnCode() == org.eclipse.jface.window.Window.OK){
					shouldSign = createSigningCertificate (sd, promptPassword.getPass());
				}
				//else {
				//TODO hmmm... stop build process? throw exception? skip signing step???
				//}
			} else {
				//signing certificate is cached, go ahead w/ signing step
				shouldSign = true;
			}
		}
		
		canSign = shouldSign;
		return shouldSign;
		
	}
	
	public boolean canSign ()
	{
		return canSign;
	}
	
	/**
	 * Only call this method if the password is known
	 * @param mtjProject
	 * @param password
	 * @param monitor
	 */
	public void sign(String password) {
		hasGuiUI = false;
		boolean canSign = preemtSignNoUI(password);
		if (canSign)
			sign ();

	}	
	public void sign() {
		DeploymentType dtype = null;
		IFolder dfolder = null;
		TargetDevice targetDevice = null;
		EList signingDetails = null;
		Project data = null;
		boolean signed = false;
		
		try {
			dtype = mtjProject.getDeploymentType();
		} catch (CoreException e) {
			displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.MidpSigning_0), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15, IStatus.ERROR); 
		}
				
		try {
			dfolder = getDeploymentFolder();
		} catch (CoreException e) {
			displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.MidpSigning_3), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15, IStatus.ERROR); 
		}
		
		try {data = mtjProject.getProjectData();}
		catch (MtjException e) {}

		//Look for signing information...
		if (data != null){						
			//get signing information from target device
			targetDevice = data.getDefaultTargetDevice();
			if ( targetDevice != null){
				signingDetails = targetDevice.getSigningDetails();
				//no signing information, skip signing process
				if (signingDetails != null){
						if (signingDetails.size() <= 0){
							return;
						}
					}else{
						displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) +  Messages.BuildProviderImpl_2), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15, IStatus.ERROR); 
						return;
					}
				} else{
					displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) +  Messages.BuildProviderImpl_3), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15, IStatus.ERROR); 
					return;
				}
		} else {
			displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) +  Messages.BuildProviderImpl_4), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15, IStatus.ERROR); 
			return;
		}
		
		
		SigningProvider signprov = null;
		try {
			signprov = BuildProviderUtil.getSigningProvider(mtjProject.getProject(), dtype);
		} catch (MtjException e) {}
		
		if ( signprov == null ) {
			displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_PROVIDER_NOT_CONFIGURED) + Messages.MidpSigning_ConfigureMtjSigningProviders), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15, IStatus.ERROR); 
			return;
		}
		
		//if already signed, need to un-sign
		try {
			if (signprov.isSigned(mtjProject, dfolder, monitor)){
				if(!signprov.unsign(mtjProject,dfolder, monitor)){
					 displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.BuildProviderImpl_6), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15, IStatus.ERROR); 
					 return;
				 } 
			}
		} catch (MtjException e) {
			displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.MidpSigning_6), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15, IStatus.ERROR);  
			 return;
		} 
		
		//	use the first signing detail		
		SigningDetail sd = ((SigningDetail)signingDetails.get(0));
	
		SigningCertificate sc = SigningsFactoryImpl.eINSTANCE.createSigningCertificate();
		
		//check for signing certificate in cache, based on the signing detail found in targetDevice
		if(keystoreCache.lookupSigningCertificate(sd.getKeystore())){
			//it's cached, get it
			sc = keystoreCache.getSigningCertificate(sd.getKeystore());
			sc.setAliasName(sd.getAlias());
		} else {
			//TODO do we want to prompt again??? don't think so,
			return;
		}
			
		try {
			signed = signprov.sign(mtjProject,getDeploymentFolder(),sc,monitor);
		} catch (MtjException e) {
			displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.BuildProviderImpl_9 + e.getMessage(), e), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15, IStatus.ERROR); 
		} catch (CoreException e) {
			displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.BuildProviderImpl_9 + e.getMessage(), e), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15, IStatus.ERROR); 
		} finally {
			//deployed JAD file is cleared by packaging step, therefore signing information is lost,
			//in order to keep JAD file and project in sync, must remove signing info from project
			if (!signed){
				displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.BuildProviderImpl_9), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15 + Messages.MidpSigning_17, IStatus.ERROR); 
				signingDetails.clear();
				try {
					//signprov.unsign(mtjProject,dfolder, monitor);
					mtjProject.setProjectData(data);
					keystoreCache.unCacheSigningCertificate(sc);
				} catch (MtjException e) {}
			}			
		}
	}

	public boolean unsign() {
		
		DeploymentType dtype = null;
		IFolder dfolder = null; 
		SigningProvider signprov = null;
		EList signingDetails = null;
		Project data = null;
		TargetDevice targetDevice = null;

		try {
			dfolder = mtjProject.getDeploymentFolder(null);

			dtype = mtjProject.getDeploymentType();
			
			try {
				signprov = BuildProviderUtil.getSigningProvider(mtjProject.getProject(), dtype);
			} catch (MtjException e){
				displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_PROVIDER_NOT_CONFIGURED) + Messages.MidpSigning_7),  Messages.BuildProviderImpl_30,  Messages.BuildProviderImpl_30, IStatus.ERROR); 
				return false;
			}
			
			if ( signprov == null ) {
				displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_PROVIDER_NOT_CONFIGURED) + Messages.MidpSigning_10),  Messages.BuildProviderImpl_30,  Messages.BuildProviderImpl_30, IStatus.ERROR); 
				return false;
			}
			if (signprov.isSigned(mtjProject, dfolder, monitor)){
				if(!signprov.unsign(mtjProject,dfolder, monitor)){
					 displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING)),  Messages.BuildProviderImpl_30, Messages.BuildProviderImpl_15, IStatus.ERROR); 
					 return false;
				 }
			}
			else{
				displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING)), Messages.BuildProviderImpl_30, Messages.MidpSigning_14, IStatus.INFO);  
				 return false;
			}
			
			data = mtjProject.getProjectData();

			//Look for signing information...
			if (data != null){						
				//get signing information from target device
				targetDevice = data.getDefaultTargetDevice();
				if ( targetDevice != null){
					signingDetails = targetDevice.getSigningDetails();
					//no signing information, skip signing process
					if (signingDetails != null){
							if (signingDetails.size() > 0){
								EList i =  targetDevice.getSigningDetails();
								if (i != null){
									i.clear();
								}
								mtjProject.setProjectData(data);
							}else {
								displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING)), Messages.BuildProviderImpl_30, Messages.MidpSigning_16, IStatus.INFO); 
							}
							
						}else{
							displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) +  Messages.BuildProviderImpl_2), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15, IStatus.ERROR); 
							return false;
						}
					} else{
						displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) +  Messages.BuildProviderImpl_3), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15, IStatus.ERROR); 
						return false;
					}
			} else {
				displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) +  Messages.BuildProviderImpl_4), Messages.BuildProviderImpl_14, Messages.BuildProviderImpl_15, IStatus.ERROR); 
				return false;
			}

		} catch (CoreException e) {
			displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING)), Messages.BuildProviderImpl_30,  Messages.BuildProviderImpl_30, IStatus.ERROR);  
			 return false;
		} catch (MtjException e) {
			displaySigningStepMessage(new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING)), Messages.BuildProviderImpl_30,  Messages.BuildProviderImpl_30, IStatus.ERROR);  
			 return false;
		}
	
	
		return true;
	}

	public class oldPasswordLengthValidator implements IInputValidator {
		public String isValid(String arg0) {
			if (checkBlankChars(arg0)){
				return Messages.BuildProviderImpl_11; 
			}
			if (arg0.length() < 1){
				return Messages.BuildProviderImpl_12; 
			}
			return null;
		}
		public boolean checkBlankChars(String text) {
			
			if (text.indexOf(" ") != -1){ //$NON-NLS-1$
				return true;
			}
			return false;
		}
	}
	
	public  void displaySigningStepMessage(MtjException e, String title, String body, int error)
	{
		if (hasGuiUI)
			BuildProviderUtil.displayBuildMessage(e, title, body, error);
		else
			throw new BuildException (e);
	}
	/**
	 * @return Returns the deploymentFolder.
	 * @throws CoreException 
	 */
	public IFolder getDeploymentFolder() throws CoreException {
		if (deploymentFolder == null)
			deploymentFolder = mtjProject.getDeploymentFolder(monitor);
		return deploymentFolder;
	}
	/**
	 * @param deploymentFolder The deploymentFolder to set.
	 */
	public void setDeploymentFolder(IFolder deploymentFolder) {
		this.deploymentFolder = deploymentFolder;
	}

}
