/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.extension.bm;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.IField;
import org.eclipse.jdt.core.IJavaModelMarker;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.ISourceRange;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.mtj.api.enumerations.BuildType;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.extension.BuildManagement;
import org.eclipse.mtj.api.extension.BuildProvider;
import org.eclipse.mtj.api.extension.MtjExtension;
import org.eclipse.mtj.api.extension.impl.MtjExtensionImpl;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.api.model.preverifier.IClassErrorInformation;
import org.eclipse.mtj.api.model.preverifier.IFieldErrorInformation;
import org.eclipse.mtj.api.model.preverifier.IMethodErrorInformation;
import org.eclipse.mtj.api.model.preverifier.PreverificationError;
import org.eclipse.mtj.api.model.preverifier.PreverificationErrorLocationType;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.core.MtjServices;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.exception.PreverificationException;

public class BuildManagementImpl extends MtjExtensionImpl implements
		BuildManagement {

	public BuildManagementImpl() {
		super();
		
		setId("org.eclipse.mtj.extension.bm"); //$NON-NLS-1$
		setDescription(Messages.BuildManagementImpl_PluginDescription);
		setVendor(Messages.BuildManagementImpl_PluginVendor);
		setVersion(Messages.BuildManagementImpl_PluginVersion);
		setType(ExtensionType.BUILD_MANAGEMENT_LITERAL);
		
	}
	
	public void build(IMtjProject project, String projectNatureId, BuildType buildType,
			RuntimePlatformDefinition targetPlatform,
			IProgressMonitor monitor) throws MtjException {
		
		BuildProvider provider = getBuildProvider(buildType);
		
		if (provider == null) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.BUILD_PROVIDER_NOT_FOUND_ERROR));
		}
		
		try {
			provider.build(project, projectNatureId, buildType, targetPlatform, monitor);
		}
		catch (PreverificationException ex) {
			PreverificationError[] errors = ex.getPreverificationError();
			try {
				handlePreverificationErrors(project.getJavaProject(), errors);
			} catch (Exception e) {
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_WHILE_ADDING_PREVERIFICATION_ERRORS),e);
			}
			
			throw ex;
		}
		catch (MtjException ex) {
			throw ex;
		}
	}
	
	private void handlePreverificationErrors(IJavaProject javaProject, PreverificationError[] errors) 
			throws JavaModelException, CoreException {
		for (int i = 0; i < errors.length; i++) {
			PreverificationError error = errors[i];
			createErrorMarkerFor(javaProject, error);
		}

	}
	
	private BuildProvider getBuildProvider(BuildType buildType) throws MtjException {
		
		MtjExtension[] extensions = 
			MtjServices.getInstance().getImplementations(ExtensionType.BUILD_PROVIDER_LITERAL, null, null,true);

		for (int i = 0; i < extensions.length; i++) {
			BuildProvider prov = (BuildProvider)extensions[i];
			BuildType[] types = prov.getSupportedTypes();
			for (int j = 0; j < types.length; j++) {
				if ( buildType.getValue() == types[j].getValue()) {
					return prov;
				}
			}
		}
		return null;
	}

	/**
	 * Create an error marker for the specific type with
	 * the specified error message.
	 * 
	 * @param error
	 */
	private void createErrorMarkerFor(IJavaProject javaProject, PreverificationError error) 
		throws JavaModelException, CoreException
	{
		IMarker marker = null;
		
		// Calculate the resource
		IClassErrorInformation classInfo = error.getLocation().getClassInformation();
		String typeName = (classInfo == null) ? "" : classInfo.getName().replace('/', '.'); //$NON-NLS-1$
		String message = error.getErrorText();
	
		IType type = javaProject.findType(typeName);
		if (type != null) {
			IResource resource = type.getResource();
			
			// Sometimes the resource doesn't come back... This is supposed
			// to be only when the resource is in an external archive.
			if (resource != null) {
				// Create the marker and set the attributes
				marker = resource.createMarker(IJavaModelMarker.JAVA_MODEL_PROBLEM_MARKER);
				marker.setAttribute(IMarker.MESSAGE, message);
				marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
				
				int lineNumber = error.getLocation().getLineNumber();
				if (lineNumber != -1) {
					marker.setAttribute(IMarker.LINE_NUMBER, lineNumber);
				}
				
				setMarkerRangeAttributes(marker, error, type);
			}
		}
		
		// Fallback position if nothing specific is possible
		if (marker == null) {
			createProjectLevelPreverifyMarker(javaProject, typeName, message);
		}
	}

	/**
	 * Set the attributes for character range on the marker.
	 * 
	 * @param marker
	 * @param error
	 * @param type
	 * @throws CoreException
	 */
	private void setMarkerRangeAttributes(IMarker marker, PreverificationError error, IType type) 
		throws CoreException 
	{
		int start = 1;
		int end = 1;
		
		IMethodErrorInformation methodInfo = error.getLocation().getMethodInformation();
		IFieldErrorInformation fieldInfo = error.getLocation().getFieldInformation();
		
		switch (error.getLocation().getLocationType().getTypeCode()) {
			case PreverificationErrorLocationType.CLASS_DEFINITION_CODE:
				{
					ISourceRange sourceRange = type.getNameRange();
					start = sourceRange.getOffset();
					end = start + sourceRange.getLength();
				}
				break;

			case PreverificationErrorLocationType.CLASS_FIELD_CODE:
				{
					IField field = type.getField(fieldInfo.getName());
					if ((field != null) && (field.exists())) {
						ISourceRange sourceRange = field.getNameRange();
						start = sourceRange.getOffset();
						end = start + sourceRange.getLength();
					}
				}
				break;

			case PreverificationErrorLocationType.METHOD_SIGNATURE_CODE:
			case PreverificationErrorLocationType.METHOD_FIELD_CODE:
			case PreverificationErrorLocationType.METHOD_INSTRUCTION_CODE:
				{
					IMethod method = methodInfo.getTypeMethod(type);
					if ((method != null) && (method.exists())) {
						ISourceRange sourceRange = method.getNameRange();
						start = sourceRange.getOffset();
						end = start + sourceRange.getLength();
					}
				}
				break;
		}
		
		// Final fallback...
		marker.setAttribute(IMarker.CHAR_START, start);
		marker.setAttribute(IMarker.CHAR_END, end);
	}

	/**
	 * Create an error marker on the project about a type verification.  This will
	 * happen when an error occurs trying to preverify and yet the type resource
	 * cannot be located for some reason.
	 * 
	 * @param typeName
	 * @param message
	 * @throws CoreException
	 */
	private void createProjectLevelPreverifyMarker(IJavaProject javaProject, String typeName, String message) 
		throws CoreException 
	{
		StringBuffer sb = new StringBuffer(Messages.BuildManagementImpl_Type);
		sb.append(typeName).append(" ").append(message); //$NON-NLS-1$
		
		IProject project = javaProject.getProject();
		IMarker marker = project.createMarker(IJavaModelMarker.JAVA_MODEL_PROBLEM_MARKER);
		marker.setAttribute(IMarker.MESSAGE, sb.toString());
		marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
	}

}
