/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 

package org.eclipse.mtj.core.launching;

import java.io.File;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.text.DateFormat;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.core.IStatusHandler;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.jdi.Bootstrap;
import org.eclipse.jdt.debug.core.JDIDebugModel;
import org.eclipse.jdt.launching.AbstractVMRunner;
import org.eclipse.jdt.launching.ExecutionArguments;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.jdt.launching.SocketUtil;
import org.eclipse.jdt.launching.VMRunnerConfiguration;
import org.eclipse.mtj.api.deployment.Deployment;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.model.IExecutablePlatform;
import org.eclipse.mtj.core.IEclipseMtjCoreConstants;
import org.eclipse.mtj.core.Messages;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.mtj.exception.MtjException;

import com.sun.jdi.VirtualMachine;
import com.sun.jdi.connect.AttachingConnector;
import com.sun.jdi.connect.Connector;
import com.sun.jdi.connect.IllegalConnectorArgumentsException;
import com.sun.jdi.connect.ListeningConnector;

public class EmulatorRunner extends AbstractVMRunner {
	private IExecutablePlatform executablePlatform;
	private DevicePlatform platform;
	private boolean debugMode;

	/**
	 * Used to attach to a VM in a seperate thread, to allow for cancellation
	 * and detect that the associated System process died before the connect
	 * occurred.
	 */
	class ConnectRunnable implements Runnable {
		private VirtualMachine fVirtualMachine = null;
		private ListeningConnector fConnector = null;
		private Map fConnectionMap = null;
		private Exception fException = null;

		/**
		 * Constructs a runnable to connect to a VM via the given connector
		 * with the given connection arguments.
		 * 
		 * @param connector
		 * @param map
		 */
		public ConnectRunnable(ListeningConnector connector, Map map) {
			fConnector = connector;
			fConnectionMap = map;
		}
		
		/**
		 * Thread entrypoint.
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			try {
				fVirtualMachine = fConnector.accept(fConnectionMap);
			} catch (IOException e) {
				fException = e;
			} catch (IllegalConnectorArgumentsException e) {
				fException = e;
			}
		}
		
		/**
		 * Returns the VM that was attached to, or <code>null</code> if none.
		 * 
		 * @return the VM that was attached to, or <code>null</code> if none
		 */
		public VirtualMachine getVirtualMachine() {
			return fVirtualMachine;
		}
		
		/**
		 * Returns any exception that occurred while attaching, or <code>null</code>.
		 * 
		 * @return IOException or IllegalConnectorArgumentsException
		 */
		public Exception getException() {
			return fException;
		}
	}

	/**
	 * Render the debug target string.
	 * 
	 * @param classToRun
	 * @param host
	 * @return
	 */
	private static String renderDebugTarget(String classToRun, int host) {
		String ret = 		MessageFormat.format(
				Messages.debugvmrunner_debug_target_string,
				new String[] { classToRun, String.valueOf(host) });
			return ret;
	}

	/**
	 * Render the process label string.
	 * 
	 * @param commandLine
	 * @return
	 */
	public static String renderProcessLabel(String[] commandLine) {
		String timestamp = 
			DateFormat.getInstance().format(
				new Date(System.currentTimeMillis()));
		String ret = MessageFormat.format (
				Messages.debugvmrunner_process_label_string,
			new String[] { commandLine[0], timestamp });
		return ret;
	}
	
	/**
	 * Render the command line string.
	 * 
	 * @param commandLine
	 * @return
	 */
	private static String renderCommandLine(String[] commandLine) {
		StringBuffer buf = new StringBuffer();
		
		if (commandLine.length > 1) {
			for (int i = 0; i < commandLine.length; i++) {
				if (i > 0) buf.append(' ');
				buf.append(commandLine[i]);
			}
		}
			
		return buf.toString();
	}
	
	/**
	 * Construct an VM runner instance for an executable emulator.
	 * 
	 * @param emulator
	 */
	public EmulatorRunner(DevicePlatform platform, IExecutablePlatform executablePlatform, String mode) throws MtjException
	{
		this.platform = platform;
		this.executablePlatform = executablePlatform;
		debugMode = ILaunchManager.DEBUG_MODE.equals(mode);
	}

	/**
	 * @see org.eclipse.jdt.launching.IVMRunner#run(org.eclipse.jdt.launching.VMRunnerConfiguration, org.eclipse.debug.core.ILaunch, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void run(
		VMRunnerConfiguration vmRunnerConfig,
		ILaunchConfiguration launchConfig,
		Deployment deployment,
		ILaunch launch,
		String projectNatureId,
		Device device, 
		IProgressMonitor monitor)
			throws CoreException, MtjException
	{
		if (debugMode) {
			runInDebug(vmRunnerConfig, launchConfig, deployment, launch, projectNatureId, device, monitor);
		} else {
			runWithoutDebug(vmRunnerConfig, launchConfig, deployment, launch, projectNatureId, device, monitor);
		}
	}
	
	/**
	 * Run the emulator with debugging.
	 * 
	 * @param vmRunnerConfig
	 * @param launchConfig
	 * @param launch
	 * @param monitor
	 * @throws CoreException
	 */
	public void runInDebug (
		VMRunnerConfiguration vmRunnerConfig,
		ILaunchConfiguration launchConfig,
		Deployment deployment,
		ILaunch launch,
		String projectNatureId,
		Device device, 
		IProgressMonitor monitor)
			throws CoreException , MtjException
	{
		// TODO fix platform creation
		platform.setDebugEnabled(true);
		
		if (monitor == null) {
			monitor = new NullProgressMonitor();
		}
		
		IProgressMonitor subMonitor = new SubProgressMonitor(monitor, 1);
		subMonitor.beginTask(Messages.debugvmrunner_launching_vm, 4); 
		subMonitor.subTask(Messages.debugvmrunner_finding_free_socket); 

		int port = SocketUtil.findFreePort();
		if (port == -1) {
			abort(
					Messages.debugvmrunner_no_free_socket, 
				null, IJavaLaunchConfigurationConstants.ERR_NO_SOCKET_AVAILABLE);
		}
		
		subMonitor.worked(1);
		
		// check for cancellation
		if (monitor.isCanceled()) {
			return;
		}		
		
		subMonitor.subTask(
				Messages.debugvmrunner_constructing_cmd_line); 

		String[] cmdLine = getCommandLine(launchConfig, deployment, port, projectNatureId, device);
		
		// check for cancellation
		if (monitor.isCanceled()) {
			return;
		}		
		
		subMonitor.worked(1);
		subMonitor.subTask(Messages.debugvmrunner_starting_VM); 

		Connector connector = getConnector();
		if (connector == null) {
			abort(
					Messages.debugvmrunner_no_connector, 
				null, 
				IJavaLaunchConfigurationConstants.ERR_CONNECTOR_NOT_AVAILABLE);
		}
		
		Map map = connector.defaultArguments();
		specifyArguments(map, port);

		Process p = null;
		try {
			try {
				// check for cancellation
				if (monitor.isCanceled()) {
					return;
				}				
				
				if ( !platform.isDebugEnabled()) {
					((ListeningConnector) connector).startListening(map);
				}
				
				File workingDir = getWorkingDir(vmRunnerConfig);
				p = exec(cmdLine, workingDir);				
				if (p == null) {
					return;
				}
				
				// check for cancellation
				if (monitor.isCanceled()) {
					p.destroy();
					return;
				}				
				
				IProcess process = DebugPlugin.newProcess(
					launch, p, 
					renderProcessLabel(cmdLine), 
					getDefaultProcessMap());
				process.setAttribute(
					IProcess.ATTR_CMDLINE, 
					renderCommandLine(cmdLine));
				
				subMonitor.worked(1);
				subMonitor.subTask(
						Messages.debugvmrunner_establishing_debug_conn); 

				// If the emulator debugger acts as a remote server,
				// delay a bit to give the emulator a chance to start
				// up.
				if (platform.isDebugEnabled()) {
					Preferences preferences = 
						MtjCorePlugin.getDefault().getPluginPreferences(); 
					int milliDelay = preferences.getInt(IEclipseMtjCoreConstants.PREF_RMTDBG_DELAY);
					try { 
						// TODO fix preferences
						Thread.sleep(milliDelay == 0 ? 5000: milliDelay); 
					} catch (InterruptedException e) {}
				}
				
				VirtualMachine vm = 
					createVirtualMachine(connector, map, p, process, monitor);

				JDIDebugModel.newDebugTarget(
					launch, vm, 
					renderDebugTarget(vmRunnerConfig.getClassToLaunch(), port), 
					process, true, false);
				subMonitor.worked(1);
				subMonitor.done();
				return;
				
			} finally {
				if (!platform.isDebugEnabled()) {
					((ListeningConnector) connector).stopListening(map);
				}
			}
		} catch (IOException e) {
			abort(
					Messages.debugvmrunner_couldnt_connect_to_vm, 
				e, IJavaLaunchConfigurationConstants.ERR_CONNECTION_FAILED);
		} catch (IllegalConnectorArgumentsException e) {
			abort(
					Messages.debugvmrunner_couldnt_connect_to_vm, 
				e, IJavaLaunchConfigurationConstants.ERR_CONNECTION_FAILED);
		}

		if (p != null) {
			p.destroy();
		}
	}

	/**
	 * Create a new VirtualMachine instances for the specified
	 * Connector and associated information.
	 * 
	 * @param connector
	 * @param map
	 * @param p
	 * @param process
	 * @param monitor
	 * @return
	 * @throws IOException
	 * @throws IllegalConnectorArgumentsException
	 * @throws CoreException
	 */
	private VirtualMachine createVirtualMachine(
			Connector connector, Map map, 
			Process p, IProcess process, 
			IProgressMonitor monitor) 
				throws IOException, 
				IllegalConnectorArgumentsException, 
				CoreException 
	{
		VirtualMachine vm = (platform.isDebugEnabled()) ?
			((AttachingConnector) connector).attach(map) :
			waitForDebuggerConnection(
					(ListeningConnector) connector, p, process, map, monitor);
		return vm;
	}

	/**
	 * Get the appropriate Connector dependent on what
	 * the IEmulator instance requires.
	 * 
	 * @return
	 */
	private Connector getConnector() {
		Connector connector = (platform.isDebugEnabled()) ?
			(Connector) getAttachingConnector() :
			(Connector) getListeningConnector();
		return connector;
	}

	/**
	 * Wait for the debugger to connect to our connector
	 * and return the new VirtualMachine.
	 * 
	 * @param connector
	 * @param p
	 * @param process
	 * @param map
	 * @param monitor
	 * @return
	 * @throws CoreException
	 * @throws IOException
	 * @throws IllegalConnectorArgumentsException
	 */
	private VirtualMachine waitForDebuggerConnection(
			ListeningConnector connector,
			Process p,
			IProcess process,
			Map map,
			IProgressMonitor monitor) 
				throws CoreException, IOException, 
				IllegalConnectorArgumentsException
	{
		VirtualMachine vm = null;
		
		boolean retry = false;
		do  {
			try {
				ConnectRunnable runnable = 
					new ConnectRunnable(connector, map);
				Thread connectThread = 
					new Thread(runnable, "Listening Connector"); //$NON-NLS-1$
				connectThread.start();
				
				while (connectThread.isAlive()) {
					if (monitor.isCanceled()) {
						connector.stopListening(map);
						p.destroy();
						
						break;
					}
					try {
						p.exitValue();
						// process has terminated - stop waiting for a connection
						try {
							connector.stopListening(map); 
						} catch (IOException e) {
							// expected
						}
						checkErrorMessage(process);
					} catch (IllegalThreadStateException e) {
						// expected while process is alive
					}
					try {
						Thread.sleep(100);
					} catch (InterruptedException e) {
					}
				}

				Exception ex = runnable.getException();
				
				if (ex instanceof IllegalConnectorArgumentsException) {
					throw (IllegalConnectorArgumentsException)ex;
				}
				if (ex instanceof InterruptedIOException) {
					throw (InterruptedIOException)ex;
				}
				if (ex instanceof IOException) {
					throw (IOException)ex;
				}
				
				vm = runnable.getVirtualMachine();
				
				break;
			} catch (InterruptedIOException e) {
				
				checkErrorMessage(process);
				
				// timeout, consult status handler if there is one
				IStatus status = new Status(
					IStatus.ERROR, 
					IEclipseMtjCoreConstants.PLUGIN_ID, 
					MtjCoreErrors.VM_CONNECT_TIMEOUT_ERROR, 
					MtjCoreErrors.getErrorMessage(MtjCoreErrors.VM_CONNECT_TIMEOUT_ERROR), e);
				IStatusHandler handler = DebugPlugin.getDefault().getStatusHandler(status);
				
				retry = false;
				if (handler == null) {
					// if there is no handler, throw the exception
					throw new CoreException(status);
				} else {
					Object result = handler.handleStatus(status, this);
					if (result instanceof Boolean) {
						retry = ((Boolean)result).booleanValue();
					}
				} 
			}
		} while (retry);
		
		return vm;
	}

	/**
	 * Run the emulator without debugging.
	 * 
	 * @param vmRunnerConfig
	 * @param launchConfig
	 * @param launch
	 * @param monitor
	 * @throws CoreException
	 */
	public void runWithoutDebug(
		VMRunnerConfiguration vmRunnerConfig,
		ILaunchConfiguration launchConfig,
		Deployment deployment,
		ILaunch launch,
		String projectNatureId,
		Device device, 
		IProgressMonitor monitor)
			throws CoreException, MtjException
	{

		if (monitor == null) {
			monitor = new NullProgressMonitor();
		}
		
		IProgressMonitor subMonitor = new SubProgressMonitor(monitor, 1);
		subMonitor.beginTask(Messages.debugvmrunner_launching_vm, 3); 
		
		// check for cancellation
		if (monitor.isCanceled()) {
			return;
		}		
		
		subMonitor.subTask(Messages.debugvmrunner_constructing_cmd_line); 

		String[] cmdLine = getCommandLine(launchConfig, deployment, -1, projectNatureId, device);
			
		// check for cancellation
		if (monitor.isCanceled()) {
			return;
		}		
		
		subMonitor.worked(1);
		subMonitor.subTask(Messages.debugvmrunner_starting_VM); 
		
		// check for cancellation
		if (monitor.isCanceled()) {
			return;
		}				
		
		File workingDir = getWorkingDir(vmRunnerConfig);
		Process p = exec(cmdLine, workingDir);				
		if (p == null) {
			return;
		}
		
		// check for cancellation
		if (monitor.isCanceled()) {
			p.destroy();
			return;
		}				
		
		IProcess process = DebugPlugin.newProcess(
			launch, p, 
			renderProcessLabel(cmdLine), 
			getDefaultProcessMap());
		process.setAttribute(
			IProcess.ATTR_CMDLINE, 
			renderCommandLine(cmdLine));
		
		subMonitor.worked(1);
	}

	/**
	 * @see org.eclipse.jdt.launching.AbstractVMRunner#getPluginIdentifier()
	 */
	protected String getPluginIdentifier() {
		return IEclipseMtjCoreConstants.PLUGIN_ID;
	}
	
	/**
	 * Add the specified arguments array to the list of arguments.
	 * 
	 * @param args
	 * @param allArgs
	 */
	protected void addArguments(String[] args, List allArgs) {
		if (args != null) {
			for (int i= 0; i < args.length; i++) {
				allArgs.add(args[i]);
			}
		}
	}
	
	/**
	 * Check for an error message and throw an exception as necessary.
	 * 
	 * @param process
	 * @throws CoreException
	 */	
	protected void checkErrorMessage(IProcess process) throws CoreException {
		String errorMessage = process.getStreamsProxy().getErrorStreamMonitor().getContents();
		
		if (errorMessage.length() == 0) {
			errorMessage = process.getStreamsProxy().getOutputStreamMonitor().getContents();
		}
		
		if (errorMessage.length() != 0) {
			abort(errorMessage, null, IJavaLaunchConfigurationConstants.ERR_VM_LAUNCH_ERROR);
		}										
	}

	/**
	 * Get the appropriate JDI AttachingConnector instance.
	 * 
	 * @return
	 */
	private AttachingConnector getAttachingConnector() {
		AttachingConnector connector = null;
		
		List connectors = 
			Bootstrap.virtualMachineManager().attachingConnectors();
			
		for (int i= 0; i < connectors.size(); i++) {
			AttachingConnector c = (AttachingConnector) connectors.get(i);
			if ("com.sun.jdi.SocketAttach".equals(c.name())) //$NON-NLS-1$
				connector = c;
		}
		
		return connector;
	}
	
	/**
	 * Get the appropriate JDI ListenerConnector instance.
	 * 
	 * @return
	 */
	private ListeningConnector getListeningConnector() {
		ListeningConnector connector = null;
		
		List connectors = 
			Bootstrap.virtualMachineManager().listeningConnectors();
			
		for (int i= 0; i < connectors.size(); i++) {
			ListeningConnector c = (ListeningConnector) connectors.get(i);
			if ("com.sun.jdi.SocketListen".equals(c.name())) //$NON-NLS-1$
				connector = c;
		}
		
		return connector;
	}
	
	/**
	 * Returns the working directory to use for the launched VM,
	 * or <code>null</code> if the working directory is to be inherited
	 * from the current process.
	 * 
	 * @return the working directory to use
	 * @exception CoreException if the working directory specified by
	 *  the configuration does not exist or is not a directory
	 */	
	private File getWorkingDir(VMRunnerConfiguration config) 
		throws CoreException 
	{
		File dir = null;
		
		String path = config.getWorkingDirectory();
		
		if (path != null) {
			dir = new File(path);
			if (!dir.isDirectory()) {
				abort(MessageFormat.format (
						Messages.debugvmrunner_workingdir_not_dir, 
						new String[] {path}),
					null,
					IJavaLaunchConfigurationConstants.ERR_WORKING_DIRECTORY_DOES_NOT_EXIST);
			}
		}
		
		return dir;
	}
	
	/**
	 * Specify new connector arguments to the JDI connector.
	 * 
	 * @param map
	 * @param portNumber
	 */
	private void specifyArguments(Map map, int portNumber) {
		Connector.IntegerArgument port = 
			(Connector.IntegerArgument) map.get("port"); //$NON-NLS-1$
		port.setValue(portNumber);
		
		Connector.IntegerArgument timeoutArg = 
			(Connector.IntegerArgument) map.get("timeout"); //$NON-NLS-1$
		if (timeoutArg != null) {
			int timeout = 
				JavaRuntime.getPreferences().getInt(JavaRuntime.PREF_CONNECT_TIMEOUT);
			timeoutArg.setValue(timeout);
		}
	}

	protected String[] getCommandLine(ILaunchConfiguration config, Deployment deployment, int port, String projectNatureId, Device device) throws MtjException, CoreException {
		List arguments = new ArrayList(12);

		String commandLineString = executablePlatform.getArguments(config, deployment, debugMode, port, projectNatureId, device);
		ExecutionArguments execArgs = new ExecutionArguments("", commandLineString); //$NON-NLS-1$
		
		File executable = executablePlatform.getExecutable();
		if (executable != null) arguments.add(executable.toString());
		arguments.addAll(Arrays.asList(execArgs.getProgramArgumentsArray()));
			
		String[] cmdLine =
			(String[]) arguments.toArray(new String[arguments.size()]);
		
		return cmdLine;
	}

	/**
	 * @see org.eclipse.jdt.launching.IVMRunner#run(org.eclipse.jdt.launching.VMRunnerConfiguration, org.eclipse.debug.core.ILaunch, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void run(VMRunnerConfiguration configuration, ILaunch launch, IProgressMonitor monitor) 
		throws CoreException 
	{
		// Method provided to meet the superclass requirement.  Is not called.
	}
}

