/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.core.util;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IAccessRule;
import org.eclipse.jdt.core.IClasspathAttribute;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.project.ProjectFactory;
import org.eclipse.mtj.api.project.TargetDevice;
import org.eclipse.mtj.api.runtimes.DeviceConfiguration;
import org.eclipse.mtj.api.runtimes.DeviceProfile;
import org.eclipse.mtj.api.runtimes.ImplementationRef;
import org.eclipse.mtj.api.runtimes.JvmImplementation;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.api.runtimes.RuntimesFactory;
import org.eclipse.mtj.api.runtimes.ServiceApi;
import org.eclipse.mtj.core.Messages;

public class MtjEmfUtil {

	/**
	 * Return the classpath as a set of Eclipse JDT classpath entries.
	 * @return
	 */
	public static IClasspathEntry[] getClasspathEntries(RuntimePlatformDefinition rpd) {
		ArrayList filerefs = new ArrayList();
		
		Iterator it = rpd.getServiceApis().iterator();
		while (it.hasNext()) {
			Object ob = ((ServiceApi)it.next()).getImplementation().getFileRef();
			if ( filerefs.indexOf(ob) == -1 ) {
				filerefs.add(ob);
			}
		}
		it = rpd.getDeviceConfigurations().iterator();
		if (it.hasNext()) {
			Object ob = ((DeviceConfiguration)it.next()).getImplementation().getFileRef();
			if ( filerefs.indexOf(ob) == -1 ) {
				filerefs.add(ob);
			}
		}
		
		IClasspathEntry[] cpEntries = new IClasspathEntry[filerefs.size()];
		
		for (int i = 0; i < cpEntries.length; i++) {
			IPath entryPath = new Path((String)filerefs.get(i));
			IClasspathAttribute[] attributes = { };
			IPath sourceAttachmentPath = null;
			IPath sourceAttachmentRootPath = null;
			IAccessRule[] NO_ACCESS_RULES = new IAccessRule[0];

			IClasspathEntry entry = JavaCore.newLibraryEntry(
				entryPath, 
				sourceAttachmentPath, 
				sourceAttachmentRootPath,
				NO_ACCESS_RULES,
				attributes,
				false);

			cpEntries[i] = entry;
		}
		
		return cpEntries;
	}

	/**
	 * This method creates deep copy from the source RuntimePlatformDefinition object
	 * 
	 * @param rpd
	 * @return
	 */
	public static RuntimePlatformDefinition clone(RuntimePlatformDefinition rpd) {
		if (rpd == null) {
			return null;
		}

		RuntimePlatformDefinition resp = RuntimesFactory.eINSTANCE.createRuntimePlatformDefinition();
		resp.setName(rpd.getName());
		
		Iterator it = rpd.getDeviceConfigurations().iterator();
		while (it.hasNext()) {
			DeviceConfiguration dc = (DeviceConfiguration)it.next();
			resp.getDeviceConfigurations().add(clone(dc));
		}
		
		it = rpd.getServiceApis().iterator();
		while (it.hasNext()) {
			ServiceApi sa = (ServiceApi)it.next();
			resp.getServiceApis().add(clone(sa));
		}
		
		it = rpd.getDeviceProfiles().iterator();
		while (it.hasNext()) {
			DeviceProfile dp = (DeviceProfile)it.next();
			resp.getDeviceProfiles().add(clone(dp, resp));
		}
		
		return resp;
	}
	
	/**
	 * This method creates deep copy from the source DeviceConfiguration object
	 * 
	 * @param dc
	 * @return
	 */
	public static DeviceConfiguration clone(DeviceConfiguration dc) {
		if (dc == null) {
			return null;
		}
		
		DeviceConfiguration newdc = RuntimesFactory.eINSTANCE.createDeviceConfiguration();
		newdc.setName(dc.getName());
		newdc.setVersion(dc.getVersion());
		newdc.setJvmRef(clone(dc.getJvmRef()));
		newdc.setImplementation(clone(dc.getImplementation()));
		
		return newdc;
	}

	/**
	 * This method creates deep copy from the source ServiceApi object
	 * 
	 * @param sa
	 * @return
	 */
	public static ServiceApi clone(ServiceApi sa) {
		if (sa == null) {
			return null;
		}
		
		ServiceApi newsa = RuntimesFactory.eINSTANCE.createServiceApi();
		newsa.setName(sa.getName());
		newsa.setVersion(sa.getVersion());
		newsa.setDescription(sa.getDescription());
		newsa.setImplementation(clone(sa.getImplementation()));
		
		return newsa;
	}

	/**
	 * This method creates deep copy from the source DeviceProfile object
	 * 
	 * @param rpd
	 * @return
	 */
	public static DeviceProfile clone(DeviceProfile dp) {
		return clone(dp, null); 
	}

	/**
	 * This method creates deep copy from the source DeviceProfile object
	 * 
	 * @param dp
	 * @param rpd
	 * @return
	 */
	public static DeviceProfile clone(DeviceProfile dp, RuntimePlatformDefinition rpd) {
		if (dp == null) {
			return null;
		}
		
		DeviceProfile newdp = RuntimesFactory.eINSTANCE.createDeviceProfile();
		newdp.setName(dp.getName());
		newdp.setVersion(dp.getVersion());

		Iterator it = dp.getServiceApis().iterator();
		while (it.hasNext()) {
			ServiceApi sa = (ServiceApi)it.next();
			if (rpd == null) {
				newdp.getServiceApis().add(clone(sa));
			}
			else {
				ServiceApi newsa = getServiceApi(sa.getName(), sa.getVersion(), rpd);
				if ( newsa == null ) {
					newdp.getServiceApis().add(clone(sa));
				}
				else {
					newdp.getServiceApis().add(newsa);
				}
			}
		}
		
		return newdp;
	}

	private static ServiceApi getServiceApi(String name, String version, RuntimePlatformDefinition rpd) {
		Iterator it = rpd.getServiceApis().iterator();
		while (it.hasNext()) {
			ServiceApi sa = (ServiceApi)it.next();
			if ( (name == null || name.equals(sa.getName())) && 
				 (version == null || version.equals(sa.getVersion())) ) {
				return sa;
			}
		}

		return null;
	}
	
	/**
	 * This method creates deep copy from the source ImplementationRef object
	 * 
	 * @param ir
	 * @return
	 */
	public static ImplementationRef clone(ImplementationRef ir) {
		if (ir == null) {
			return null;
		}
		
		ImplementationRef newir = RuntimesFactory.eINSTANCE.createImplementationRef();
		newir.setFileRef(ir.getFileRef());
		newir.setJavadocRef(ir.getJavadocRef());
		newir.setSourceRef(ir.getSourceRef());
		
		return newir;
	}

	/**
	 * This method creates deep copy from the source JvmImplementation object
	 * 
	 * @param ji
	 * @return
	 */
	public static JvmImplementation clone(JvmImplementation ji) {
		if (ji == null) {
			return null;
		}
		
		JvmImplementation newjir = RuntimesFactory.eINSTANCE.createJvmImplementation();
		newjir.setFileRef(ji.getFileRef());
		newjir.setName(ji.getName());
		newjir.setVendor(ji.getVendor());
		newjir.setVersion(ji.getVersion());
		
		return newjir;
	}

	/**
	 * This method creates deep copy from the source TargetDevice object
	 * 
	 * @param td
	 * @return
	 */
	public static TargetDevice clone(TargetDevice td) {
		if (td == null) {
			return null;
		}
		
		TargetDevice newtd = ProjectFactory.eINSTANCE.createTargetDevice();
		newtd.setName(td.getName());
		newtd.setDescription(td.getDescription());
		newtd.setDevicePlatformName(td.getDevicePlatformName());
		newtd.setDeviceName(td.getDeviceName());
		newtd.setDeviceGroupName(td.getDeviceGroupName());
		newtd.setRuntimePlatform(clone(td.getRuntimePlatform()));
		
		return newtd;
	}

	/**
	 * This method returns true if device 1 is same as device 2.
	 * 
	 * @param device1
	 * @param device2
	 * @return
	 */
	public static boolean equals(Device device1, Device device2) {
		return ( ( device1.getName() != null && device1.getName().equals(device2.getName()) ) || 
				 ( device1.getName() == null && device2.getName() == null )
			   ) &&
			   ( ( device1.getVendor() != null && device1.getVendor().equals(device2.getVendor()) ) || 
				 ( device1.getVendor() == null && device2.getVendor() == null )
			   ) &&
			   ( ( device1.getVersion() != null && device1.getVersion().equals(device2.getVersion()) ) || 
				 ( device1.getVersion() == null && device2.getVersion() == null )
			   );
	}
	
	public static String toString(Device device) {
		return
		(device.getVendor() != null ? device.getVendor() + " " : "") + //$NON-NLS-1$ //$NON-NLS-2$
		(device.getName() != null ? device.getName() + " " : Messages.MtjEmfUtil_UnknownName + " ") + //$NON-NLS-1$ //$NON-NLS-2$
		(device.getVersion() != null ? Messages.MtjEmfUtil_Version + " " + device.getVersion() : "");  //$NON-NLS-1$ //$NON-NLS-2$
	}
}
