/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 

package org.eclipse.mtj.core.internal.ota;

import java.io.File;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.mtj.core.IEclipseMtjCoreConstants;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.mortbay.http.HttpContext;
import org.mortbay.http.HttpListener;
import org.mortbay.http.HttpServer;
import org.mortbay.http.SocketListener;
import org.mortbay.http.handler.NotFoundHandler;
import org.mortbay.util.Log;
import org.mortbay.util.MultiException;
import org.mortbay.util.OutputStreamLogSink;
 
/**
 * Singleton management of the Over the Air HTTP Server.
 */
public class OTAServer {
	/** Singleton instance of the OTAServer */
	public static final OTAServer instance = new OTAServer();
	
	/**
	 * Return the port to be listened on for OTA connections.
	 * 
	 * @return
	 */
	public static int getPort() {
		HttpListener listener = instance.getHttpServer().getListeners()[0];
		return listener.getPort();
	}

	/**
	 * 
	 * @uml.property name="httpServer"
	 * @uml.associationEnd 
	 * @uml.property name="httpServer" multiplicity="(1 1)"
	 */
	// The HttpServer instance in use for serving the content
	private HttpServer httpServer;

	
	/**
	 * Private constructor
	 */
	private OTAServer() {
		super();
	}
	
	/**
	 * Start the HttpServer if it has not already been started.
	 * 
	 * @throws MultiException
	 */
	public void start() 
		throws MultiException 
	{
		HttpServer server = getHttpServer();
		if (!server.isStarted()) {
			server.start();
		}
	}
	
	/**
	 * Stop the HttpServer if it is currently running.
	 * 
	 * @throws InterruptedException
	 */
	public void stop() 
		throws InterruptedException 
	{
		HttpServer server = getHttpServer();
		if (server.isStarted()) {
			server.stop();
		}
	}
	
	/**
	 * Configure the context(s) for the HttpServer.
	 * 
	 * @param server
	 */
	private void configureContext(HttpServer server) {
		// Configure the context
		HttpContext context = new HttpContext();
		context.setContextPath("/ota/*"); //$NON-NLS-1$
		server.addContext(context);
		
		// Configure the handlers in the context
		context.addHandler(new OTAHandler());
		context.addHandler(new NotFoundHandler());	
	}

	/**
	 * Configure the HttpServer listener.
	 * 
	 * @param server
	 */
	private void configureListener(HttpServer server) {
		SocketListener listener = new SocketListener();
		listener.setMinThreads(1);
		listener.setMaxThreads(2);
		listener.setMaxIdleTimeMs(10000);
		listener.setDaemon(true);

		Preferences prefs = MtjCorePlugin.getDefault().getPluginPreferences(); 
		if (prefs.getBoolean(IEclipseMtjCoreConstants.PREF_OTA_PORT_DEFINED)) {
			int specifiedPortNumber = prefs.getInt(IEclipseMtjCoreConstants.PREF_OTA_PORT);
			listener.setPort(specifiedPortNumber);		
		}
		
		server.addListener(listener);
	}

	/**
	 * Configure the Jetty logging support so that it writes
	 * out to our metadata directory.
	 */
	private void configureLogging() 
		throws CoreException 
	{
		// Set some System properties so that they are available
		// when creating the logging
		System.setProperty("LOG_FILE_RETAIN_DAYS", "5"); //$NON-NLS-1$ //$NON-NLS-2$
		
		// Calculate the name of the file to be used for logging
		IPath stateLocation = MtjCorePlugin.getDefault().getStateLocation();
		IPath logDirectory = stateLocation.append("jetty"); //$NON-NLS-1$
		
		File logDirectoryFile = logDirectory.toFile();
		if (!logDirectoryFile.exists()) {
			logDirectoryFile.mkdir();
		}

		File logFile = new File(logDirectoryFile, "logfile"); //$NON-NLS-1$
		String filename = logFile + "yyyy_mm_dd.txt"; //$NON-NLS-1$
		
		// Create the output sink and add it to the logging class
		OutputStreamLogSink sink = new OutputStreamLogSink(filename);
		sink.start();
		Log.instance().add(sink);
	}

	/**
	 * Create and configure the HTTP server instance.
	 * 
	 * @return
	 */
	private HttpServer createHttpServer() {
		HttpServer server = new HttpServer();

		configureListener(server);
		configureContext(server);
		
		return server;
	}

	/**
	 * Get the HttpServer instance.
	 * 
	 * @return
	 * 
	 * @uml.property name="httpServer"
	 */
	private HttpServer getHttpServer() {
		if (httpServer == null) {
			httpServer = createHttpServer();
			try {
				configureLogging();
			} catch (CoreException e) {
				MtjCorePlugin.log(IStatus.WARNING, "configureLogging", e); //$NON-NLS-1$
			}
		}

		return httpServer;
	}

}

