/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 

package org.eclipse.mtj.internal.utils;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Reader;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

/**
 * A special Properties subclass that reads and writes using ':' instead of '='
 * for the key/value separator.
 */
public class ColonDelimitedProperties extends Properties {

	/**
	 * Provde default SerialVersionID
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * Creates an empty property list with no default values.
	 */
	public ColonDelimitedProperties() {
		super();
	}

	/**
	 * Creates a property list with the specified defaults.
	 *
	 * @param   defaults   the defaults.
	 */
	public ColonDelimitedProperties(Properties defaults) {
		super(defaults);
	}

	/**
	 * @see java.util.Properties#load(java.io.InputStream)
	 */
	public synchronized void load(InputStream inStream) throws IOException {
		Reader dataReader = new InputStreamReader(inStream);
		load(dataReader);
	}

	/**
	 * Load the colon-delimited properties from the specified
	 * Reader.
	 * 
	 * @param reader the reader containing the properties
	 * @throws IOException if an error occurs loading the properties
	 */
	public synchronized void load(Reader reader)
		throws IOException
	{
		BufferedReader bufferedReader = new BufferedReader(reader);
		
		String line = null;
		while ((line = bufferedReader.readLine()) != null) {
			// Find the first colon and break the string up
			line = line.trim();
			if ((line.length() > 0) && !line.startsWith("#")) { //$NON-NLS-1$
				int colonIndex = line.indexOf(':');
				if (colonIndex != -1) {
					String name = line.substring(0, colonIndex);
					if (colonIndex + 2 < line.length()) {
						String value = line.substring(colonIndex + 2);
						setProperty(name, value);
					}
				}
			}
		}
	}
	
	/**
	 * @see java.util.Properties#store(java.io.OutputStream, java.lang.String)
	 */
	public synchronized void store(OutputStream out, String header)
		throws IOException 
	{
		OutputStreamWriter osw = new OutputStreamWriter(out, "UTF-8"); //$NON-NLS-1$
		PrintWriter writer = new PrintWriter(osw);
		
		Iterator iter = entrySet().iterator();
		while (iter.hasNext()) {
			Map.Entry entry = (Map.Entry) iter.next();
			writer.println(entry.getKey() + ": " + entry.getValue()); //$NON-NLS-1$
		}
		
		writer.flush();
	}
}
