/**
 * Copyright (c) 2003-2005 Craig Setera
 * All Rights Reserved.
 * Licensed under the Eclipse Public License - v 1.0
 * For more information see http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.mtj.extension.preverify.internal;

import org.objectweb.asm.Type;
import org.objectweb.asm.tree.analysis.SimpleVerifier;

/**
 * A subclass of the SimpleVerifier that allows a classloader to be
 * specified for use in locating and loading classes used during
 * verification.  This verifier also extends the type system provided
 * by the SimpleVerifier, allowing boolean, byte, character and short
 * arrays to be passed through the type system.
 * <p />
 * Copyright (c) 2003-2005 Craig Setera<br>
 * All Rights Reserved.<br>
 * Licensed under the Eclipse Public License - v 1.0<p/>
 * <br>
 * $Revision: 1.1.1.1 $
 * <br>
 * $Date: 2006/08/30 18:52:02 $
 * <br>
 * @author Craig Setera
 */
public class SimpleVerifierPlusClassloader extends SimpleVerifier {
	private ClassLoader classLoader;

	/**
	 * Construct a new verifier.
	 * 
	 * @param classLoader
	 */
	public SimpleVerifierPlusClassloader(ClassLoader classLoader) {
		super();
		this.classLoader = classLoader;
	}

	/**
	 * @see org.objectweb.asm.tree.analysis.SimpleVerifier#getClass(org.objectweb.asm.Type)
	 */
	protected Class getClass(Type t) {
		Class clazz = null;
		
		if (classLoader == null) {
			clazz = super.getClass(t); 
		} else {
	        try {
	        	String className = getRootClassName(t);
	        	clazz = classLoader.loadClass(className);
	        	
	        } catch (ClassNotFoundException e) {
	        	e.printStackTrace();
	            throw new RuntimeException(e.toString());
	        }
		}

		return clazz; 
	}

	/**
	 * Return the root class name for the specified type, having
	 * removed array prefixes and such.
	 * 
	 * @param type
	 * @return
	 */
	protected String getRootClassName(Type type) {
		String rootClassName = null;
		
    	switch (type.getSort()) {
	    	case Type.BOOLEAN:
	    		rootClassName = "java.lang.Boolean"; //$NON-NLS-1$
	    		break;
	    		
	    	case Type.CHAR:
	    		rootClassName = "java.lang.Character"; //$NON-NLS-1$
	    		break;
	    		
	    	case Type.BYTE:
	    		rootClassName = "java.lang.Byte"; //$NON-NLS-1$
	    		break;
	    		
	    	case Type.SHORT:
	    		rootClassName = "java.lang.Short"; //$NON-NLS-1$
	    		break;
	    		
	    	case Type.INT:
	    		rootClassName = "java.lang.Integer"; //$NON-NLS-1$
	    		break;
	    		
	    	case Type.FLOAT:
	    		rootClassName = "java.lang.Float"; //$NON-NLS-1$
	    		break;
	    		
	    	case Type.LONG:
	    		rootClassName = "java.lang.Long"; //$NON-NLS-1$
	    		break;
	    		
	    	case Type.DOUBLE:
	    		rootClassName = "java.lang.Double"; //$NON-NLS-1$
	    		break;
	    		
	    	case Type.ARRAY:
	    		rootClassName = getRootClassName(type.getElementType());
	    		break;
	    		
	    	case Type.OBJECT:
	    		rootClassName = type.getInternalName().replace('/', '.');
	        	break;
    	}
    	
    	return rootClassName;
	}
}
