/*******************************************************************************
* Copyright (c) 2006 Nokia Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
* Contributors:
*IBM - Templates additions
*******************************************************************************/

package org.eclipse.mtj.extension.devide;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.mtj.core.util.DirUtil;
import org.eclipse.mtj.core.version.Version;
import org.eclipse.mtj.extension.devide.config.LibrarySpecification;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;

/**
 * The main plugin class to be used in the desktop.
 */
public class MtjDevIdePlugin extends AbstractUIPlugin {

	public static final String PLUGIN_ID = "org.eclipse.mtj.extension.devide"; //$NON-NLS-1$
	public static final String J2ME_CONFIGURATIONS_ID = Messages.MtjDevIdePlugin_ConfigurationId;
	public static final String J2ME_PROFILES_ID = Messages.MtjDevIdePlugin_ProfileId;
	public static final String TEMPLATES_DIR = "templates"; //$NON-NLS-1$
	public static final String MIDP_TEMPLATES_DIR = "MIDP"; //$NON-NLS-1$
	public static final String PPRO_TEMPLATES_DIR = "PPRO"; //$NON-NLS-1$
	public static final String JAR_FILE_EXTENSION_SEPERATOR = ".jar!"; //$NON-NLS-1$
	public static final String JAR_FILE_SEPERATOR = "!"; //$NON-NLS-1$
	public static final String FILE_PROTOCOL = "file:"; //$NON-NLS-1$
	public static final String TEMPLATE_EXTENSION = ".template"; //$NON-NLS-1$
	public static final byte MTJ_TEMPLATE_MIDP = 1;
	public static final byte MTJ_TEMPLATE_PPRO = 2;

	// The configuration specifications provided by extension points
	private LibrarySpecification[] configSpecs;
	
	// The profile specifications provided by extension points
	private LibrarySpecification[] profileSpecs;

	//The shared instance.
	private static MtjDevIdePlugin plugin;
	
	/**
	 * The constructor.
	 */
	public MtjDevIdePlugin() {
		plugin = this;
	}

	/**
	 * This method is called upon plug-in activation
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
	}

	/**
	 * This method is called when the plug-in is stopped
	 */
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
		plugin = null;
	}

	/**
	 * Returns the shared instance.
	 */
	public static MtjDevIdePlugin getDefault() {
		return plugin;
	}

	/**
	 * Returns an image descriptor for the image file at the given
	 * plug-in relative path.
	 *
	 * @param path the path
	 * @return the image descriptor
	 */
	public static ImageDescriptor getImageDescriptor(String path) {
		return AbstractUIPlugin.imageDescriptorFromPlugin(PLUGIN_ID, path);
	}
	
	/**
	 * Return the configuration specifications provided by the plugin
	 * extension points.
	 * 
	 * @return the configuration specifications
	 * @throws CoreException 
	 */
	public static LibrarySpecification[] getConfigurationSpecifications() 
		throws CoreException 
	{
		MtjDevIdePlugin plugin = getDefault();
		if (plugin.configSpecs == null) {
			IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint(
					PLUGIN_ID, 
					J2ME_CONFIGURATIONS_ID); 
			IConfigurationElement[] configElements = point.getConfigurationElements();
			plugin.configSpecs = new LibrarySpecification[configElements.length];	
			
			for (int i = 0; i < configElements.length; i++) {
				plugin.configSpecs[i] = createComponentSpecification(configElements[i]);
			}
		}
		
		return plugin.configSpecs;
	}
	
	/**
	 * Create a component specification for the specified configuration
	 * element
	 * @param element
	 * @return
	 * @throws CoreException 
	 */
	private static LibrarySpecification createComponentSpecification(IConfigurationElement element) 
		throws CoreException 
	{
		String id = element.getAttribute("id"); //$NON-NLS-1$
		String name = element.getAttribute("name"); //$NON-NLS-1$
		String versionString = element.getAttribute("version"); //$NON-NLS-1$
		Version version = new Version(versionString);

		LibrarySpecification specification = new LibrarySpecification();
		specification.setIdentifier(id);
		specification.setName(name);
		specification.setVersion(version);
		
		return specification;
	}

	/**
	 * Return the profile specifications provided by the plugin
	 * extension points.
	 * 
	 * @return the profile specifications
	 * @throws CoreException 
	 */
	public static LibrarySpecification[] getProfileSpecifications() 
		throws CoreException 
	{
		MtjDevIdePlugin plugin = getDefault();
		if (plugin.profileSpecs == null) {
			IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint(
					PLUGIN_ID, 
					J2ME_PROFILES_ID); 
			IConfigurationElement[] configElements = point.getConfigurationElements();
			plugin.profileSpecs = new LibrarySpecification[configElements.length];	
			
			for (int i = 0; i < configElements.length; i++) {
				plugin.profileSpecs[i] = createComponentSpecification(configElements[i]);
			}
		}
		
		return plugin.profileSpecs;
	}

	/**
	 * This method returns the plugin's icons directory location in the file system.
	 * @return plugin's icons directory path in the file system
	 */
	public static String getPluginIconsDir()
	{
		return getInstallLocation().append("/icons").toString(); //$NON-NLS-1$
	}
	
	/**
	 * This method populates the plugin's imageRegistry with icons found in the directory.
	 * @return plugin's icons directory path
	 */
	protected void initializeImageRegistry (ImageRegistry imageRegistry)
	{
		super.initializeImageRegistry(imageRegistry);
		try
		{
			ArrayList images = DirUtil.getAllFilesInDirectory (getPluginIconsDir(), true, true, "*.gif"); //$NON-NLS-1$
			String fileName;
			String imageKey;
			int pos;
			for (Iterator iter = images.iterator(); iter.hasNext();)
			{
				fileName = (String) iter.next();
				pos = fileName.indexOf("/icons/"); //$NON-NLS-1$
				if (pos != -1)
				{
					pos += 7;
					imageKey = fileName.substring(pos);
					imageRegistry.put(imageKey, ImageDescriptor.createFromURL(new URL("file:" + fileName))); //$NON-NLS-1$
				}
			}
		}
		catch (Exception e)
		{
			e.getMessage();
		}
	}

	public static ImageRegistry getMtjImageRegistry ()
	{
		return getDefault().getImageRegistry();
	}
	
	
	public static String getPluginId ()
	{
		return getDefault().getBundle().getSymbolicName();
	}

	/**
	 * This method returns the plugin's installation directory.
	 * @return plugin's installation directory path
	 */
	public static Path getInstallLocation()
	{
		try
		{
			Bundle bundle = getDefault().getBundle();
			URL url = FileLocator.find(bundle, new Path(""), null);  //$NON-NLS-1$
			String s1 = FileLocator.resolve(url).getFile();
			if(s1.startsWith("/")) //$NON-NLS-1$
				s1 = s1.substring(1);
			s1 = (new Path(s1)).toOSString();
			String s;
			if(s1.endsWith(File.separator))
				s = s1;
			else
				s = s1 + File.separator;
			return new Path(s);
		}
		catch(Exception exception)
		{
			return null;
		}
	}

	/**
	 * Returns the templates .
	 */
	public static String [] getTemplateFiles(byte type){
		File templatesDir = null;
		String [] dirFileArray;
		JarFile pluginjarfile;
		List<String> filteredFileList = new ArrayList<String>();
				
		templatesDir = new File(getTemplatesPath(type).toOSString());
		if (isJarfile(templatesDir.getPath()))
		{
			String jarFileName = templatesDir.toString();
			
			int index = jarFileName.indexOf(JAR_FILE_SEPERATOR);
			if(index != -1){
				jarFileName = jarFileName.substring(0,index);
			}
			
			try {
				pluginjarfile = new JarFile(jarFileName);
				Enumeration jarentries = null;
				
				jarentries = pluginjarfile.entries();
				String entrieName = null;
				
				while(jarentries.hasMoreElements())
				{
					entrieName = ((ZipEntry)jarentries.nextElement()).getName();
					if(entrieName.contains((type == MTJ_TEMPLATE_MIDP ? MIDP_TEMPLATES_DIR : PPRO_TEMPLATES_DIR)) && entrieName.endsWith(TEMPLATE_EXTENSION)){
						filteredFileList.add(getLastSegment(entrieName));
					}
				}
			} catch (IOException e1) {
				filteredFileList.add("NULL"); //$NON-NLS-1$
			}
		} else {
			dirFileArray = templatesDir.list();
			for (int i = 0; i < dirFileArray.length; i++){
				if (dirFileArray[i].endsWith(TEMPLATE_EXTENSION)){
					filteredFileList.add(dirFileArray[i]);
				}
			}
		}
		return (String[])filteredFileList.toArray(new String[filteredFileList.size()]);			
	}

	public static Path getTemplatesPath(byte type) {
		Path pa;
		StringBuffer sb = new StringBuffer(getInstallLocation().toOSString());
		sb.append(getTemplatesSegments(type));
		pa = new Path(removeProtocol(sb.toString()));
		return pa;
	}
	
	public static String getTemplatesSegments(byte type){
		Path p = new Path(TEMPLATES_DIR  + File.separator + (type == MTJ_TEMPLATE_MIDP ? MIDP_TEMPLATES_DIR : PPRO_TEMPLATES_DIR));
		return  p.toOSString();
	}

	private static String removeProtocol(String templatesPath) {
		int index = 0;
		if (templatesPath.startsWith(FILE_PROTOCOL))
			index = FILE_PROTOCOL.length();
		return templatesPath.substring(index); 
	}

	private static boolean isJarfile(String templatesPath) {
		return templatesPath.contains(JAR_FILE_EXTENSION_SEPERATOR);
	}
	
	private static String getLastSegment(String entrieName) {
		Path tempPath = new Path(entrieName);
		return tempPath.lastSegment();
	}
}
