/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 

package org.eclipse.mtj.extension.devide.executable;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Set;
import java.util.StringTokenizer;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.mtj.api.deployment.Deployment;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.enumerations.DeviceCommunicationProtocol;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.api.model.MidletDeployment;
import org.eclipse.mtj.core.IEclipseMtjCoreConstants;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.mtj.core.launching.ILaunchConstants;
import org.eclipse.mtj.exception.MtjException;

public abstract class AbstractExecutable {

	/**
	 * Add the debug arguments to the current command line arguments.
	 * 
	 * @param arguments
	 * @param port
	 * @throws CoreException
	 */
	protected void addDebugArguments(ArrayList arguments, int port, boolean debugServer) 
		throws CoreException 
	{
		addXDebugArgument(arguments);
		// TODO add selection to somewhere
		// addXNoagentArgument(arguments);
		addXRunjdwpArgument(arguments, port, debugServer);
	}

	/**
	 * Add the -Xdebug argument to the arguments.
	 * 
	 * @param arguments
	 */
	protected void addXDebugArgument(ArrayList arguments) 	
		throws CoreException 
	{
		arguments.add("-Xdebug"); //$NON-NLS-1$
	}

	/**
	 * Add the -Xnoagent argument to the arguments.
	 * 
	 * @param arguments
	 */
	protected void addXNoagentArgument(ArrayList arguments)
		throws CoreException
	{
		arguments.add("-Xnoagent"); //$NON-NLS-1$
	}

	/**
	 * Add the -Xrunjdwp argument to the arguments.
	 * 
	 * @param arguments
	 * @param port
	 */
	protected void addXRunjdwpArgument(ArrayList arguments, int port, boolean debugServer)
		throws CoreException
	{
		String hostname = "localhost"; //$NON-NLS-1$
		
		try {
			// The Sun toolkit (as of WTK 2.3) won't accept localhost, so we try
			// to get a valid local host name
			InetAddress address = InetAddress.getLocalHost();
			hostname = address.getHostName();
		} catch (UnknownHostException e) {
			MtjCorePlugin.throwCoreException(IStatus.WARNING, -999, e);
		}
		
		StringBuffer argument = new StringBuffer("-Xrunjdwp:transport=dt_socket,suspend=y,server="); //$NON-NLS-1$
		argument.append(debugServer ? "y" : "n"); //$NON-NLS-1$ //$NON-NLS-2$
		// TODO hostname from settings
		argument.append(",address="); //.append(hostname).append(":"); //$NON-NLS-1$
		argument.append(port);
				
		arguments.add(argument.toString());
	}

	/**
	 * Add the classpath to the arguments.
	 * 
	 * @param arguments
	 * @param configuration
	 * @throws CoreException
	 */
	protected void addClasspath(
		ArrayList arguments, 
		ILaunchConfiguration configuration,
		String natureId,
		IProgressMonitor monitor) 
			throws CoreException 
	{
		arguments.add("-classpath"); //$NON-NLS-1$
		addDeploymentClasspath(arguments, configuration, natureId, monitor);
	}

	/**
	 * Add the classpath for running the midlets in
	 * this project.
	 * 
	 * @param progArgs
	 * @param configuration
	 * @param monitor
	 * @throws CoreException
	 */
	private void addDeploymentClasspath(
		ArrayList progArgs, 
		ILaunchConfiguration configuration,
		String natureId,
		IProgressMonitor monitor) 
			throws CoreException 
	{
		progArgs.add(getQuotedString(getClasspathString(configuration, natureId, monitor)));
	}
	
	/**
	 * Return the specified string appropriately quoted.
	 * 
	 * @param string
	 * @return
	 */
	protected String getQuotedString(String string) {
		StringBuffer sb = new StringBuffer("\""); //$NON-NLS-1$
		sb.append(string);
		sb.append("\""); //$NON-NLS-1$
		
		return sb.toString();
	}
	
	/**
	 * Get the emulators classpath string.
	 * 
	 * @param launchConfig
	 * @param monitor
	 * @return
	 * @throws CoreException
	 */
	protected String getClasspathString(
		ILaunchConfiguration launchConfig, 
		String natureId,
		IProgressMonitor monitor) 
			throws CoreException 
	{
		Set cpEntries = getClasspathEntries(launchConfig, natureId, monitor);
		
		// Convert to a valid classpath string
		StringBuffer cp = new StringBuffer();
		
		Iterator iter = cpEntries.iterator();
		while (iter.hasNext()) {
			IPath path = (IPath) iter.next();
			cp.append(path.toOSString());
			if (iter.hasNext()) cp.append(File.pathSeparatorChar);
		}
				
		return cp.toString();
	}

	/**
	 * Get the list classpath entries, mapped into the verified output
	 * folder.
	 * 
	 * @param launchConfig
	 * @param monitor
	 * @return
	 * @throws CoreException
	 * @throws JavaModelException
	 */
	private Set getClasspathEntries(ILaunchConfiguration launchConfig, String natureId, IProgressMonitor monitor) 
		throws CoreException, JavaModelException 
	{
		// Do some setup
		IJavaProject javaProject = getJavaProject(launchConfig);
		IMtjProject suite = getMtjProject(launchConfig);
		
		// The verified output locations
		IPath classesPath = suite.getVerifiedClassesOutputFolder(monitor).getLocation();
		IPath libsPath = suite.getVerifiedLibrariesOutputFolder(monitor).getLocation();
		
		// Build up the list of IPath entries based on the project
		// classpath setup to maintain relative ordering
		ClasspathCollectionVisitor visitor = 
			new ClasspathCollectionVisitor(classesPath, libsPath);
		visitor.getRunner().run(javaProject, visitor, natureId, monitor);
		Set cpEntries = visitor.getCpEntries();
		
		return cpEntries;
	}

	protected abstract IMtjProject getMtjProject(IJavaProject javaProject) throws MtjException;
	
	/**
	 * Return the midlet suite which is being launched.
	 * 
	 * @param configuration
	 * @return
	 * @throws CoreException
	 */
	protected IMtjProject getMtjProject(ILaunchConfiguration configuration) 
		throws CoreException 
	{
		IJavaProject javaProject = getJavaProject(configuration);
		try {
			return getMtjProject(javaProject);
		} catch (MtjException e) {
			// timeout, consult status handler if there is one
			IStatus status = new Status(
				IStatus.ERROR, 
				IEclipseMtjCoreConstants.PLUGIN_ID, 
				MtjCoreErrors.PROJECT_INSTANTIATION_ERROR, 
				MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_INSTANTIATION_ERROR), e);
			throw new CoreException(status);
		}
	}
	
	/**
	 * Returns the Java project specified by the given 
	 * launch configuration, or <code>null</code> if none.
	 * 
	 * @param configuration launch configuration
	 * @return the Java project specified by the given 
	 *  launch configuration, or <code>null</code> if none
	 * @exception CoreException if unable to retrieve the attribute
	 */
	protected IJavaProject getJavaProject(ILaunchConfiguration configuration) throws CoreException {
		String projectName = getJavaProjectName(configuration);
		if (projectName != null) {
			projectName = projectName.trim();
			if (projectName.length() > 0) {
				IProject project = 
					MtjCorePlugin.getWorkspace().getRoot().getProject(projectName);
				IJavaProject javaProject = JavaCore.create(project);
				if (javaProject != null && javaProject.exists()) {
					return javaProject;
				}
			}
		}
		return null;
	}
	
	/**
	 * Returns the Java project name specified by the given 
	 * launch configuration, or <code>null</code> if none.
	 * 
	 * @param configuration launch configuration
	 * @return the Java project name specified by the given 
	 *  launch configuration, or <code>null</code> if none
	 * @exception CoreException if unable to retrieve the attribute
	 */
	private String getJavaProjectName(ILaunchConfiguration configuration) throws CoreException {
		return configuration.getAttribute(
				IJavaLaunchConfigurationConstants.ATTR_PROJECT_NAME, 
				(String)null);
	}

	/**
	 * Add an argument to the program arguments if it is
	 * found in the launch configuration, or skip it 
	 * otherwise.
	 * 
	 * @param progArgs
	 * @param configuration
	 * @param configKey
	 * @param commandString
	 * @param keyValueSpace
	 */
	protected void addLaunchConfigurationArgument(
			ArrayList progArgs, 
			ILaunchConfiguration configuration, 
			String configKey, 
			String commandString, 
			boolean keyValueSpace) 
				throws CoreException 
		{
			String configValue = configuration.getAttribute(configKey, (String)null);
				
			if (configValue != null) {
				if (keyValueSpace) {
					progArgs.add(commandString);
					progArgs.add(configValue);
				} else {
					progArgs.add(commandString + configValue);
				}
			}
		}

	protected void addLaunchConfigurationArgument(
		ArrayList progArgs, 
		ILaunchConfiguration configuration, 
		Device device, 
		String commandString, 
		boolean keyValueSpace) 
			throws CoreException 
	{
		String configValue = device.getName();
			
		if (configValue != null) {
			if (keyValueSpace) {
				progArgs.add(commandString);
				progArgs.add(configValue);
			} else {
				progArgs.add(commandString + configValue);
			}
		}
	}

	/**
	 * Add the descriptor argument.
	 * 
	 * @param arguments
	 * @param configuration
	 * @throws CoreException
	 */
	protected void addDescriptorArgument(ArrayList arguments, Deployment deployment, ILaunchConfiguration configuration, Device device) 
		throws CoreException 
	{
		if (deployment instanceof MidletDeployment) {
			addDescriptorArgument( arguments, (MidletDeployment)deployment,  configuration, device);
		}
	}
	
	protected void addDescriptorArgument(ArrayList arguments, MidletDeployment deployment, ILaunchConfiguration configuration, Device device) 
	throws CoreException 
{
	if (!shouldDoOTA(configuration, device)) {
		File jadFile = deployment.getJad();
		if (jadFile != null) {
			arguments.add(getQuotedString("-Xdescriptor:" +	jadFile.getAbsolutePath())); //$NON-NLS-1$
		}
	}
}

	/**
	 * Return a boolean indicating whether or not the emulation should
	 * be launched as OTA.
	 * 
	 * @param configuration
	 * @return
	 * @throws CoreException
	 */
	protected boolean shouldDoOTA(ILaunchConfiguration configuration, Device device) throws CoreException {
		if ( device != null ) {
			return device.getCommunicationProtocol().getValue() == DeviceCommunicationProtocol.PROTOCOL_OTA;
		}
		return false;
	}

	/**
	 * Add the miscellaneous argument settings to the arguments.
	 * 
	 * @param arguments
	 * @param configuration
	 * @throws CoreException
	 */
	protected void addMiscellaneousArguments(ArrayList arguments, ILaunchConfiguration configuration, Device device) throws CoreException {
		addLaunchConfigurationArgument(arguments, configuration, 
			ILaunchConstants.VERBOSITY_OPTIONS, "-Xverbose:", false); //$NON-NLS-1$
		addLaunchConfigurationArgument(arguments, configuration, 
			ILaunchConstants.HEAP_SIZE, "-Xheapsize:", false); //$NON-NLS-1$
	}

	/**
	 * Add the security domain to the arguments if specified by the
	 * user on the launch configuration.
	 * 
	 * @param progArgs
	 * @param configuration
	 * @throws CoreException
	 */
	protected void addSecurityDomain(ArrayList progArgs, ILaunchConfiguration configuration) 
		throws CoreException
	{
		String domainName = configuration.getAttribute(
			ILaunchConstants.SECURITY_DOMAIN,
			ILaunchConstants.NO_SECURITY_DOMAIN);
		
		if (!domainName.equals(ILaunchConstants.NO_SECURITY_DOMAIN)) {
			progArgs.add("-Xdomain:" + domainName); //$NON-NLS-1$
		}
	}

	/**
	 * Add the extra parameters specified by the user on the launch configuration.
	 * 
	 * @param arguments
	 * @param configuration
	 * @throws CoreException
	 */
	protected void addExtraLaunchArguments(ArrayList arguments, ILaunchConfiguration configuration)
		throws CoreException
	{
		String extraArguments = configuration.getAttribute(ILaunchConstants.LAUNCH_PARAMS, ""); //$NON-NLS-1$
		StringTokenizer st = new StringTokenizer(extraArguments);
		while (st.hasMoreTokens()) {
			arguments.add(st.nextToken());
		}
	}

	/**
	 * Add the emulation target to the command line, taking into account
	 * the over the air versus midlet class choices.
	 * 
	 * @param arguments
	 * @param configuration
	 * @throws CoreException
	 */
	protected void addEmulationTarget(ArrayList arguments, ILaunchConfiguration configuration, Deployment deployment, Device device) throws CoreException {
		// Handle OTA versus class launching
		if (shouldDoOTA(configuration, device)) {
			addOverTheAirParameter(arguments, configuration, deployment);
		} else {
			addLaunchConfigurationArgument(arguments, configuration, 
				ILaunchConstants.EXECUTABLE_CLASS, " ", false); //$NON-NLS-1$
		}
	}

	/**
	 * Add the parameters for OTA support.
	 * 
	 * @param args
	 * @param configuration
	 * @throws CoreException
	 */
	private void addOverTheAirParameter(ArrayList args, ILaunchConfiguration configuration, Deployment deployment) throws CoreException {
		String url = getOTAURL(configuration, deployment);
		if ( url != null ) {
			String param = "-Xjam:transient=" + getOTAURL(configuration, deployment);  		 //$NON-NLS-1$
			args.add(param);
		}
	}

	/**
	 * Return the Over the Air URL for accessing the JAD file
	 * via the built-in OTA HTTP server.
	 * @param launchConfig
	 * @return
	 * @throws CoreException
	 */
	protected String getOTAURL(ILaunchConfiguration launchConfig, Deployment deployment) throws CoreException
	{
		if (deployment instanceof MidletDeployment) {
			return getOTAURL( launchConfig, (MidletDeployment)deployment);
		}
		
		return null;
	}

/*
	protected String getOTAURL(ILaunchConfiguration launchConfig, MidletDeployment deployment)
		throws CoreException
	{
		// If we are doing OTA launching, make sure that the
		// OTA server has been started
		try {
			OTAServer.instance.start();
		} catch (MultiException e) {
			MtjCorePlugin.log(IStatus.WARNING, "launch", e);
		}
		
		IMtjProject mtjProject;
		try {
			mtjProject = MtjProject.getMtjProject(getJavaProject(launchConfig));
		} catch (MtjException e) {
			// timeout, consult status handler if there is one
			IStatus status = new Status(
				IStatus.ERROR, 
				IEclipseMtjCoreConstants.PLUGIN_ID, 
				MtjCoreErrors.PROJECT_INSTANTIATION_ERROR, 
				MtjCoreErrors.getErrorMessage(MtjCoreErrors.PROJECT_INSTANTIATION_ERROR), e);
			throw new CoreException(status);
		} catch (CoreException e) {
			throw e;
		}
		String projectName = mtjProject.getProject().getName();
		String jadName = deployment.getJad().getName();
		
		StringBuffer sb = new StringBuffer();
		sb
			.append("http://localhost:")
			.append(OTAServer.getPort())
			.append("/ota/")
			.append(urlEncoded(projectName))
			.append('/')
			.append(urlEncoded(jadName));
		
		return sb.toString();
	}
*/
	/**
	 * Encode the specified information for a URL.
	 * 
	 * @param value
	 * @return
	 */
	protected String urlEncoded(String value) {
		String encoded = null;
		
		try {
			encoded = URLEncoder.encode(value, "UTF-8"); //$NON-NLS-1$
		} catch (UnsupportedEncodingException e) {
			// Should never happen
		}
		
		return encoded;
	}

	/**
	 * Add the keep-alive port to the command arguments.
	 * 
	 * @param arguments
	 * @param configuration
	 * @throws CoreException
	 */
	protected void addKeepAlivePort(ArrayList arguments, ILaunchConfiguration configuration) throws CoreException {
		// Add the keep alive port as zero, so it is ignored
		arguments.add("0"); //$NON-NLS-1$
	}

	/**
	 * Convert the specified arguments to a command-line string.
	 * 
	 * @param args
	 * @return
	 */
	protected String convertArgsToString(ArrayList args) {
		// Convert the arguments list into a string...
		StringBuffer sb = new StringBuffer();
		Iterator argsIter = args.iterator();
		while (argsIter.hasNext()) {
			String argument = (String) argsIter.next();
			sb.append(argument);
			
			if (argsIter.hasNext()) sb.append(' ');
		}
		
		return sb.toString();		
	}

	/**
	 * Add the descriptor argument.
	 * 
	 * @param arguments
	 * @param configuration
	 * @throws CoreException
	 */
	protected void addDescriptorArgument(ArrayList arguments, ILaunchConfiguration configuration, Deployment deployment) 
	throws CoreException 
	{
		if ( deployment instanceof MidletDeployment) {
			addDescriptorArgument( arguments,  configuration, (MidletDeployment)deployment);
		}
	}
	
	protected void addDescriptorArgument(ArrayList arguments, ILaunchConfiguration configuration, MidletDeployment deployment, Device device) 
		throws CoreException 
	{
		if (!shouldDoOTA(configuration, device)) {
			File jadFile = deployment.getJad();
			if (jadFile.exists()) {
				arguments.add(getQuotedString("-Xdescriptor:" +	jadFile.getAbsolutePath())); //$NON-NLS-1$
			}
		}
	}

}
