/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 
package org.eclipse.mtj.extension.devide.launching;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.ProtectionDomain;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.api.project.TargetDevice;
import org.eclipse.mtj.core.IEclipseMtjCoreConstants;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.mtj.core.launching.ILaunchConstants;
import org.eclipse.mtj.core.ui.MtjCoreUiPlugin;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.devide.Messages;
import org.eclipse.mtj.extension.devide.project.MtjProject;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

/**
 * Launch configuration tab for the selection of the
 * emulated device and any device parameters.
 */
public class DeviceTab extends AbstractLaunchConfigurationTab {
	
	// Widgets	
	private Label deviceLabel;
	private Text extraParameters;
	private Button projectDefButton;
	private Combo platformDevicesCombo;
	private Button specificDefButton;
	private Combo securityDomainCombo;
	
	// The currently selected java project
	private JavaTab javaTab;
	
	/**
	 * Default constructor
	 */
	public DeviceTab(JavaTab javaTab) {
		super();
		this.javaTab = javaTab;
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		Font font = parent.getFont();
		Composite comp = new Composite(parent, SWT.NONE);
		setControl(comp);
		
		GridLayout topLayout = new GridLayout();
		comp.setLayout(topLayout);
		
		// The platform definition specification
		Group group = new Group(comp, SWT.TITLE);
		group.setText(Messages.launchtab_device_target_device_label);
		group.setLayout(new GridLayout(2, false));
		group.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		projectDefButton = new Button(group, SWT.RADIO);
		projectDefButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				platformDevicesCombo.setEnabled(false);
				updateSecurityDomains();
				updateLaunchConfigurationDialog();
			}
		});
		Label projectDefLabel = new Label(group, SWT.NONE);
		projectDefLabel.setText(Messages.launchtab_device_project_default_device_label);
		
		specificDefButton = new Button(group, SWT.RADIO);
		specificDefButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				platformDevicesCombo.setEnabled(true);
				updateTagrgetDevices();
				updateLaunchConfigurationDialog();
			}
		});
		platformDevicesCombo = new Combo(group, SWT.DROP_DOWN | SWT.READ_ONLY);
		platformDevicesCombo.setFont(font);
		platformDevicesCombo.setEnabled(false);
		platformDevicesCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		platformDevicesCombo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				updateSecurityDomains();
				updateLaunchConfigurationDialog();
			}
		});
	
		Label securityDomainLabel = new Label(comp, SWT.NONE);
		securityDomainLabel.setText("Security Domain:"); //$NON-NLS-1$
		securityDomainLabel.setFont(font);
		
		securityDomainCombo = new Combo(comp, SWT.DROP_DOWN | SWT.READ_ONLY);
		securityDomainCombo.setFont(font);
		securityDomainCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		securityDomainCombo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				updateLaunchConfigurationDialog();
			}
		});
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#getImage()
	 */
	public Image getImage() {
		ImageDescriptor descriptor = MtjCoreUiPlugin.getIconImage("cellphone_icon.gif"); //$NON-NLS-1$
		return descriptor.createImage();
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#getName()
	 */
	public String getName() {
		return Messages.launchtab_device_target_device_label;
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#initializeFrom(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	public void initializeFrom(ILaunchConfiguration launchConfig) {
		initializePlatformDefinitionFrom(launchConfig);
		try {
			initializeDevicesFrom(launchConfig);
		} catch (MtjException e) {
			showError("Error while initializing Project's target devices", e); //$NON-NLS-1$
		}
		
		updateSecurityDomains();
		
		String domain = getStringOrDefault(
			launchConfig, 
			ILaunchConstants.SECURITY_DOMAIN, 
			ILaunchConstants.NO_SECURITY_DOMAIN);
		securityDomainCombo.setText(domain);
		
		updateLaunchConfigurationDialog();
	}

	/**
	 * Initialize the device selection from the selected launch configuration.
	 * 
	 * @param launchConfig
	 */
	private void initializeDevicesFrom(ILaunchConfiguration launchConfig) throws MtjException {
		if (specificDefButton.getSelection()) {
			platformDevicesCombo.setEnabled(true);
			updateTagrgetDevices();
			
			String deviceName = getStringOrDefault(
				launchConfig,
				ILaunchConstants.EMULATED_DEVICE_NAME,
				""); //$NON-NLS-1$
			
			platformDevicesCombo.select(platformDevicesCombo.indexOf(deviceName));
		}
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#isValid(ILaunchConfiguration)
	 */
	public boolean isValid(ILaunchConfiguration config) {
		
		setErrorMessage(null);
		setMessage(null);
		
		TargetDevice targetDevice = getSelectedTargetDevice();

		if (targetDevice == null) {
			setErrorMessage(
					Messages.launchtab_device_error_Target_device_not_selected);
			return false;
		}
				
		return true;
	}

	private IJavaProject getJavaProject() {
		return javaTab.getJavaProject();
	}
	
	private void updateTagrgetDevices() {
		platformDevicesCombo.removeAll();
		try {
			IMtjProject project = MtjProject.getMtjProject(getJavaProject());
			if ( project != null ) {
				TargetDevice defaultdevice = project.getProjectData().getDefaultTargetDevice();
				if ( defaultdevice != null ) {
					platformDevicesCombo.add(
					  defaultdevice.getName() != null ? defaultdevice.getName() :
					  "" + defaultdevice.getDeviceName() + " - " + defaultdevice.getDevicePlatformName()); //$NON-NLS-1$ //$NON-NLS-2$
				}
				Iterator it = project.getProjectData().getTargetDevices().iterator();
				while (it.hasNext()) {
					TargetDevice device = (TargetDevice)it.next();
					platformDevicesCombo.add(
							device.getName() != null ? device.getName() :
								  "" + device.getDeviceName() + " - " + device.getDevicePlatformName()); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}
		}
		catch (MtjException ex) {
			showError(Messages.DeviceTab_8, ex);
		}
	}
	
	/**
	 * Initialize the platform definition setting from the specified launch
	 * configuration.
	 * 
	 * @param launchConfig
	 * @throws CoreException
	 */
	private void initializePlatformDefinitionFrom(ILaunchConfiguration launchConfig) 
	{
		boolean useProjectPlatform = true;
		try {
			useProjectPlatform =
				launchConfig.getAttribute(ILaunchConstants.USE_PROJECT_PLATFORM_DEF, true);
		} catch (CoreException e) {
			MtjCorePlugin.log(IStatus.WARNING, Messages.DeviceTab_9, e);
		}
		projectDefButton.setSelection(useProjectPlatform);
		specificDefButton.setSelection(!useProjectPlatform);
		
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#performApply(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void performApply(ILaunchConfigurationWorkingCopy launchConfig) {
		
		launchConfig.setAttribute(
				ILaunchConstants.USE_PROJECT_PLATFORM_DEF, 
				projectDefButton.getSelection());
		
		TargetDevice device = getSelectedTargetDevice();
		if (device != null) {
				launchConfig.setAttribute(
						ILaunchConstants.EMULATION_PLATFORM_DEF,
						device.getDevicePlatformName());
				launchConfig.setAttribute(
						ILaunchConstants.EMULATED_DEVICE,
						device.getDeviceName());
				launchConfig.setAttribute(
						ILaunchConstants.EMULATED_DEVICE_NAME,
						device.getName() != null ? device.getName() :
							  "" + device.getDeviceName() + " - " + device.getDevicePlatformName()); //$NON-NLS-1$ //$NON-NLS-2$
		}
		
		launchConfig.setAttribute(
			ILaunchConstants.SECURITY_DOMAIN, 
			securityDomainCombo.getText()); 
		
	}

	private TargetDevice getSelectedTargetDevice() {
		if (projectDefButton.getSelection()) {
			TargetDevice device = null;
			try {
				IMtjProject project = MtjProject.getMtjProject(getJavaProject());
				if ( project != null ) {
					device = project.getProjectData().getDefaultTargetDevice();
				}
			}
			catch (MtjException ex) {
				showError(Messages.DeviceTab_12, ex);
			}
			return device;
		}
		else {
			if (platformDevicesCombo.getItemCount() > 0 && platformDevicesCombo.getSelectionIndex() != -1) { 
				String deviceName =
					platformDevicesCombo.getItem(platformDevicesCombo.getSelectionIndex());
				String devPlatName = ""; //$NON-NLS-1$
				String devName = ""; //$NON-NLS-1$
				if ( deviceName != null && deviceName.indexOf(" - ") != -1 ) { //$NON-NLS-1$
					devPlatName = "" + deviceName.substring(deviceName.indexOf(" - ")+3); //$NON-NLS-1$ //$NON-NLS-2$
					devName = "" + deviceName.substring(0,deviceName.indexOf(" - ")); //$NON-NLS-1$ //$NON-NLS-2$
				}
				Iterator it = null;
				try {
					IMtjProject project = MtjProject.getMtjProject(getJavaProject());
					TargetDevice defaultdevice = project.getProjectData().getDefaultTargetDevice();
					if ( defaultdevice != null && 
						 ( ( defaultdevice.getName() != null && deviceName.equals(defaultdevice.getName()) ) ||
						   ( defaultdevice.getName() == null && devPlatName.equals(defaultdevice.getDevicePlatformName()) && devName.equals(defaultdevice.getDeviceName()))
						 )
						) {
						return defaultdevice;
					}
					it = project.getProjectData().getTargetDevices().iterator();
				}
				catch (MtjException ex) {
					showError(Messages.DeviceTab_20, ex);
				}
				while (it.hasNext()) {
					TargetDevice device = (TargetDevice)it.next();
					if ( device != null && 
							 ( ( device.getName() != null && deviceName.equals(device.getName()) ) ||
							   ( device.getName() == null && devPlatName.equals(device.getDevicePlatformName()) && devName.equals(device.getDeviceName()))
							 )
							) {
							return device;
					}
				}
			}
		}
		return null;
	}
	
	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#setDefaults(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void setDefaults(ILaunchConfigurationWorkingCopy launchConfig) {
		launchConfig.setAttribute(ILaunchConstants.USE_PROJECT_PLATFORM_DEF, true);
		TargetDevice device = null;
		try {
			IJavaProject javaProject = getJavaProject();
			if ( javaProject != null ) {
				IMtjProject project = MtjProject.getMtjProject(javaProject);
				device = project.getProjectData().getDefaultTargetDevice();
			}
		}
		catch (MtjException ex) {
			showError(Messages.DeviceTab_21, ex);
		}
		if ( device != null ) {
			launchConfig.setAttribute(
					ILaunchConstants.EMULATION_PLATFORM_DEF,
					device.getDevicePlatformName());
			launchConfig.setAttribute(
				ILaunchConstants.EMULATED_DEVICE,
				device.getDeviceName());
		}
		launchConfig.setAttribute(
				ILaunchConstants.SECURITY_DOMAIN, 
				ILaunchConstants.NO_SECURITY_DOMAIN);
		launchConfig.setAttribute(ILaunchConstants.LAUNCH_PARAMS, ""); //$NON-NLS-1$
	}

	/**
	 * Set the currently selected java project.
	 * @param project
	 */
	public void setJavaProject(IJavaProject project) {
		if (project != null && specificDefButton.getSelection()) {
			updateTagrgetDevices();
			updateSecurityDomains();
			updateLaunchConfigurationDialog();
		}
	}

	/**
	 * Get a string attribute from the launch configuration or
	 * the specified default value.
	 * 
	 * @param launchConfig
	 * @param attributeName
	 * @param defaultValue
	 * @return
	 */
	private String getStringOrDefault(
		ILaunchConfiguration launchConfig,
		String attributeName,
		String defaultValue)
	{
		String value = null;
		
		try {
			value = launchConfig.getAttribute(attributeName, defaultValue);
		} catch (CoreException e) {
			MtjCorePlugin.log(IStatus.WARNING, e);
			value = defaultValue;
		}
		
		return value;
	}
	
	/**
	 * Update the security domains combo box. 
	 */
	private void updateSecurityDomains() {
		boolean enabled = false;
		
		securityDomainCombo.removeAll();
		
		TargetDevice targetdevice = getSelectedTargetDevice();
		if ( targetdevice != null ) {
			Device device = null;
			try {
				IMtjProject project = MtjProject.getMtjProject(getJavaProject());
				device = project.getDevice(targetdevice);
			}
			catch (MtjException ex) {
				showError(Messages.DeviceTab_23, ex);
			}
	
			if (device != null) {
				ArrayList list = new ArrayList();
				Iterator it = device.getProtectionDomains().iterator();
				while (it.hasNext()) {
					list.add(it.next());
				}
				
				String[] securityDomains = new String[list.size()+1];
				securityDomains[0] = ILaunchConstants.NO_SECURITY_DOMAIN;
				int i = 1;
				it = list.iterator();
				while (it.hasNext()) {
					securityDomains[i] = ((ProtectionDomain)it.next()).getName();
					i++;
				}
				
				if ( list.size() > 0 ) {
					enabled = true;
				}
				
				securityDomainCombo.setItems(securityDomains);
				securityDomainCombo.setText(ILaunchConstants.NO_SECURITY_DOMAIN);
			}
		}
		
		securityDomainCombo.setEnabled(enabled);
	}
	
	private void showError(String definition, Exception ex) {
		Status status = new Status( IStatus.ERROR, 
									IEclipseMtjCoreConstants.PLUGIN_ID, 
									-1, (ex != null ? ex.toString() : ""),  //$NON-NLS-1$
									ex);
		ErrorDialog.openError(MtjCoreUiPlugin.getActiveWindowShell(), Messages.DeviceTab_25, definition, status);

	}
}
