/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 
package org.eclipse.mtj.extension.devide.launching;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.search.IJavaSearchScope;
import org.eclipse.jdt.debug.ui.launchConfigurations.JavaMainTab;
import org.eclipse.jdt.internal.debug.ui.JDIDebugUIPlugin;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.mtj.core.launching.ILaunchConstants;
import org.eclipse.mtj.core.ui.util.ExecutableSelectionDialogCreator;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.devide.Messages;
import org.eclipse.mtj.extension.devide.nature.MtjNature;
import org.eclipse.mtj.extension.devide.project.MtjProject;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.SelectionDialog;

/**
 * Specialization of the JavaMainTab for selecting Midlets
 * for launch.
 */
public class JavaTab extends JavaMainTab {
	private Button fSearchButton;
	private Button fProjButton;
	
	/**
	 * Get a resource string.
	 * 
	 * @param key
	 * @return
	 */
	// The device tab in the launch configuration
	private DeviceTab deviceTab;
	
	// Dialog widgets
	private Label projectLabel;
	private Text executableText;

	/**
	 * Default constructor 
	 */
	public JavaTab() {
		super();
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#createControl(Composite)
	 */
	public void createControl(Composite parent) {
		Composite comp = new Composite(parent, SWT.NONE);
		setControl(comp);
		
		GridLayout topLayout = new GridLayout();
		comp.setLayout(topLayout);		
		
		createVerticalSpacer(comp, 1);
		createProjectComponents(comp);
		
		createVerticalSpacer(comp, 1);
		createExecutableComponents(comp);
		
		updateEnablement();
	}
	
	/**
	 * Get the device tab for the launch configuration.
	 * 
	 * @return
	 */
	public DeviceTab getDeviceTab() {
		return deviceTab;
	}
		
	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#getName()
	 */
	public String getName() {
		return Messages.launchtab_executable_title;
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#initializeFrom(ILaunchConfiguration)
	 */
	public void initializeFrom(ILaunchConfiguration config) {
		// this.updateMainTypeFromConfig(config);
		//super.initializeFrom(config);
		try {
			String projectName = config.getAttribute(IJavaLaunchConfigurationConstants.ATTR_PROJECT_NAME, EMPTY_STRING);	
			fProjText.setText(projectName);
		}//end try 
		catch (CoreException ce) {
			//JDIDebugUIPlugin.log(ce);
		}

		
		
		String midletName = getStringOrDefault(
			config,
			ILaunchConstants.EXECUTABLE_CLASS,
			""); //$NON-NLS-1$
		executableText.setText(midletName);
		
		updateEnablement();
	}
	
	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#isValid(ILaunchConfiguration)
	 */
	public boolean isValid(ILaunchConfiguration config) {
		
		setErrorMessage(null);
		setMessage(null);
		
		String name = fProjText.getText().trim();
		if (name.length() > 0) {
			if (!MtjCorePlugin.getDefault().getWorkspace().getRoot().getProject(name).exists()) {
				setErrorMessage(
						Messages.launchtab_executable_error_project_does_not_exist);
				return false;
			}
		}

		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(fProjText.getText().trim());
		try {
			if (!MtjNature.hasMtjNature(project)) {
					setErrorMessage(
							Messages.launchtab_executable_error_project_has_not_mtj_nature);
					return false;
			}
		} catch (CoreException e) {
			setErrorMessage(
					Messages.launchtab_executable_error_while_checking_project_nature);
			return false;
		}
		
		name = executableText.getText().trim();
		if (name.length() == 0) {
			setErrorMessage(
					Messages.launchtab_executable_error_Executable_not_specified);
			return false;
		}
				
		return true;
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#performApply(ILaunchConfigurationWorkingCopy)
	 */
	public void performApply(ILaunchConfigurationWorkingCopy config) {
		config.setAttribute(
			IJavaLaunchConfigurationConstants.ATTR_PROJECT_NAME, 
			(String)fProjText.getText());
		config.setAttribute(
			ILaunchConstants.EXECUTABLE_CLASS,
			(String)executableText.getText());
	}
			
	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#setDefaults(ILaunchConfigurationWorkingCopy)
	 */
	public void setDefaults(ILaunchConfigurationWorkingCopy config) {
		IJavaElement javaElement = getContext();
		if (javaElement != null) {
			initializeJavaProject(javaElement, config);
		} else {
			// We set empty attributes for project & main type so that when one config is
			// compared to another, the existence of empty attributes doesn't cause an
			// incorrect result (the performApply() method can result in empty values
			// for these attributes being set on a config if there is nothing in the
			// corresponding text boxes)
			config.setAttribute(
				IJavaLaunchConfigurationConstants.ATTR_PROJECT_NAME, 
				""); //$NON-NLS-1$
		}
		
		config.setAttribute(ILaunchConstants.EXECUTABLE_CLASS, ""); //$NON-NLS-1$
	}


	/**
	 * Set the device tab for the launch configuration.
	 * 
	 * @param tab
	 */
	public void setDeviceTab(DeviceTab tab) {
		deviceTab = tab;
	}

	/**
	 * Show a dialog that allows selection of Midlet
	 * subclasses.
	 */
	protected void handleSearchButtonSelected() {
		try {
			IJavaProject javaProject = getJavaProject();
			
			if ( javaProject != null ) {
				
				IMtjProject mtjProject = MtjProject.getMtjProject(javaProject);
				IJavaSearchScope searchScope = mtjProject.getJavaSearchScope(javaProject);
				
				SelectionDialog dialog = 
					ExecutableSelectionDialogCreator.createExecutableSelectionDialog(
						getShell(), 
						getLaunchConfigurationDialog(), 
						searchScope,
						false);
	
				if (dialog.open() == SelectionDialog.OK) {
					Object[] results = dialog.getResult();
					if ((results != null) && (results.length > 0)) {
						IType type = (IType) results[0];
						if (type != null) {
							executableText.setText(type.getFullyQualifiedName());
							javaProject = type.getJavaProject();
							fProjText.setText(javaProject.getElementName());
						}
					}
				}
			}
		} catch (CoreException e) {
			MtjCorePlugin.log(IStatus.ERROR, Messages.JavaTab_3, e);
		} catch (MtjException e) {
			MtjCorePlugin.log(IStatus.ERROR, Messages.JavaTab_4, e);
		}
	}

	/**
	 * Create the components that handle the executable information.
	 * 
	 * @param comp
	 */
	private void createExecutableComponents(Composite parent) {
		GridData gd;
		
		Font font = parent.getFont();
		Group group = new Group(parent, SWT.NONE);
		group.setText(Messages.JavaTab_5); // TODO I18N
		
		// Set up the layout
		GridLayout mainLayout = new GridLayout();
		mainLayout.numColumns = 2;
		mainLayout.marginHeight = 0;
		mainLayout.marginWidth = 0;
		group.setLayout(mainLayout);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		group.setLayoutData(gd);
		group.setFont(font);
		
		createExecutableComponents(group);
		
	}
	
	protected void updateMainTypeFromConfig(ILaunchConfiguration config) {
		String mainTypeName = EMPTY_STRING;
		try {
			mainTypeName = config.getAttribute(IJavaLaunchConfigurationConstants.ATTR_MAIN_TYPE_NAME, EMPTY_STRING);
		}//end try 
		catch (CoreException ce) {JDIDebugUIPlugin.log(ce);}	
		if(fMainText != null)
			fMainText.setText(mainTypeName);	
	}


	/**
	 * Create the components that make up the midlet prompting.
	 * 
	 * @param parent
	 */
	private void createExecutableComponents(Group parent) {
		GridData gd;

		Font font = parent.getFont();		
		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout mainLayout = new GridLayout();
		mainLayout.numColumns = 3;
		mainLayout.marginHeight = 0;
		mainLayout.marginWidth = 0;
		composite.setLayout(mainLayout);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		composite.setLayoutData(gd);
		composite.setFont(font);
		
		executableText = new Text(composite, SWT.SINGLE | SWT.BORDER);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		executableText.setLayoutData(gd);
		executableText.setFont(font);
		executableText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent evt) {
				updateLaunchConfigurationDialog();
			}
		});
		
		fSearchButton = createPushButton(
			composite, 
			Messages.launchtab_executable_search, 
			null);
		fSearchButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent evt) {
				handleSearchButtonSelected();
			}
		});
	}

	protected void createMainTypeEditor(Composite parent, String text, Button[] buttons) {
		Font font= parent.getFont();
		Group mainGroup= new Group(parent, SWT.NONE);
		mainGroup.setText(text);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		mainGroup.setLayoutData(gd);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		mainGroup.setLayout(layout);
		mainGroup.setFont(font);
		fMainText = new Text(mainGroup, SWT.SINGLE | SWT.BORDER);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		fMainText.setLayoutData(gd);
		fMainText.setFont(font);
		fMainText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				updateLaunchConfigurationDialog();
			}
		});
		/*fSearchButton = createPushButton(
				mainGroup, 
				Messages.getString("launchtab.executable.search"), 
				null);
		// fSearchButton = createPushButton(mainGroup, LauncherMessages.AbstractJavaMainTab_2, null); 
		fSearchButton.addSelectionListener(new SelectionListener() {
			public void widgetDefaultSelected(SelectionEvent e) {
			}
			public void widgetSelected(SelectionEvent e) {
				handleSearchButtonSelected();
			}
		});*/
		if(buttons != null) {
			for(int i = 0; i < buttons.length; i++) {
				buttons[i].setParent(mainGroup);
			}//end for
		}//end if
	}
	
	/**
	 * Create the components that handle the project information.
	 * 
	 * @param parent
	 */
	private void createProjectComponents(Composite parent) {
		GridData gd;

		Font font = parent.getFont();
		Composite projComp = new Composite(parent, SWT.NONE);
		GridLayout projLayout = new GridLayout();
		projLayout.numColumns = 2;
		projLayout.marginHeight = 0;
		projLayout.marginWidth = 0;
		projComp.setLayout(projLayout);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		projComp.setLayoutData(gd);
		projComp.setFont(font);
		
		projectLabel = new Label(projComp, SWT.NONE);
		projectLabel.setText(Messages.launchtab_executable_project);
		gd = new GridData();
		gd.horizontalSpan = 2;
		projectLabel.setLayoutData(gd);
		projectLabel.setFont(font);
		
		fProjText = new Text(projComp, SWT.SINGLE | SWT.BORDER);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		fProjText.setLayoutData(gd);
		fProjText.setFont(font);
		fProjText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent evt) {
				updateLaunchConfigurationDialog();
				deviceTab.setJavaProject(getJavaProject());
			}
		});
		fProjButton = createPushButton(
			projComp, 
			Messages.launchtab_executable_browse, 
			null);
		fProjButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent evt) {
				handleProjectButtonSelected();
			}
		});
	}

	protected void createProjectEditor(Composite parent) {
		Font font= parent.getFont();
		Group group= new Group(parent, SWT.NONE);
		group.setText(Messages.JavaTab_Main); 
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		group.setLayoutData(gd);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		group.setLayout(layout);
		group.setFont(font);
		fProjText = new Text(group, SWT.SINGLE | SWT.BORDER);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		fProjText.setLayoutData(gd);
		fProjText.setFont(font);
		fProjText.addModifyListener(getDefaultListener());

		/*
		fProjButton = createPushButton(
				group, 
				Messages.getString("launchtab.executable.browse"), 
				null);
		fProjButton.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent evt) {
					handleProjectButtonSelected();
				}
		});
		*/
		//fProjButton = createPushButton(group, LauncherMessages.AbstractJavaMainTab_1, null); 
		//fProjButton.addSelectionListener(fListener);
	}	

	
	public IJavaProject getJavaProject() {
		if ( fProjText == null || fProjText.getText() == null || fProjText.getText().equals(Messages.JavaTab_6) ) {
			return null;
		}
		return super.getJavaProject();
	}
	
	/**
	 * Get a string attribute from the launch configuration or
	 * the specified default value.
	 * 
	 * @param launchConfig
	 * @param attributeName
	 * @param defaultValue
	 * @return
	 */
	private String getStringOrDefault(
		ILaunchConfiguration launchConfig,
		String attributeName,
		String defaultValue)
	{
		String value = null;
		
		try {
			value = launchConfig.getAttribute(attributeName, defaultValue);
		} catch (CoreException e) {
			MtjCorePlugin.log(IStatus.WARNING, e);
			value = defaultValue;
		}
		
		return value;
	}
	
	/**
	 * Update the enablement of controls based on the status of the
	 * other controls.
	 */
	private void updateEnablement() {
		executableText.setEnabled(true);
		fSearchButton.setEnabled(true);
	}
}
