/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM - initial API and implementation
 *******************************************************************************/
package org.eclipse.mtj.extension.smgmt.ui.eclipse.preferences;

import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.window.Window;
import org.eclipse.mtj.api.extension.MtjExtension;
import org.eclipse.mtj.api.extension.SecurityManagement;
import org.eclipse.mtj.core.ui.util.GridUtil;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.smgmt.ui.SecurityManagmentUIMessages;
import org.eclipse.mtj.extension.smgmt.ui.utils.SmgmtUtils;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;


/**
 * This class represents a preference page that
 * is contributed to the Preferences dialog. By 
 * subclassing <samp>FieldEditorPreferencePage</samp>, we
 * can use the field support built into JFace that allows
 * us to create a page that is small and knows how to 
 * save, restore and apply itself.
 * <p>
 * This page is used to modify preferences only. They
 * are stored in the preference store that belongs to
 * the main plug-in class. That way, preferences can
 * be accessed directly via the preference store.
 */

public class SecurityManagementPreferencePage
	extends PreferencePage
	implements IWorkbenchPreferencePage {
	
	private Button editButton;
	
	private Table SecurityToolPathTable;

	private final static int SMgmtDescriptionIndex = 0;
	private final static int SMgmtVendorNameIndex = 1;
	private final static int SMgmtVersionIndex = 2;
	private final static int SMgmtLocationIndex = 3;

	private MtjExtension[] securityList;
	private TableItem lastEnabled = null;

	public SecurityManagementPreferencePage() {
		super();
		
		setDescription(SecurityManagmentUIMessages.SecurityManagementPreferencePage_0 + 
				SecurityManagmentUIMessages.SecurityManagementPreferencePage_1 + 
				SecurityManagmentUIMessages.SecurityManagementPreferencePage_2); 
		
		noDefaultAndApplyButton();
		try {
			securityList = (SecurityManagement[])SmgmtUtils.getSecurityManagementProvidersList();
		} catch (MtjException e) {
			securityList = null;
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.IWorkbenchPreferencePage#init(org.eclipse.ui.IWorkbench)
	 */
	public void init(IWorkbench workbench) {
	}

	@Override
	protected Control createContents(final Composite parent) {
		
		if (securityList == null){
			setErrorMessage(SecurityManagmentUIMessages.SecurityManagementPreferencePage_10);
		} else if (securityList.length <= 0){
			setErrorMessage(SecurityManagmentUIMessages.SecurityManagementPreferencePage_6); 
		}

		Composite mainPageComposite = new Composite(parent, SWT.BORDER);
		
		GridLayout c1gridLayout = new GridLayout();
		c1gridLayout.numColumns = 2;
		
		mainPageComposite.setLayout(c1gridLayout);
		GridData dataMain = GridUtil.createFill();
        mainPageComposite.setLayoutData(dataMain);
		
		Composite tableComposite = new Composite(mainPageComposite, SWT.NULL);
        FillLayout fillLayout = new FillLayout();
        tableComposite.setLayout(fillLayout);
        GridData gdTableComp = new GridData(GridData.FILL_BOTH);
        tableComposite.setLayoutData(gdTableComp);
		
		SecurityToolPathTable = new Table(tableComposite, SWT.BORDER | SWT.CHECK | SWT.SINGLE |SWT.H_SCROLL |SWT.V_SCROLL| SWT.FULL_SELECTION );

		SecurityToolPathTable.setHeaderVisible(true);
		SecurityToolPathTable.setLinesVisible(true);
		SecurityToolPathTable.setFont(parent.getFont());
		
		TableColumn SMgmgtDescrition = new TableColumn(SecurityToolPathTable, SWT.LEFT);
		SMgmgtDescrition.setText(SecurityManagmentUIMessages.SecurityManagementPreferencePage_9); 
		
		TableColumn SMgmtVendorName = new TableColumn(SecurityToolPathTable, SWT.LEFT);
		SMgmtVendorName.setText(SecurityManagmentUIMessages.SecurityManagementPreferencePage_3); 

		TableColumn SMgmtVersion = new TableColumn(SecurityToolPathTable, SWT.LEFT);
		SMgmtVersion.setText(SecurityManagmentUIMessages.SecurityManagementPreferencePage_5); 
		
		TableColumn SMgmtLocation = new TableColumn(SecurityToolPathTable, SWT.LEFT);
		SMgmtLocation.setText(SecurityManagmentUIMessages.SecurityManagementPreferencePage_4); 

		//Don't try to populate if no Security Managers found
		if (securityList != null && securityList.length > 0){
			for(int i = 0; i < securityList.length; i++){
				TableItem item = new TableItem(SecurityToolPathTable,SWT.NONE);
				
				item.setText(SMgmtDescriptionIndex, securityList[i].getDescription());
				item.setText(SMgmtVendorNameIndex, securityList[i].getVendor());
				item.setText(SMgmtVersionIndex, securityList[i].getVersion());
				try {
					item.setText(SMgmtLocationIndex, ((SecurityManagement)securityList[i]).getToolLocation(null));
				} catch (MtjException e1) {
					item.setText(SMgmtLocationIndex, ""); //$NON-NLS-1$
				}
	
					if (((SecurityManagement)securityList[i]).isActive())
					{
						if (lastEnabled != null){
							if (lastEnabled.equals(item))
								item.setChecked(true);
							else
								item.setChecked(false);
						}
						else{
							item.setChecked(true);
							lastEnabled = item;						
						}
						
					} else {
						item.setChecked(false);
					}			
			}
		}
		SMgmgtDescrition.pack();
		SMgmtVendorName.pack();
		SMgmtVersion.pack();
		SMgmtLocation.pack();
		SecurityToolPathTable.pack();

		SecurityToolPathTable.addListener(SWT.Selection, new Listener() {
		      public void handleEvent(Event event) {
		    	  if (SecurityToolPathTable.getSelectionCount() > 0){
		    		  editButton.setEnabled(true);		
		    	  }
		    	  
		    	  if (event.detail == SWT.CHECK)
		    	  {
		    		  if (lastEnabled == null){
		    			  ((TableItem)event.item).setChecked(true);
		    			  lastEnabled = (TableItem)event.item;
		    		  }
		    		  else if (!lastEnabled.equals((TableItem)event.item)){
		    			  lastEnabled.setChecked(false);
		    			  ((TableItem)event.item).setChecked(true);
		    			  lastEnabled = (TableItem)event.item;
		    		  }
		    	  }
				return;
	      }
		    });
      
		Composite rightColumnButtons = new Composite(mainPageComposite, SWT.BORDER);
		GridData gd = new GridData();
		gd.grabExcessVerticalSpace = true;
		gd.horizontalIndent = 10;
		gd.verticalAlignment = GridData.FILL;
		
        rightColumnButtons.setLayoutData(gd);
		gd.horizontalSpan = 1;

		GridLayout c1gridLayoutButtons = new GridLayout();
		c1gridLayoutButtons.numColumns = 1;
		rightColumnButtons.setLayout(c1gridLayoutButtons);
		
		editButton =  new Button(rightColumnButtons, SWT.PUSH);
		editButton.setText(SecurityManagmentUIMessages.SecurityManagementPreferencePage_8); 
		editButton.setEnabled(false);
		editButton.addSelectionListener(new SelectionAdapter (){
			public void widgetSelected(SelectionEvent e) {
				TableItem ti = SecurityToolPathTable.getSelection()[0];
				SecurityManagementEditor SME = new SecurityManagementEditor(getShell(),ti.getText(SMgmtVendorNameIndex), ti.getText(SMgmtVersionIndex), ti.getText(SMgmtLocationIndex), ti.getText(SMgmtDescriptionIndex));
				SME.open();
				if(SME.getReturnCode() == Window.OK){
					ti.setText(SMgmtLocationIndex, SME.getPath());	
				}
			}
		});		
		return mainPageComposite;
	}
	

//	@Override
	public boolean performOk() {
		
		boolean okToStore = false; //SmgmtUtils.storeSecurityManagers(SecurityToolPathTable.get);
		
		for(int i = 0; i < securityList.length; i++){
		try {
			((SecurityManagement)securityList[i]).storeToolLocation(SecurityToolPathTable.getItem(i).getText(SMgmtLocationIndex), null);
			((SecurityManagement)securityList[i]).setActive(SecurityToolPathTable.getItem(i).getChecked());
		} catch (MtjException e) {
			setErrorMessage(SecurityManagmentUIMessages.SecurityManagementPreferencePage_7); 
			e.printStackTrace();
		}
	}
		okToStore = SmgmtUtils.storeSecurityManagers(securityList);

		if (okToStore) {
			return super.performOk();
		} else {
			return okToStore;
		}
	}
}