/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM - initial API and implementation
 *******************************************************************************/
package org.eclipse.mtj.extension.sp.wtk.impl;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.mtj.api.deployment.Deployment;
import org.eclipse.mtj.api.enumerations.DeploymentType;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.extension.SigningProvider;
import org.eclipse.mtj.api.extension.impl.MtjExtensionImpl;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.api.model.MidletDeployment;
import org.eclipse.mtj.api.signings.SigningCertificate;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.sp.wtk.SigningProviderPlugin;
import org.eclipse.mtj.jad.util.ConvertNativeToUTF8;
import org.eclipse.mtj.jad.util.JadFileDescriptor;

public class SigningProviderImpl extends MtjExtensionImpl implements SigningProvider{
	
	private IPreferenceStore sigingProviderPrefStore;
	private SigningCertificate targetSigningCertificate;
	
	public SigningProviderImpl() {
		super();
		setId(SigningProviderPlugin.getDefault().getBundle().getSymbolicName());
		setVendor("Eclipse.org"); //$NON-NLS-1$
		setVersion("1.0.0"); //$NON-NLS-1$
		setDescription(Messages.SigningProviderImpl_12); 
		setType(ExtensionType.SIGNING_PROVIDER_LITERAL);
		sigingProviderPrefStore = SigningProviderPlugin.getDefault().getPreferenceStore();
	}
	/**
	 * Method returns the deployment types that are supported by itself.
	 * 
	 * @return
	 */
	public DeploymentType[] getSupportedTypes() throws MtjException {
		// This Signing provider only supports MIDP 
		DeploymentType [] supportedDeploymentTypes = {DeploymentType.DEPLOYMENT_TYPE_MIDLET_LITERAL};
		return supportedDeploymentTypes;
	}
	private boolean isSupported(Deployment deployment) throws MtjException
	{
		boolean supported = false;
//		DeploymentType [] typesSupported = getSupportedTypes();
//		for (int i = 0; i < typesSupported.length; i++)
//		{
//			if (deployment.getType() == typesSupported[i])
//				supported = true;
//		}
		if (deployment instanceof MidletDeployment)
			supported = true;
		
		return supported;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.mtj.api.extension.SigningProvider#sign(org.eclipse.mtj.api.model.IMtjProject, org.eclipse.core.resources.IFolder, org.eclipse.mtj.api.signings.SigningCertificate, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public boolean sign(IMtjProject project, IFolder targetFolder,
			SigningCertificate certificates, IProgressMonitor monitor) throws MtjException {
		boolean success = false;
		targetSigningCertificate = certificates;
		Deployment deployment;
		try {
			deployment = project.getDeployment();
			monitor.beginTask(Messages.SigningProviderImpl_0,100); 
			//if (deployment instanceof MidletDeployment) {
			if (isSupported(deployment)){
								
				String jadNameLoc = ((MidletDeployment)deployment).getJad().getAbsolutePath();
				String jarNameLoc = ((MidletDeployment)deployment).getJar().getAbsolutePath();
				
				//Clear temp files from possible prior failed sign requests
				clearTempFiles(jadNameLoc);

				success = signMIDPApplication(jadNameLoc,jarNameLoc, monitor);	
				IFile JadFile = getProjectjadIFile(project);
				if (JadFile != null)
					JadFile.refreshLocal(IResource.DEPTH_ZERO, monitor);
			}
			else{
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_DEPLOYMENT_NOT_SUPPORTED) + " " + Messages.SigningProviderImpl_7);  //$NON-NLS-1$
			}
		} catch (CoreException e) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), e);
		}
		
		monitor.done();
		return success;
	}
	private boolean clearTempFiles(String jadNameLoc) {
		boolean success = false;
		File tempfile = new File(jadNameLoc+ SigningProviderImplConstants.JADTOOL_TEMP_FILE_POSTFIX);
		
		if ( (tempfile.exists() && tempfile.canWrite()) ){
			success = tempfile.delete();
		}	
		return success;		
	}
	
	public String getExtJar() {
		// TODO Auto-generated method stub
		return null;
	}
	public void setExtJar(String value) {
		// TODO Auto-generated method stub
		
	}
	/* (non-Javadoc)
	 * @see org.eclipse.mtj.api.extension.SigningProvider#unsign(org.eclipse.mtj.api.model.IMtjProject, org.eclipse.core.resources.IFolder, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public boolean unsign(IMtjProject project, IFolder targetFolder, IProgressMonitor monitor) throws MtjException {
		JadFileDescriptor jadDescriptor;
		
		monitor.beginTask(Messages.SigningProviderImpl_8, 100); 
		
		IFile JadFile = getProjectjadIFile(project);
	
			try {

				if (isSupported(project.getDeployment())) {
					if (JadFile != null) {
						jadDescriptor = JadFileDescriptor.getDescriptor(JadFile);
						monitor.worked(20);
						jadDescriptor.removeProperty(JadFileDescriptor.S_JAR_RSA_SHA1);
						jadDescriptor.removeProperty(JadFileDescriptor.S_CERTIFICATE1_1);
						monitor.worked(20);
						//run the descriptor content through a native->latin-1+escapes filter
						JadFile.setContents(ConvertNativeToUTF8.convertNativeToUTF8(jadDescriptor.serialize()), true,true, null);
						monitor.worked(60);
						
						JadFile.refreshLocal(IResource.DEPTH_ZERO, monitor);
					}
					else
						throw new MtjException(Messages.SigningProviderImpl_9); 
				}
				else 
				{
					throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_DEPLOYMENT_NOT_SUPPORTED) + " " + Messages.SigningProviderImpl_7);  //$NON-NLS-1$
				}
			} catch (CoreException e2) {
				monitor.done();
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), e2);
			} catch (IOException e) {
				monitor.done();
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), e);
			}
		
		monitor.done();
		return true;
	}
	
	/*
	 * 
	 * 
	 */
	private boolean signMIDPApplication( String jadFile, String jarFile, IProgressMonitor monitor) throws MtjException {
		boolean success = false;
		monitor.worked(10);

		if(addSignature(jadFile, jarFile))
		{
			success = true;
			monitor.worked(20);
		} else{
			monitor.done();
			return false;
		}
		
		if (addCertificate(jadFile))
		{
			success = true;
			monitor.worked(20);
		}

		return success;
	}	

	/*
	 * 
	 * 
	 */
	private boolean addSignature( String jadFile, String jarFile) throws MtjException {
		
		boolean cmdSuccessful = true;
		String cmdArgs[] = generateSignMIDPApplicationCmd(jadFile, jarFile);
		Process p = runSPCmd(cmdArgs);

		BufferedReader cmdErrorStream = new BufferedReader(new InputStreamReader(p.getErrorStream()));

		String cmdError=""; //$NON-NLS-1$
		try {
			if ((cmdError=cmdErrorStream.readLine()) != null) {
				while (cmdError.equals("")){ //$NON-NLS-1$
					cmdError=cmdErrorStream.readLine();
				}		
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + " : " + cmdError); //$NON-NLS-1$
			}
			// if no errors, copy temp output jad, back to  original input jad 
			else
			{
				cmdSuccessful = overWriteOriginalFile(jadFile);
			}

		} catch (IOException ee) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), ee);
		} 
		
		return cmdSuccessful;
	}
	
	/*
	 * 
	 * 
	 */
	private String[] generateSignMIDPApplicationCmd( String jadFile, String jarFile) throws MtjException {

		String[] signMIDPApplicationCmdArgs = {
				SigningProviderImplConstants.JAVA, SigningProviderImplConstants.JAR,
				getSigningTool(),
				SigningProviderImplConstants.JADTOOL_ADD_SIG,
				SigningProviderImplConstants.JADTOOL_JAR_FILE, jarFile,
				SigningProviderImplConstants.JADTOOL_KEY_PASS, targetSigningCertificate.getKeystorePassword(),
				SigningProviderImplConstants.JADTOOL_ALIAS, targetSigningCertificate.getAliasName(),
				SigningProviderImplConstants.JADTOOL_KEYSTORE, targetSigningCertificate.getKeystore(),
				SigningProviderImplConstants.JADTOOL_ENCODING, System.getProperty("file.encoding"), //$NON-NLS-1$ 
				SigningProviderImplConstants.JADTOOL_IN_JAD, jadFile,
				// Output jad should be a temp file to protect the original 
				// jad file in case of errors during the signing process
				// Jadtool.jar clears jad out if errors occurred
				SigningProviderImplConstants.JADTOOL_OUT_JAD, jadFile + SigningProviderImplConstants.JADTOOL_TEMP_FILE_POSTFIX,
				};	
		return signMIDPApplicationCmdArgs;
	}
	
	
	/**
	 * Execute the array of commands passed in
	 * @param cmd
	 * @return the process generated by the execution of the commads
	 * @throws MtjException 
	 */
	private Process runSPCmd (String[] cmd) throws MtjException {
		
		Process p = null;

		try {p =	Runtime.getRuntime().exec(cmd);} catch (IOException e) {	}
		if (p == null)
		 {
			StringBuffer str = new StringBuffer(""); //$NON-NLS-1$
			
			for (int i = 0; i < cmd.length; i++)
			{
				str.append(" " + cmd[i]); //$NON-NLS-1$
			}
		
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + " could not execute: " + "[" + str +"]");   //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
		}
		return p;	
	}
	
	
	/**
	 * This Method is used to overwrite an original file
	 * with a temp version of the file
	 * for example: when signing a jad file, you can create a temporary jad file
	 * with the signature and certificate, if there are no errors in the process,
	 * then overwrite the original (unsigned) jad file, with the signed version of the 
	 * file
	 * This method assumes that the temp file name convention is: original name + "temp"
	 * @param fileName original file name,  
	 * @return boolean success status
	 * @throws MtjException 
	 */
	private boolean overWriteOriginalFile(String fileName) throws MtjException{
		boolean success = false;
		File originalFile = new File(fileName);
		File tempfile = new File(fileName+ SigningProviderImplConstants.JADTOOL_TEMP_FILE_POSTFIX);
		
		if ( (originalFile.exists() && originalFile.canWrite()) && (tempfile.exists() && tempfile.canWrite())){
			
			success = originalFile.delete();
			if(!success)
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + " " + Messages.SigningProviderImpl_3 + fileName);  //$NON-NLS-1$
			success = tempfile.renameTo(originalFile);	
			if(!success)
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + " " + Messages.SigningProviderImpl_4 + fileName + Messages.SigningProviderImpl_5 + fileName + SigningProviderImplConstants.JADTOOL_TEMP_FILE_POSTFIX);  //$NON-NLS-1$
		}	
		return success;
	}	
	
    /*
	 * Get the signing tool location.
	 * 
	 */
	private final String getSigningTool() throws MtjException {
		String signingToolLocation = getToolLocation(null);

		if (signingToolLocation == null) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + " " + Messages.SigningProviderImpl_2);   //$NON-NLS-1$
		}
		
		StringBuffer buffer = new StringBuffer();
		buffer.append(signingToolLocation)
		.append(File.separator)
		.append("bin") //$NON-NLS-1$						
		.append(File.separator)
		.append("JadTool.jar"); //$NON-NLS-1$
			
		return buffer.toString();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.mtj.api.extension.SigningProvider#isSigned(org.eclipse.mtj.api.deployment.Deployment, org.eclipse.core.resources.IFolder, org.eclipse.core.resources.IFile, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public boolean isSigned(IMtjProject mtjProject, IFolder targetFolder, IProgressMonitor monitor) throws MtjException {
		boolean isSigned = false;
		
		IFile JadFile = getProjectjadIFile(mtjProject);
		JadFileDescriptor jadDescriptor;
		try 
		{
			if (JadFile != null) {
			    jadDescriptor = JadFileDescriptor.getDescriptor(JadFile);
			    // check if the JAD file for the MIDlet build is signed
			    if ((jadDescriptor.getProperty(JadFileDescriptor.JAR_RSA_SHA1)!=null)||(jadDescriptor.getProperty(JadFileDescriptor.CERTIFICATE1)!=null)){
			    	isSigned = true;
			    }			    		    
			}//(jadFile!=null)
			else
				throw new MtjException(Messages.SigningProviderImpl_9); 
		} catch (CoreException e2) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), e2);
		} catch (IOException e) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), e);
		}

		return isSigned;
	}
	private IFile getProjectjadIFile(IMtjProject project) throws MtjException {	
		IFile jadIFile = null;
		Deployment dp = null;
		try {
			dp = project.getDeployment();
		} catch (CoreException e) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.SigningProviderImpl_10, e); 
		}
		if(dp.getType() == DeploymentType.DEPLOYMENT_TYPE_MIDLET_LITERAL)
		{
			try {
				jadIFile = project.getDeploymentFolder(null).getFile(((MidletDeployment)dp).getJad().getName());
			} catch (CoreException e) {
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.SigningProviderImpl_11, e); 
			}
		}
		
		return jadIFile;
	}

	/*
	 * 
	 * 
	 */
	private boolean addCertificate( String jadFile) throws MtjException {
		boolean cmdSuccessful = true;
		String cmdArgs[] = generateCertMIDPApplicationCmd(jadFile);
		Process p = runSPCmd(cmdArgs);
		
		BufferedReader cmdErrorStream = new BufferedReader(new InputStreamReader(p.getErrorStream()));

		String cmdError=""; //$NON-NLS-1$
		try {
			if ((cmdError=cmdErrorStream.readLine()) != null) {
				while (cmdError.equals("")){ //$NON-NLS-1$
					cmdError=cmdErrorStream.readLine();
				}
				
				if (cmdError.toLowerCase().indexOf("keystore could not be loaded")>=0) {   //$NON-NLS-1$
					throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_BAD_KEYSTORE_OR_PASSWORD) + " " + cmdError); //$NON-NLS-1$
				}
				else
					throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + " : " + cmdError); //$NON-NLS-1$
			}
			// if no errors, copy temp output jad, back to  original input jad
			else
			{
				cmdSuccessful = overWriteOriginalFile(jadFile);
			}
		} catch (IOException ee) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), ee);
		} 
		return cmdSuccessful;
	}
	
	/*
	 * 
	 * 
	 */
	private String[] generateCertMIDPApplicationCmd( String jadFile) throws MtjException {

		String[] signMIDPApplicationCmdArgs = {
				SigningProviderImplConstants.JAVA, SigningProviderImplConstants.JAR,
				getSigningTool(),
				SigningProviderImplConstants.JADTOOL_ADD_CERT,
				SigningProviderImplConstants.JADTOOL_ALIAS, targetSigningCertificate.getAliasName(),
				SigningProviderImplConstants.JADTOOL_KEYSTORE, targetSigningCertificate.getKeystore(),
				SigningProviderImplConstants.JADTOOL_ENCODING, System.getProperty("file.encoding"), //$NON-NLS-1$
				SigningProviderImplConstants.JADTOOL_IN_JAD, jadFile,
				// Output jad should be a temp file to protect the original 
				// jad file in case of errors during the signing process
				// Jadtool.jar clears jad out if errors occurred
				SigningProviderImplConstants.JADTOOL_OUT_JAD, jadFile + SigningProviderImplConstants.JADTOOL_TEMP_FILE_POSTFIX
				};
		
		return signMIDPApplicationCmdArgs;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.mtj.api.extension.SigningProvider#getToolLocation(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public String getToolLocation(IProgressMonitor monitor) throws MtjException
	{
		return sigingProviderPrefStore.getString(SigningProviderImplConstants.SIGNING_TOOL_LOCATION);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.mtj.api.extension.SigningProvider#storeToolLocation(java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void storeToolLocation(String loc, IProgressMonitor monitor) throws MtjException{
		sigingProviderPrefStore.setValue(SigningProviderImplConstants.SIGNING_TOOL_LOCATION, loc);
	}
	
public String [] listrootcert(String rootcert) throws MtjException{
		
		String[] cmdArgs = null;
		//ArrayList aliases = new ArrayList();
		List<String> aliases = new ArrayList<String>();
		
		try {
			cmdArgs = generateOpenRootCertCmd(rootcert);
			Process p = runSPCmd(cmdArgs);
			
			BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getInputStream()));

			String cmdOutput;
			
				while ((cmdOutput=cmdOutputStream.readLine()) != null) {
					
					if (cmdOutput.indexOf("Key ")>=0) { //$NON-NLS-1$
						StringBuffer cmds = new StringBuffer(cmdOutput);
						for (int i = 0; i <= 2; i++){
							cmds.append(" "); //$NON-NLS-1$
							if ((cmdOutput=cmdOutputStream.readLine()) != null)
								cmds.append(cmdOutput);
						}
						aliases.add(cmds.toString());
					
					} else if (cmdOutput.toLowerCase().indexOf("error")>=0) {  //$NON-NLS-1$
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + " " + cmdOutput); //$NON-NLS-1$
					} 
				}
			} catch (IOException ee) {
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), ee);
			}
		
		return (String[])aliases.toArray(new String[aliases.size()]);
	}
	private String[] generateOpenRootCertCmd(String rootcert) throws MtjException {
		String[] newKeyCmdArgs = {
			SigningProviderImplConstants.JAVA, SigningProviderImplConstants.JAR,
			getMESecurityManagementTool(),
			SigningProviderImplConstants.MEKEYTOOL_LIST,
			SigningProviderImplConstants.MEKEYTOOL_MEKEYSTORE,
			rootcert			
			};		
		return newKeyCmdArgs;
	}

	public void importToRootCert(String rootcert, String keystore, String alias, String password) throws MtjException {
		String[] cmdArgs = null;
			
		cmdArgs = generateImportToRootCertCmd(rootcert, keystore, alias, password);
		Process p = runSPCmd(cmdArgs);
		
		BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getInputStream()));

		String cmdOutput;
		try {
			while ((cmdOutput=cmdOutputStream.readLine()) != null) {
						
				if (cmdOutput.length() !=0) { 

					if (cmdOutput.toLowerCase().indexOf("Owner already has this key in the ME keystore")>=0) {   //$NON-NLS-1$
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SECURITY_ALIAS_DUPLICATE) + " " +cmdOutput); //$NON-NLS-1$
					} 
					else {
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) + " " + cmdOutput); //$NON-NLS-1$
					}
				} 
			}
		} catch (IOException ee) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY), ee);
		}
	}
	private String[] generateImportToRootCertCmd(String rootcert, String keystore, String alias, String password) throws MtjException {
		String[] importCertificateCmdArgs = 
		{	SigningProviderImplConstants.JAVA, SigningProviderImplConstants.JAR,
			getMESecurityManagementTool(),	
			SigningProviderImplConstants.MEKEYTOOL_IMPORT,
			SigningProviderImplConstants.MEKEYTOOL_MEKEYSTORE, rootcert,	
			SigningProviderImplConstants.MEKEYTOOL_KEYSTORE, keystore,
			SigningProviderImplConstants.MEKEYTOOL_STOREPASS, password,
			SigningProviderImplConstants.MEKEYTOOL_ALIAS, alias
		};		
		return importCertificateCmdArgs;
	}

	public boolean isRootCertToolAvailable() throws MtjException {
		return (getMESecurityManagementTool() != null) ? true : false;
	}
	private String getMESecurityManagementTool() throws MtjException {
		String securityToolLocation = getToolLocation(null);
		
		if (securityToolLocation == null ||
			securityToolLocation.length() <= 0 ||
			securityToolLocation.equals(Messages.PreferenceInitializer_0)	 
			){
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_PROVIDER_NOT_CONFIGURED) + "\n" + getId() + Messages.SigningProviderImpl_19); //$NON-NLS-1$
			}
		
		StringBuffer buffer = new StringBuffer();
		buffer.append(securityToolLocation)
		.append(File.separator)
		.append("bin") //$NON-NLS-1$						
		.append(File.separator)
		.append("MEKeyTool.jar"); //$NON-NLS-1$
			
		return buffer.toString();
	}
	public void removeCertFromRoot(String rootcert, int removeindex) throws MtjException {
		String[] cmdArgs = null;
		
		cmdArgs = generateDeleteFromRootCertCmd(rootcert, removeindex);
		Process p = runSPCmd(cmdArgs);
		
		BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getInputStream()));

		String cmdOutput;
		
		try {
			while ((cmdOutput=cmdOutputStream.readLine()) != null) {						
				if (cmdOutput.length() !=0) { 
					throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) + " " + cmdOutput); //$NON-NLS-1$
				} 
			}
		} catch (IOException ee) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY), ee);
		}
	}
	private String[] generateDeleteFromRootCertCmd(String rootcert, int removeindex) throws MtjException {
		String[] removeCertificateCmdArgs = 
		{	SigningProviderImplConstants.JAVA, SigningProviderImplConstants.JAR,
			getMESecurityManagementTool(),	
			SigningProviderImplConstants.MEKEYTOOL_DELETE,
			SigningProviderImplConstants.MEKEYTOOL_MEKEYSTORE, rootcert,	
			SigningProviderImplConstants.MEKEYTOOL_NUMBER, Integer.toString(removeindex)
		};
			
		return removeCertificateCmdArgs;
	}
}
