/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/

package org.eclipse.mtj.extension.dplp.nokia.pcsuite.impl;

import java.io.File;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.mtj.extension.dplp.nokia.pcsuite.PCSuiteMessages;
import org.eclipse.mtj.extension.dplp.nokia.pcsuite.util.StringManipulator;

import com.nokia.phone.deploy.CONA;
import com.nokia.phone.deploy.event.ConnectionLayerEvent;
import com.nokia.phone.deploy.event.TerminalEvent;
import com.nokia.phone.deploy.event.TerminalListener;


public class Deployer {

	private transient TerminalObserver terminalObserver = null;

	private transient boolean connected = false;

	private transient boolean updateTerminalList = false;

	private transient Vector<TerminalListener> terminalListeners = null;

	private static final int TERMINAL_CONNECTED = 0;

	private static final int TERMINAL_DISCONNECTED = 1;

	private int comPort = 1;
	
	private static boolean OSSupportsDeployment = true;
	
	private boolean deploying_ = false;
	
	public static void main(String[] args) {
		// Don't continue if dll's were not found
		if (!CONA.getInstance().getDllFound()) {
			System.out.println("ConnAPI.dll not found"); //$NON-NLS-1$
			System.exit(0);
		}
		
		Deployer deployer = new Deployer();
		
		// Open connection, if it doesn't work out, don't continue
		deployer.openConnectionLayer();
		if (!deployer.connected) {
			System.out.println("connection failed"); //$NON-NLS-1$
			System.exit(0);
		}
		
		String[] terminals = deployer.getTerminals();
		// If there are no terminals, don't continue
		if (terminals.length == 0) {
			System.out.println("no devices found"); //$NON-NLS-1$
			deployer.closeConnectionLayer();
			System.exit(0);
		}
		// Print out all terminals
		for (int i = 0; i < terminals.length; i++) {
			System.out.println("device found " + terminals[i] //$NON-NLS-1$
					+ " using " //$NON-NLS-1$
					+ deployer.getConnectionType(terminals[i]));
			try {
				deployer.deploy(new File("c:\\runtime-workspace\\test\\deployed\\Boids.jad"), new File("c:\\runtime-workspace\\test\\deployed\\Boids.jar"), terminals[i]); //$NON-NLS-1$ //$NON-NLS-2$
			} catch (UnsatisfiedLinkError e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

	}

	/**
	 * Try to load the native library if running in a Windows environment.
	 */
	static {
		if (System.getProperty("os.name").toLowerCase().startsWith("linux")) { //$NON-NLS-1$ //$NON-NLS-2$
			OSSupportsDeployment = false;
		}
	}
	
	public Deployer(){
		this.initialize();
	}

	private void initialize() {
		if (terminalListeners == null) {
			terminalListeners = new Vector<TerminalListener>();
		}
		connected = false;
		updateTerminalList = false;
		
		// Open connection
		openConnectionLayer();

		if (!connected) {
			System.out.println("connection failed"); //$NON-NLS-1$
		}
	}
	
	/**
	 * Adds a terminal listener which will be notified when a terminal is
	 * connected to or disconnected from the computer. To open the connection
	 * layer, you must add a terminal listener.
	 * 
	 * @param terminalListener
	 *            listener to add
	 */
	public void addTerminalListener(TerminalListener terminalListener) {
		terminalListeners.add(terminalListener);
		// Inform the new terminal listener about current status
		if (!OSSupportsDeployment) {
			terminalListener
					.connectionLayerEventOccurred(new ConnectionLayerEvent(
							this,
							ConnectionLayerEvent.DEPLOYMENT_NOT_SUPPORTED_ON_THIS_OPERATING_SYSTEM,
							PCSuiteMessages.Deployer_10));
		} else if (!CONA.getInstance().getDllFound()) {
			terminalListener
					.connectionLayerEventOccurred(new ConnectionLayerEvent(
							this,
							ConnectionLayerEvent.FAILED_TO_OPEN_NDSJ2METOOBEX_DLL,
							PCSuiteMessages.Deployer_11));
		} else if (terminalListeners.size() == 1 && !connected
				&& OSSupportsDeployment) {
			openConnectionLayer();
		}

		if (!connected) {
			terminalListener
					.connectionLayerEventOccurred(new ConnectionLayerEvent(
							this,
							ConnectionLayerEvent.FAILED_TO_OPEN_SERVICE_LAYER,
							PCSuiteMessages.Deployer_12));
		} else {
			terminalListener
					.connectionLayerEventOccurred(new ConnectionLayerEvent(
							this, ConnectionLayerEvent.SERVICE_LAYER_OPENED,
							PCSuiteMessages.Deployer_13));
		}
	}

	/**
	 * Removes all TerminalListeners and closes communications with the
	 * connection layer.
	 */
	public void shutdown() {
		while (terminalListeners.size() > 0) {
			terminalListeners.remove(0);
		}
		closeConnectionLayer();
	}

	/**
	 * Removes a terminal listener.
	 * 
	 * @param terminalListener
	 *            listener to remove
	 */
	public void removeTerminalListener(TerminalListener terminalListener) {
		terminalListeners.remove(terminalListener);
		if (terminalListeners.size() == 0 && connected) {
			closeConnectionLayer();
		}
	}

	/**
	 * Opens the platform connection layer using native method.
	 */
	protected void openConnectionLayer() {
		if (CONA.getInstance().getDllFound() && !connected) {
			// connected = connectServiceLayer( comPort );
			connected = CONA.getInstance().connect();
			if (connected) {
				updateTerminalList = true;
				terminalObserver = new TerminalObserver();
			}
			try {
				Thread.sleep(2000);
			} catch (InterruptedException e) {

			}
		}
	}
	
	/**
	 * Closes the platform connection layer using native method.
	 */
	private void closeConnectionLayer() {
		updateTerminalList = false;
		try {
			Thread.sleep(3050);
		} catch (InterruptedException e) {
			// No error reporting
		}
		if (connected) {
			boolean disconnected = CONA.getInstance().disconnect();
			if (!disconnected) {
				generateConnectionLayerEvent(
						ConnectionLayerEvent.FAILED_TO_CLOSE_SERVICE_LAYER,
						PCSuiteMessages.Deployer_14);
			}
			connected = !disconnected;
		}
	}

	/**
	 * Returns the current connection status.
	 * 
	 * @return true if connection to service layer is open, otherwise false
	 * @uml.property name="connected"
	 */
	public boolean isConnected() {
		return connected;
	}

	/**
	 * Returns the terminals that are currently connected to the computer. The
	 * name includes an ID number for the session to identify them from each
	 * other.
	 * 
	 * @return terminal names as a String array, if no terminals are connected,
	 *         returns a zero-length String array.
	 */
	public String[] getTerminals() {
		if (!connected) {
			openConnectionLayer();
		}
		if (terminalObserver != null) {
			Vector result = new Vector();
			Hashtable openConnections = terminalObserver.getTerminals();
			Enumeration keys = openConnections.keys();
			while (keys.hasMoreElements()) {
				result.add(keys.nextElement());
			}
			return (String[]) result.toArray(new String[] {});
		} else {
			return new String[0];
		}
	}

	/**
	 * Returns the connection type of a particular terminal connected to the
	 * computer.
	 * 
	 * @param terminal
	 *            name of the terminal, complete with session ID
	 * @return connection type ("RS232", "IrDA", "USB" or "Bluetooth") of the
	 *         terminal
	 */
	public String getConnectionType(String terminal) {
		if (!connected) {
			openConnectionLayer();
		}
		if (terminalObserver != null) {
			return (String) terminalObserver.getTerminals().get(terminal);
		} else {
			return ""; //$NON-NLS-1$
		}
	}

	/**
	 * Returns the current array of terminal listeners.
	 * 
	 * @return array of terminal listeners
	 * @uml.property name="terminalListeners"
	 */
	public TerminalListener[] getTerminalListeners() {
		return (TerminalListener[]) (terminalListeners
				.toArray(new TerminalListener[] {}));
	}

	private void generateTerminalEvent(int type, String terminalName,
			String connectionType) {
		TerminalEvent terminalEvent = null;
		switch (type) {
		case TERMINAL_CONNECTED:
			terminalEvent = new TerminalEvent(this, terminalName,
					connectionType);
			for (int i = 0; i < terminalListeners.size(); i++) {
				((TerminalListener) terminalListeners.elementAt(i))
						.terminalConnected(terminalEvent);
			}
			break;
		case TERMINAL_DISCONNECTED:
			terminalEvent = new TerminalEvent(this, terminalName,
					connectionType);
			for (int i = 0; i < terminalListeners.size(); i++) {
				((TerminalListener) terminalListeners.elementAt(i))
						.terminalDisconnected(terminalEvent);
			}
			break;
		}
	}

	private void generateConnectionLayerEvent(int id, String message) {
		ConnectionLayerEvent connectionLayerEvent = new ConnectionLayerEvent(
				this, id, message);

		for (int i = 0; i < terminalListeners.size(); i++) {
			((TerminalListener) terminalListeners.elementAt(i))
					.connectionLayerEventOccurred(connectionLayerEvent);
		}
	}

	/**
	 * Returns the available com ports. Current implementation returns a long
	 * list of com ports because Java can't enumerate available ports. (Java
	 * Communications API might be used in the future.)
	 * 
	 * @return available com ports
	 */
	public int[] getComPorts() {
		return new int[] { 1, 2 };
	}

	/**
	 * Returns the currently used com port setting.
	 * 
	 * @return com port setting
	 * @uml.property name="comPort"
	 */
	public int getComPort() {
		return comPort;
	}
	
	/**
	 * Changes the com port for serial transmissions. COM port value is by
	 * default one (1). To change the com port setting the connection layer will
	 * have to disconnect and reconnect using the new setting. This might take a
	 * few seconds.
	 * 
	 * @param comPort
	 *            new com port to use for serial transmissions
	 * @uml.property name="comPort"
	 */
	public void setComPort(int comPort) {
		if (this.comPort != comPort && connected) {
			this.comPort = comPort;
		}
		this.comPort = comPort;
	}

	/**
	 * Deploys the JAR and JAD files to the selected terminal. Works only with
	 * Nokia ISA devices. Returns immediately, but starts a Thread that relays
	 * information about the deployment to registered {@link TerminalListener}
	 * s.
	 * 
	 * @param jad
	 *            JAD file to deploy
	 * @param jar
	 *            JAR file to deploy
	 * @param terminal
	 *            to deploy to, use the name you got from {@link #getTerminals}
	 * @throws IOException
	 *             if there is an error opening or reading the JAR or JAD file
	 * @throws UnsatisfiedLinkError
	 *             if ConnAPI.dll has not been loaded correctly. Either the
	 *             DLL was not found or it was an old and incompatible version.
	 * @see #getTerminals
	 * 
	 */
	public void deploy(File jad, File jar, String terminal) throws IOException,
			UnsatisfiedLinkError {
		if (!CONA.getInstance().getDllFound()) {
			throw new UnsatisfiedLinkError(PCSuiteMessages.Deployer_16);
		}
		if (!connected) {
			openConnectionLayer();
			if (!connected) {
				generateConnectionLayerEvent(
						ConnectionLayerEvent.FAILED_TO_OPEN_SERVICE_LAYER,
						PCSuiteMessages.Deployer_17);
				return;
			} else {
				generateConnectionLayerEvent(
						ConnectionLayerEvent.SERVICE_LAYER_OPENED,
						PCSuiteMessages.Deployer_18);
			}
		}

		int id = terminalObserver.getID(terminal);

		if (!jad.exists()) {
			throw new IOException(PCSuiteMessages.Deployer_19
					+ jad.getAbsolutePath());
		}

		if (!jar.exists()) {
			throw new IOException(PCSuiteMessages.Deployer_20
					+ jar.getAbsolutePath());
		}

		if (id != -1) {
			new MIDletSuiteDeploymentThread(jad, jar, id).start();
		} else {
			generateConnectionLayerEvent(
					ConnectionLayerEvent.FAILED_TO_FIND_DESTINATION_DEVICE,
					PCSuiteMessages.Deployer_21 + terminal);
		}
	}
	
	private boolean openConnection(int ind) {
		boolean openC = CONA.getInstance().openFileSystem(ind);
		if (openC) {
			return true;
		}

		return false;
	}

	private boolean installFile(String filepath, String filename,
			String jadfile) {
		boolean binstallfile = false;

		if (filename.toLowerCase().endsWith(".sis")) { //$NON-NLS-1$
			binstallfile = CONA.getInstance().installApplication(filepath,
					filename, null, CONA.CONA_APPLICATION_TYPE_SIS, true);
		} else {
			binstallfile = CONA.getInstance().installApplication(filepath,
					filename, jadfile, CONA.CONA_APPLICATION_TYPE_JAVA, true);
		}

		if (binstallfile) {
			return true;
		} else {
			return false;
		}
	}

	private boolean closeConnection() {
		return CONA.getInstance().closeFileSystem();
	}
	
	/**
	 * Deploys a JAD and JAR file to a ISA terminal device.
	 */
	private class MIDletSuiteDeploymentThread extends Thread {
		private File deployJAD = null;

		private File deployJAR = null;

		private int deployID = -1;

		/**
		 * Constructor for the deployment thread. Deployment starts when this
		 * thread is run.
		 * 
		 * @param jad
		 *            JAD to deploy
		 * @param jar
		 *            JAR to deploy
		 * @param id
		 *            terminal ID to which the files are deployed
		 */
		MIDletSuiteDeploymentThread(File jad, File jar, int id) {
			deployJAD = jad;
			deployJAR = jar;
			deployID = id;
		}

		/**
		 * Thread interface that does the actual deployment.
		 */
		public void run() {
			boolean success = true;
			String jarfilepath = deployJAR.getParentFile().getAbsolutePath();
			String jadfilename = deployJAD.getName();
			String jarfilename = deployJAR.getName();

			generateConnectionLayerEvent(
					ConnectionLayerEvent.OPENING_CONNECTION,
					PCSuiteMessages.Deployer_23 + deployID);

			boolean bopenconnection = openConnection(deployID);

			if (bopenconnection) {
				generateConnectionLayerEvent(
						ConnectionLayerEvent.CONNECTION_OPENED,
						PCSuiteMessages.Deployer_24 + deployID);

				String strFilepath = jarfilepath + "\\"; //$NON-NLS-1$

				success = installFile(strFilepath, jarfilename, jadfilename);

				if (success) {
					generateConnectionLayerEvent(
							ConnectionLayerEvent.DEPLOYMENT_SUCCESSFUL,
							PCSuiteMessages.Deployer_26);
					generateConnectionLayerEvent(
							ConnectionLayerEvent.FILE_DEPLOYED,
							PCSuiteMessages.Deployer_27);
				} else {
					generateConnectionLayerEvent(
							ConnectionLayerEvent.FAILED_TO_DEPLOY_FILE,
							PCSuiteMessages.Deployer_28 + jarfilename);

					generateConnectionLayerEvent(
							ConnectionLayerEvent.DEPLOYMENT_FAILED,
							PCSuiteMessages.Deployer_29);
				}
			} else { // open connection failed
				generateConnectionLayerEvent(
						ConnectionLayerEvent.FAILED_TO_OPEN_CONNECTION,
						PCSuiteMessages.Deployer_30 + deployID);
				generateConnectionLayerEvent(
						ConnectionLayerEvent.DEPLOYMENT_FAILED,
						PCSuiteMessages.Deployer_31);
				return;
			}

			closeConnection();

			generateConnectionLayerEvent(
					ConnectionLayerEvent.CONNECTION_CLOSED,
					PCSuiteMessages.Deployer_32 + deployID);
		}
	}

	
	/**
	 * Thread class that keeps track of the connected devices.
	 */
	private class TerminalObserver implements Runnable {
		Vector allTerminals = new Vector();

		Vector rs232Terminals = new Vector();

		Vector irDATerminals = new Vector();

		Vector usbTerminals = new Vector();

		Vector btTerminals = new Vector();

		TerminalObserver() {
			new Thread(this).start();
		}

		public Hashtable getTerminals() {
			Hashtable hashtable = new Hashtable();
			String terminal = ""; //$NON-NLS-1$
			String connection = ""; //$NON-NLS-1$
			for (int i = 0; i < rs232Terminals.size(); i++) {
				terminal = (String) rs232Terminals.elementAt(i);
				connection = "RS232"; //$NON-NLS-1$
				hashtable.put(terminal, connection);
			}
			for (int i = 0; i < irDATerminals.size(); i++) {
				terminal = (String) irDATerminals.elementAt(i);
				connection = "IrDA"; //$NON-NLS-1$
				hashtable.put(terminal, connection);
			}
			for (int i = 0; i < usbTerminals.size(); i++) {
				terminal = (String) usbTerminals.elementAt(i);
				connection = "USB"; //$NON-NLS-1$
				hashtable.put(terminal, connection);
			}
			for (int i = 0; i < btTerminals.size(); i++) {
				terminal = (String) btTerminals.elementAt(i);
				connection = "Bluetooth"; //$NON-NLS-1$
				hashtable.put(terminal, connection);
			}
			return hashtable;
		}

		public int getID(String terminal) {
			String idString = terminal.substring(terminal.indexOf("(ID:") + 4, //$NON-NLS-1$
					terminal.indexOf(")")).trim(); //$NON-NLS-1$
			return Integer.parseInt(idString);
		}

		public void run() {
			while (updateTerminalList) {
				if (deploying_) {
					try {
						Thread.sleep(3000);
					} catch (InterruptedException e) {
						// No error reporting
					}
					continue;
				}

				CONA.getInstance().updateDeviceList();

				String irDATerminalList = getIrDATerminals();
				String rs232TerminalList = getRS232Terminals();
				String usbTerminalList = getUSBTerminals();
				String btTerminalList = getBTTerminals();

				String completeTerminalList = irDATerminalList + "," //$NON-NLS-1$
						+ rs232TerminalList + "," + usbTerminalList + "," //$NON-NLS-1$ //$NON-NLS-2$
						+ btTerminalList;

				StringTokenizer irDATokenizer = new StringTokenizer(
						irDATerminalList, ","); //$NON-NLS-1$
				StringTokenizer rs232Tokenizer = new StringTokenizer(
						rs232TerminalList, ","); //$NON-NLS-1$
				StringTokenizer usbTokenizer = new StringTokenizer(
						usbTerminalList, ","); //$NON-NLS-1$
				StringTokenizer btTokenizer = new StringTokenizer(
						btTerminalList, ","); //$NON-NLS-1$
				StringTokenizer allTerminalsTokenizer = new StringTokenizer(
						completeTerminalList, ","); //$NON-NLS-1$

				while (irDATokenizer.hasMoreTokens()) {
					String irDATerminal = irDATokenizer.nextToken();
					if (!isOnTerminalList(allTerminals, irDATerminal)) {
						allTerminals.add(irDATerminal);
						irDATerminals.add(irDATerminal);
						generateTerminalEvent(TERMINAL_CONNECTED, irDATerminal,
								"IrDA"); //$NON-NLS-1$
					}
				}

				while (rs232Tokenizer.hasMoreTokens()) {
					String rs232Terminal = rs232Tokenizer.nextToken();
					if (!isOnTerminalList(allTerminals, rs232Terminal)) {
						allTerminals.add(rs232Terminal);
						rs232Terminals.add(rs232Terminal);
						generateTerminalEvent(TERMINAL_CONNECTED,
								rs232Terminal, "RS232"); //$NON-NLS-1$
					}
				}

				while (usbTokenizer.hasMoreTokens()) {
					String usbTerminal = usbTokenizer.nextToken();
					if (!isOnTerminalList(allTerminals, usbTerminal)) {
						allTerminals.add(usbTerminal);
						usbTerminals.add(usbTerminal);
						generateTerminalEvent(TERMINAL_CONNECTED, usbTerminal,
								"USB"); //$NON-NLS-1$
					}
				}

				while (btTokenizer.hasMoreTokens()) {
					String btTerminal = btTokenizer.nextToken();
					if (!isOnTerminalList(allTerminals, btTerminal)) {
						allTerminals.add(btTerminal);
						btTerminals.add(btTerminal);
						generateTerminalEvent(TERMINAL_CONNECTED, btTerminal,
								"Bluetooth"); //$NON-NLS-1$
					}
				}

				Vector currentTerminals = new Vector();

				while (allTerminalsTokenizer.hasMoreTokens()) {
					String terminal = allTerminalsTokenizer.nextToken();
					currentTerminals.add(terminal);
				}

				for (int i = 0; i < allTerminals.size(); i++) {
					String terminal = (String) allTerminals.elementAt(i);
					boolean found = false;
					for (int j = 0; j < currentTerminals.size(); j++) {
						String currentTerminal = (String) currentTerminals
								.elementAt(j);
						if (currentTerminal.compareTo(terminal) == 0) {
							found = true;
							break;
						}
					}

					if (!found) {
						if (isOnTerminalList(irDATerminals, terminal)) {
							removeTerminal(allTerminals, terminal);
							removeTerminal(irDATerminals, terminal);
							generateTerminalEvent(TERMINAL_DISCONNECTED,
									terminal, "IrDA"); //$NON-NLS-1$
						} else if (isOnTerminalList(rs232Terminals, terminal)) {
							removeTerminal(allTerminals, terminal);
							removeTerminal(rs232Terminals, terminal);
							generateTerminalEvent(TERMINAL_DISCONNECTED,
									terminal, "RS232"); //$NON-NLS-1$
						} else if (isOnTerminalList(usbTerminals, terminal)) {
							removeTerminal(allTerminals, terminal);
							removeTerminal(usbTerminals, terminal);
							generateTerminalEvent(TERMINAL_DISCONNECTED,
									terminal, "USB"); //$NON-NLS-1$
						} else if (isOnTerminalList(btTerminals, terminal)) {
							removeTerminal(allTerminals, terminal);
							removeTerminal(btTerminals, terminal);
							generateTerminalEvent(TERMINAL_DISCONNECTED,
									terminal, "Bluetooth"); //$NON-NLS-1$
						}
						i = -1;
					}
				}
				try {
					Thread.sleep(3000);
				} catch (InterruptedException e) {
					// No error reporting
				}
			}
		}

		private String getRS232Terminals() {
			String devices = CONA.getInstance().getDevices(
					CONA.CONAPI_MEDIA_SERIAL);
			if (devices == null) {
				devices = ""; //$NON-NLS-1$
			}
			return StringManipulator.substitute(devices, "E ", "EPOC device "); //$NON-NLS-1$ //$NON-NLS-2$
		}

		/**
		 * @return Returns the irDATerminals.
		 * @uml.property name="irDATerminals"
		 */
		private String getIrDATerminals() {
			String devices = CONA.getInstance().getDevices(
					CONA.CONAPI_MEDIA_IRDA);
			if (devices == null) {
				devices = ""; //$NON-NLS-1$
			}
			return StringManipulator.substitute(devices, "E ", "EPOC device "); //$NON-NLS-1$ //$NON-NLS-2$
		}

		private String getUSBTerminals() {
			String devices = CONA.getInstance().getDevices(
					CONA.CONAPI_MEDIA_USB);
			if (devices == null) {
				devices = ""; //$NON-NLS-1$
			}
			return StringManipulator.substitute(devices, "E ", "EPOC device "); //$NON-NLS-1$ //$NON-NLS-2$
		}

		private String getBTTerminals() {
			String devices = CONA.getInstance().getDevices(
					CONA.CONAPI_MEDIA_BLUETOOTH);
			if (devices == null) {
				devices = ""; //$NON-NLS-1$
			}

			return devices;
		}

		private boolean isOnTerminalList(Vector terminals, String terminal) {
			for (int i = 0; i < terminals.size(); i++) {
				if (terminal.compareTo((String) terminals.elementAt(i)) == 0) {
					return true;
				}
			}
			return false;
		}

		private int whichTerminalOnList(Vector terminals, String terminal) {
			for (int i = 0; i < terminals.size(); i++) {
				if (terminal.compareTo((String) terminals.elementAt(i)) == 0) {
					return i;
				}
			}
			return -1;
		}

		private void removeTerminal(Vector terminals, String terminal) {
			for (int i = 0; i < terminals.size(); i++) {
				if (terminal.compareTo((String) terminals.elementAt(i)) == 0) {
					terminals.remove(i);
				}
			}
		}
	}

}
