/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/

package org.eclipse.mtj.extension.dplp.nokia.pcsuite.util;


import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;

/**
 * A utility class for manipulating strings.
 */
public class StringManipulator {
    private static final char QUOTE_CHAR   = '\\';
    private static final String REP_CHARS_ = "<>&\n"; //$NON-NLS-1$
    private static final String[] REP_STRINGS_ = { "&lt;", //$NON-NLS-1$
                                                   "&gt;", //$NON-NLS-1$
                                                   "&amp;", //$NON-NLS-1$
                                                   "<br>" }; //$NON-NLS-1$
    private StringManipulator() {}

    /**
     * Substitutes all occurrences of oldString in text with newString. This
     * should be used if String.replaceAll() can't be used due to regular
     * expression symbols in the String.
     * @param text text to modify
     * @param oldString a string that will be replaced
     * @param newString a string that will replace the old string
     * @return text with all occurrences of oldString replaced with newString
     */
    public static String substitute(String text, String oldString,
                                    String newString) {
        String begin = ""; //$NON-NLS-1$
        String end = ""; //$NON-NLS-1$
        int len = oldString.length();
        int beginPtr = text.indexOf(oldString);
        while (beginPtr > -1) {
            begin = text.substring(0,beginPtr);
            end = text.substring(beginPtr+len);
            text = begin + newString + end;
            beginPtr = text.indexOf(oldString);
        }

        return text;
    }

    /**
     * Finds the components of a string. A component is a word surrounded
     * with spaces or multiple words surrounded with delimiters.
     * @param text text to componentisize
     * @param delimiter delimiter for a string component
     * @return all components from the given string
     */
    public static String[] getComponents( String text, char delimiter )
    {
      List components = new LinkedList();
      StringBuffer buffer = new StringBuffer();
      boolean delimited = false;
      for( int i = 0; i < text.length(); i++ )
      {
        char curChar = text.charAt(i);
        if( Character.isWhitespace( curChar ) && !delimited )
        {
          if( buffer.toString().length() > 0 )
          {
            components.add( buffer.toString() );
          }
          buffer = new StringBuffer();
        }
        else if( curChar == delimiter && delimited )
        {
          delimited = false;
          if( buffer.toString().length() > 0 )
          {
            components.add( buffer.toString() );
          }
          buffer = new StringBuffer();
        }
        else if( Character.isWhitespace( curChar ) && delimited )
        {
          buffer.append( curChar );
        }
        else if( curChar == delimiter && !delimited )
        {
          delimited = true;
/*          if( buffer.toString().length() > 0 )
          {
            components.add( buffer.toString() );
          }
          buffer = new StringBuffer();*/
        }
        else
        {
          buffer.append( curChar );
        }
      }
      if( buffer.toString().length() > 0 )
      {
        components.add( buffer.toString() );
      }
      return (String[])components.toArray(new String[]{});
    }

    /**
     * Returns combined class path.
     * @return emulator's class path (for CLDC and MIDP API classes)
     */
    public static String[] combineClassPaths( String[] emulatorClassPath,
                                              String userClassPath ) {
        Vector wholeClassPath
            = new Vector( Arrays.asList( emulatorClassPath ));
        wholeClassPath.add( userClassPath );
        return (String[])wholeClassPath.toArray( new String[]{} );
    }

    /**
     * Returns combined class path.
     * @return emulator's class path (for CLDC and MIDP API classes)
     */
    public static String[] combineClassPaths( File[] emulatorClassPath,
                                              String userClassPath ) {
        Vector wholeClassPath = new Vector();
        for( int i = 0; i < emulatorClassPath.length; i++ )
        {
          wholeClassPath.add( emulatorClassPath[i].getAbsolutePath() );
        }
        wholeClassPath.add( userClassPath );
        return (String[])wholeClassPath.toArray( new String[]{} );
    }

    /**
     * Returns class path separated with File.pathSeparators.
     * @param class_path
     * @return class path as String
     */
    public static String getClassPathString( File[] class_path )
    {
      String ret = ""; //$NON-NLS-1$
      for( int i = 0; i < class_path.length; i++ )
      {
        ret = ret + class_path[i].getAbsolutePath();
        if( i+1 < class_path.length )
        {
          ret = ret + File.pathSeparator;
        }
      }
      return ret;
    }

    /**
     * Returns class path as files.
     * @return class path as files
     */
    public static File[] getClassPathAsFiles( String[] classPath ) {
        File files[] = new File[classPath.length];
        for( int i = 0; i < classPath.length; i++ )
            {
                files[i] = new File(classPath[i]);
            }
        return files;
    }

    /**
     * Returns class path as URLs.
     * @return class path as URLs
     */
    public static URL[] getClassPathAsURLs( String[] classPath )
    {
      File files[] = getClassPathAsFiles( classPath );
      URL urls[] = new URL[files.length];
      for( int i = 0; i < files.length; i++ )
      {
        try
        {
          urls[i] = files[i].toURL();
        }
        catch( MalformedURLException e )
        {
          System.out.println( "Error converting path to URL: " //$NON-NLS-1$
                              + e.toString() );
        }
      }
      return urls;
    }

    /**
     * Returns class path as URLs.
     * @return class path as URLs
     */
    public static URL[] getClassPathAsURLs( File[] class_path )
    {
      URL urls[] = new URL[class_path.length];
      for( int i = 0; i < class_path.length; i++ )
      {
        try
        {
          urls[i] = class_path[i].toURL();
        }
        catch( MalformedURLException e )
        {
          System.out.println( "Error converting path to URL: " //$NON-NLS-1$
                              + e.toString() );
        }
      }
      return urls;
    }

    /**
     * Returns HTML safe version of the given string.
     * The characters in the given string are replaced according
     * to the following table:
     * <table border="1" cellpadding="4">
     * <tr><th>from</th><th>to</th></tr>
     * <tr><td>&lt;</td><td>&amp;lt;</td></tr>
     * <tr><td>&gt;</td><td>&amp;gt;</td></tr>
     * <tr><td>&amp;</td><td>&amp;amp;</td></tr>
     * <tr><td><code>'\n'</code></td><td>&lt;br&gt;</td></tr>
     * </table>
     */
    public static String toHtml( String s ) {
        StringBuffer buf = new StringBuffer();
        out: for( int i = 0; i < s.length(); i++ ) {
            for( int c = 0; c < REP_CHARS_.length(); c++ ) {
                if( s.charAt( i ) == REP_CHARS_.charAt( c ) ) {
                    buf.append( REP_STRINGS_[c] );
                    continue out;
                }
            }
            buf.append( s.charAt( i ) );
        }
        return buf.toString();
    }

    /**
     * Returns unquoted version of the given string.
     * Quoted newlines ("\\n") are replaced by newline characters ('\n').
     */
    public static String unquote( String s ) {
        StringBuffer buf = new StringBuffer();
        boolean quoted = false;
        for( int i = 0; i < s.length(); i++ ) {
            if( quoted ) {
                buf.append( s.charAt( i ) == 'n' ? '\n' : s.charAt( i ) );
                quoted = false;
                continue;
            }
            if( s.charAt( i ) == QUOTE_CHAR ) {
                quoted = true;
            } else {
                buf.append( s.charAt( i ) );
            }
        }
        return buf.toString();
    }
}

