package org.eclipse.mtj.extension.sp.impl.j9;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.mtj.api.deployment.Deployment;
import org.eclipse.mtj.api.enumerations.DeploymentType;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.extension.SigningProvider;
import org.eclipse.mtj.api.extension.impl.MtjExtensionImpl;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.api.model.MidletDeployment;
import org.eclipse.mtj.api.signings.SigningCertificate;
import org.eclipse.mtj.core.MtjCoreErrors;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.sp.j9.J9SigningPlugin;
import org.eclipse.mtj.jad.util.ConvertNativeToUTF8;
import org.eclipse.mtj.jad.util.JadFileDescriptor;


public class J9SigningProvider extends MtjExtensionImpl  implements SigningProvider{

	private IPreferenceStore sigingProviderPrefStore;
	private SigningCertificate targetSigningCertificate;
	
	public J9SigningProvider() {
		super();
		setId(J9SigningPlugin.getDefault().getBundle().getSymbolicName());
		setVendor(Messages.J9SigningProvider_PluginVendor); 
		setVersion(Messages.J9SigningProvider_PluginVersion); 
		setDescription(Messages.J9SigningProvider_15); 
		setType(ExtensionType.SIGNING_PROVIDER_LITERAL);
		sigingProviderPrefStore = J9SigningPlugin.getDefault().getPreferenceStore();
	}

	/**
	 * Method returns the deployment types that are supported by itself.
	 * 
	 * @return
	 */
	public DeploymentType[] getSupportedTypes() throws MtjException {
		//This Signing provider only supports MIDP 
		DeploymentType [] supportedDeploymentTypes = {DeploymentType.DEPLOYMENT_TYPE_MIDLET_LITERAL};
		return supportedDeploymentTypes;
	}

	private boolean isSupported(Deployment deployment) throws MtjException
	{
		boolean supported = false;
//		DeploymentType [] typesSupported = getSupportedTypes();
//		for (int i = 0; i < typesSupported.length; i++)
//		{
//			if (deployment.getType() == typesSupported[i])
//				supported = true;
//		}
		if (deployment instanceof MidletDeployment)
			supported = true;
		
		return supported;
	}
	/**
	 * Method signs the deployment with the certificates.
	 * 
	 * @param deployment
	 * @param targetFolder
	 * @param certificates
	 * @param monitor
	 * @return success
	 */
	public boolean sign(IMtjProject project , IFolder targetFolder, SigningCertificate certificates, IProgressMonitor monitor) throws MtjException {
		boolean cmdSuccessful = false;
		monitor.beginTask(Messages.J9SigningProvider_0,4);  
		targetSigningCertificate = certificates;
		Deployment deployment;
		try {
			deployment = project.getDeployment();
			if (isSupported(deployment)){
				
				String jadNameLoc = ((MidletDeployment)deployment).getJad().getAbsolutePath();
				String jarNameLoc = ((MidletDeployment)deployment).getJar().getAbsolutePath();

				cmdSuccessful = signMIDPApplication(jadNameLoc,jarNameLoc, monitor);	
				
				IFile JadFile = getProjectjadIFile(project);
				if (JadFile != null)
					JadFile.refreshLocal(IResource.DEPTH_ZERO, monitor);
			}
			else{
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_DEPLOYMENT_NOT_SUPPORTED) + " " + Messages.J9SigningProvider_6);  //$NON-NLS-1$
			}
		} catch (CoreException e) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), e);
			
		}
	
		
		monitor.done();
		return cmdSuccessful;
	}

	/*
	 * 
	 * 
	 */
	private boolean signMIDPApplication( String jadFile, String jarFile, IProgressMonitor monitor) throws MtjException {
			
		boolean success = true;
		monitor.beginTask(Messages.J9SigningProvider_7, 100); 
		
		String cmdError=""; //$NON-NLS-1$
		try {
			String cmdArgs[] = generateSignApplicationCmd( jadFile);
			Process p = runMIDPSecurityCmd(cmdArgs);
			monitor.worked(40);
			BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getErrorStream()));	
				
			while ((cmdError=cmdOutputStream.readLine()) != null) {
				String errorMessage = Messages.J9SigningProvider_1;
				if (cmdError.toLowerCase().indexOf(errorMessage.toLowerCase())>=0) {   
					//MessageDialog.openError(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), Messages.getString("J9SigningProvider.2"), cmdError);  //$NON-NLS-1$
					//break;
					//success = false;
					monitor.done();
					if (cmdError.toLowerCase().indexOf("invalid keystore")>=0) {   //$NON-NLS-1$
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_BAD_KEYSTORE_OR_PASSWORD) + " " + cmdError); //$NON-NLS-1$
					} else if (cmdError.toLowerCase().indexOf("keystore not found")>=0) {   //$NON-NLS-1$
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_KEYSOTRE_NOT_FOUND) + " " + cmdError); //$NON-NLS-1$
					} else if (cmdError.toLowerCase().indexOf("Incorrect password")>=0) {   //$NON-NLS-1$
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_INVALID_KEY_PASSWORD) + " " + cmdError); //$NON-NLS-1$
					}else {
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + " " + cmdError); //$NON-NLS-1$
					}
				}
			}
			
		}  catch (IOException e) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), e);
		}
		
		monitor.done();
		return success;
	}
	
	public boolean unsign(IMtjProject project, IFolder targetFolder, IProgressMonitor monitor) throws MtjException {
		JadFileDescriptor jadDescriptor;
		
		monitor.beginTask(Messages.J9SigningProvider_8, 100); 
		
		IFile JadFile = getProjectjadIFile(project);		
			
			try {
				if (isSupported(project.getDeployment())) {
					if (JadFile != null) {
						jadDescriptor = JadFileDescriptor.getDescriptor(JadFile);
						monitor.worked(20);
						jadDescriptor.removeProperty(JadFileDescriptor.S_JAR_RSA_SHA1);
						jadDescriptor.removeProperty(JadFileDescriptor.S_CERTIFICATE1_1);
						monitor.worked(20);
						//run the descriptor content through a native->latin-1+escapes filter
						JadFile.setContents(ConvertNativeToUTF8.convertNativeToUTF8(jadDescriptor.serialize()), true,true, null);
						monitor.worked(60);
						
						JadFile.refreshLocal(IResource.DEPTH_ZERO, monitor);
					}
					else
						throw new MtjException(Messages.J9SigningProvider_9); 
					}
				else 
				{
					throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_DEPLOYMENT_NOT_SUPPORTED) + " " + Messages.J9SigningProvider_6);  //$NON-NLS-1$
				}
				
			} catch (CoreException e2) {
				monitor.done();
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), e2);
			} catch (IOException e) {
				monitor.done();
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), e);
			}

		monitor.done();
		return true;
	}

	public boolean isSigned(IMtjProject mtjProject, IFolder targetFolder, IProgressMonitor monitor) throws MtjException {
		boolean isSigned = false;
		
		IFile JadFile = getProjectjadIFile(mtjProject);
		JadFileDescriptor jadDescriptor;
		try 
		{
			if (JadFile != null) {
			    jadDescriptor = JadFileDescriptor.getDescriptor(JadFile);
			    // check if the JAD file for the MIDlet build is signed
			    if ((jadDescriptor.getProperty(JadFileDescriptor.JAR_RSA_SHA1)!=null)||(jadDescriptor.getProperty(JadFileDescriptor.CERTIFICATE1)!=null)){
			    	isSigned = true;
			    }			    		    
			}//(jadFile!=null)
			else
				throw new MtjException(Messages.J9SigningProvider_12); 
		} catch (CoreException e2) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), e2);
		} catch (IOException e) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), e);
		}
	
		return isSigned;

	}
	private IFile getProjectjadIFile(IMtjProject project) throws MtjException {	
		IFile jadIFile = null;
		Deployment dp = null;
		
		try {
			dp = project.getDeployment();
		} catch (CoreException e) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.J9SigningProvider_13, e); 
		}
		if(dp.getType() == DeploymentType.DEPLOYMENT_TYPE_MIDLET_LITERAL)
		{
			try {
				jadIFile = project.getDeploymentFolder(null).getFile(((MidletDeployment)dp).getJad().getName());
			} catch (CoreException e) {
				throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.J9SigningProvider_14, e); 
			}
		}
		return jadIFile;
	}

	public String getToolLocation(IProgressMonitor monitor) throws MtjException {
		return sigingProviderPrefStore.getString(J9SigningProviderConstants.SIGNING_TOOL_LOCATION);		
	}

	public void storeToolLocation(String loc, IProgressMonitor monitor) throws MtjException {	
		sigingProviderPrefStore.setValue(J9SigningProviderConstants.SIGNING_TOOL_LOCATION, loc);
	}

	public String getExtJar() {
		// TODO Auto-generated method stub
		return null;
	}

	public void setExtJar(String value) {
		// TODO Auto-generated method stub
		
	}
	/*
	 * 
	 * 
	 */
	private String[] generateSignApplicationCmd( String jadFile) throws MtjException {
		
		String[] signMIDPApplicationCmdArgs = {getJadSignerLocation(),
					 getConsoleEncoding(),
					 J9SigningProviderConstants.SIGN,
					 J9SigningProviderConstants.KEYSTORE,targetSigningCertificate.getKeystore(),
					 J9SigningProviderConstants.STOREPASS,targetSigningCertificate.getKeystorePassword(),
					 J9SigningProviderConstants.ALIAS,targetSigningCertificate.getAliasName(),
					 J9SigningProviderConstants.KEYPASS,targetSigningCertificate.getCertificatePassword(),
					 J9SigningProviderConstants.JADFILE,jadFile};
					
		return signMIDPApplicationCmdArgs;
	}
	
	private static String getConsoleEncoding ()
	{
		return  "-J-Dconsole.encoding=" + System.getProperty("file.encoding"); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
    /*
	 * Get the signing tool location.
	 * 
	 */
	private final String getJadSignerLocation() throws MtjException {
		String signingToolLocation = getToolLocation(null);
		
		if (signingToolLocation == null) {
			//MessageDialog.openError(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(),Messages.getString("J9SigningProvider.3"), Messages.getString("J9SigningProvider.4")); 	 //$NON-NLS-1$ //$NON-NLS-2$
			//return ""; //$NON-NLS-1$
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + Messages.J9SigningProvider_4); 
		}
		
		StringBuffer buffer = new StringBuffer("\"");
		buffer.append(signingToolLocation)
		.append(File.separator)
		.append("bin") //$NON-NLS-1$						
		.append(File.separator)
		.append("jadsigner.exe") //$NON-NLS-1$
		.append("\""); //$NON-NLS-1$
		return buffer.toString();
	}
	
	/**
	 * Execute the array of commands passed in
	 * @param cmd
	 * @return the process generated by the execution of the commads
	 * @throws MtjException 
	 */
	private Process runMIDPSecurityCmd (String[] cmd) throws MtjException {
		
		Process p = null;
		try {p =	Runtime.getRuntime().exec(cmd);	} 
		catch (Exception e){}
		
		if (p == null)
		 {
			StringBuffer str = new StringBuffer(""); //$NON-NLS-1$
			
			for (int i = 0; i < cmd.length; i++)
			{
				str.append(" " + cmd[i]); //$NON-NLS-1$
			}
		
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + " could not execute: " + "[" + str +"]");   //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
		}
		
		return p;	
	}

	public String [] listrootcert(String rootcert) throws MtjException {
		String[] cmdArgs = null;
		//ArrayList aliases = new ArrayList();
		List<String> aliases = new ArrayList<String>();

		cmdArgs = generateOpenRootCertCmd(rootcert);
		Process p = runMIDPSecurityCmd(cmdArgs);

		BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getInputStream()));
//		BufferedReader cmdErrStream = new BufferedReader(new InputStreamReader(p.getErrorStream()));

		String cmdOutput;
		
		try {
			//the ready call is unreliable, don't try to read error stream
//			//if (cmdErrStream.ready()){
//				while ((cmdOutput=cmdErrStream.readLine()) != null) {
//					if (cmdOutput.length() > 0){
//						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) + " " + cmdOutput); //$NON-NLS-1$
//					} 				
//				}
//			//}
		
			//the ready call is unreliable
			//if (cmdOutputStream.ready()){
				while ((cmdOutput=cmdOutputStream.readLine()) != null) {
					if (cmdOutput.indexOf(J9SigningProviderConstants.CERTIFICATE_DELIMITER)>=0) { 
						StringBuffer cmds = new StringBuffer(cmdOutput);
						for (int i = 0; i <= 5; i++){
							cmds.append(" "); //$NON-NLS-1$
							if ((cmdOutput=cmdOutputStream.readLine()) != null){
								if (cmdOutput.equals("")){ //$NON-NLS-1$
									break;	//some records do not include all data expected
								}
								cmds.append(cmdOutput);
							}
						}
						aliases.add(cmds.toString());
					}
				}
			//}
		} catch (IOException ee) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING), ee);
		}
		if (aliases.isEmpty()){
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SIGNING) + " Root certificate is empty or invalid");   //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
		}

		return (String[])aliases.toArray(new String[aliases.size()]);
	}

	private String[] generateOpenRootCertCmd(String rootcert) throws MtjException {
	String[] newKeyCmdArgs = {
		getMESecurityManagementTool(),
		J9SigningProviderConstants. MEKEYTOOL_LIST,
		J9SigningProviderConstants.MEKEYTOOL_MEKEYSTORE,
		rootcert			
		};
	
	return newKeyCmdArgs;
	}

public void importToRootCert(String rootcert, String keystore, String alias, String password)throws MtjException {
	String[] cmdArgs = null;
		
	cmdArgs = generateImportToRootCertCmd(rootcert, keystore, alias, password);
	Process p = runMIDPSecurityCmd(cmdArgs);
	
	BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getErrorStream()));

	String cmdOutput;
	try {
		while ((cmdOutput=cmdOutputStream.readLine()) != null) {
					
			if (cmdOutput.length() !=0) { 

				if (cmdOutput.toLowerCase().indexOf("already exists in keystore")>=0) {   //$NON-NLS-1$
					throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SECURITY_ALIAS_DUPLICATE) + " " +cmdOutput); //$NON-NLS-1$
				} 
				else {
					throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY) + " " + cmdOutput); //$NON-NLS-1$
				}
			} 
		}
	} catch (IOException ee) {
		throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY), ee);
	}
}
private String[] generateImportToRootCertCmd(String rootcert, String keystore, String alias, String password) throws MtjException {
	String[] importCertificateCmdArgs = 
	{	
		getMESecurityManagementTool(),
		J9SigningProviderConstants.MEKEYTOOL_IMPORT,
		J9SigningProviderConstants.MEKEYTOOL_MEKEYSTORE,
		rootcert,
		J9SigningProviderConstants.MEKEYTOOL_ALIAS, alias,
		J9SigningProviderConstants.MEKEYTOOL_KEYSTORE,keystore,
		J9SigningProviderConstants.MEKEYTOOL_STOREPASS,password};

	return importCertificateCmdArgs;
}

public boolean isRootCertToolAvailable() throws MtjException {
	return (getMESecurityManagementTool() != null) ? true : false;
}

private String getMESecurityManagementTool() throws MtjException {
	String securityToolLocation = getToolLocation(null);
	
	if (securityToolLocation == null ||
		securityToolLocation.length() <= 0 ||
		securityToolLocation.equals(Messages.J9SigningProviderPrefInitializer_0)	 
		){
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.SIGNING_PROVIDER_NOT_CONFIGURED) + "\n" + getId() + Messages.J9SigningProvider_19); //$NON-NLS-1$
		}
	
	StringBuffer buffer = new StringBuffer("\""); //$NON-NLS-1$
	buffer.append(securityToolLocation)
	.append(File.separator)
	.append("bin") //$NON-NLS-1$						
	.append(File.separator)
	.append("MEkeytool.exe") //$NON-NLS-1$
	.append("\""); //$NON-NLS-1$
		
	return buffer.toString();
	}

	public void removeCertFromRoot(String rootcert, int removeindex) throws MtjException {
		String[] cmdArgs = null;

		cmdArgs = generateDeleteFromRootCertCmd(rootcert, removeindex);
		Process p = runMIDPSecurityCmd(cmdArgs);

		//BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getInputStream()));
		BufferedReader cmdOutputStream = new BufferedReader(new InputStreamReader(p.getErrorStream()));

		String cmdOutput;

		try {
			while ((cmdOutput = cmdOutputStream.readLine()) != null) {

				if (cmdOutput.length() != 0) {
						throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY)+ " " + cmdOutput); //$NON-NLS-1$
				}
			}
		} catch (IOException ee) {
			throw new MtjException(MtjCoreErrors.getErrorMessage(MtjCoreErrors.ERROR_BASE_SECURITY), ee);
		}
	}

	private String[] generateDeleteFromRootCertCmd(String rootcert,
			int removeindex) throws MtjException {
		String[] removeCertificateCmdArgs = {
				getMESecurityManagementTool(), 
				J9SigningProviderConstants.MEKEYTOOL_DELETE,
				J9SigningProviderConstants.MEKEYTOOL_MEKEYSTORE, rootcert,
				J9SigningProviderConstants.MEKEYTOOL_NUMBER, Integer.toString(removeindex) };

		return removeCertificateCmdArgs;
	}
}
