/**
 * Copyright (c) 2003-2005 Craig Setera
 * All Rights Reserved.
 * Licensed under the Eclipse Public License - v 1.0
 * For more information see http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.mtj.extension.devide.jadeditor;

import java.io.IOException;
import java.util.Iterator;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IStorageEditorInput;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.editor.IFormPage;
import org.eclipse.ui.part.EditorPart;

// TODO Error status area... Updated by field editors

/**
 * Editor for specifying the properties of a MIDP 
 * application descriptor.
 * <p />
 * Copyright (c) 2003-2005 Craig Setera<br>
 * All Rights Reserved.<br>
 * Licensed under the Eclipse Public License - v 1.0<p/>
 * <br>
 * $Revision: 1.2 $
 * <br>
 * $Date: 2006/09/05 15:12:25 $
 * <br>
 * @author Craig Setera
 * @see EditorPart
 */
public class JADEditor extends FormEditor 
{

	// The preference store we are using to 
	// represent the edit state	
	private ManifestPreferenceStore preferenceStore;

	// The underlying file being operated on
	private IFile jadFile;
	
	// The most recent timestamp of the JAD file
	private long modificationStamp;

	// The pages in the editor
	private JADRequiredPropertiesEditorPage requiredPropertiesEditorPage;
	private JADMidletsEditorPage midletsEditorPage;
	private JADOptionalPropertiesEditorPage optionalPropertiesEditorPage;
	private JADOTAPropertiesEditorPage otaPropertiesEditorPage;
	private JADUserDefinedPropertiesEditorPage userDefinedPropertiesEditorPage;

	/**
	 * @see org.eclipse.ui.part.MultiPageEditorPart#createPages()
	 */
	protected void addPages() {
		try {
			String title = JadEditorStrings.editor_jad_tab_required_properties;
			requiredPropertiesEditorPage = new JADRequiredPropertiesEditorPage(this);
			addPage(requiredPropertiesEditorPage);
			
			title = JadEditorStrings.editor_jad_tab_midlets;
			midletsEditorPage = new JADMidletsEditorPage(this, title);
			addPage(midletsEditorPage);

			title = JadEditorStrings.editor_jad_tab_optional_properties;
			optionalPropertiesEditorPage = new JADOptionalPropertiesEditorPage(this);
			addPage(optionalPropertiesEditorPage);

			title = JadEditorStrings.editor_jad_tab_ota;
			otaPropertiesEditorPage = new JADOTAPropertiesEditorPage(this);
			addPage(otaPropertiesEditorPage);

			title = JadEditorStrings.editor_jad_tab_user_defined_properties;
			userDefinedPropertiesEditorPage = new JADUserDefinedPropertiesEditorPage(this, title);
			addPage(userDefinedPropertiesEditorPage);
			
		} catch (PartInitException e) {
			MtjCorePlugin.log(IStatus.ERROR, e);
		}
	}

	/**
	 * @see org.eclipse.ui.ISaveablePart#isDirty()
	 */
	public boolean isDirty() {
		boolean dirty = false;
		
		Iterator pageIter = pages.iterator();
		while (pageIter.hasNext()) {
			IFormPage formPage = (IFormPage) pageIter.next();

			// If the control hasn't been created yet, it can't be dirty yet
			if ((formPage != null) && (formPage.getPartControl() != null)) {
				if (formPage.isDirty())  {
					dirty = true;
					break;
				}
			}
		}
		
		return dirty;
	}
	
	/**
	 * @see org.eclipse.ui.part.EditorPart#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void doSave(IProgressMonitor monitor) {
		monitor.beginTask("", getPageCount() + 1);
		
		int i = 0;
		Iterator pageIter = pages.iterator();
		while (pageIter.hasNext()) {
			IFormPage formPage = (IFormPage) pageIter.next();

			// If the control hasn't been created yet, it can't be dirty yet
			if ((formPage != null) && (formPage.getPartControl() != null)) {
				if (formPage.isDirty())  {
					formPage.doSave(monitor);
				}
			}
			
			monitor.worked(i + 1);
		}

		try {
			preferenceStore.save();
			
			if ((jadFile != null) && (jadFile.exists())) {
				jadFile.refreshLocal(IResource.DEPTH_ZERO, monitor);
			}
		} catch (IOException e) {
			MtjCorePlugin.log(IStatus.ERROR, e);
		} catch (CoreException e) {
			MtjCorePlugin.log(IStatus.ERROR, e);
		}
		
		monitor.done();
		editorDirtyStateChanged();
	}

	/**
	 * @see org.eclipse.ui.part.EditorPart#doSaveAs()
	 */
	public void doSaveAs() {
		// Not allowed...
	}

	/**
	 * Get the IPreferenceStore in use for the edit
	 * function.
	 * 
	 * @return
	 */
	public ManifestPreferenceStore getPreferenceStore() {
		return preferenceStore;
	}

	/**
	 * @see org.eclipse.ui.part.EditorPart#gotoMarker(org.eclipse.core.resources.IMarker)
	 */
	public void gotoMarker(IMarker marker) {
		// Nothing to do here...
	}

	/**
	 * @see org.eclipse.ui.part.EditorPart#isSaveAsAllowed()
	 */
	public boolean isSaveAsAllowed() {
		// Don't allow "Save As..."
		return false;
	}

	/**
	 * @see org.eclipse.ui.part.MultiPageEditorPart#setFocus()
	 */
	public void setFocus() {
		if ((jadFile != null) && (jadFile.getModificationStamp() > modificationStamp)) {
			if (shouldReloadFile()) {
				updateEditorInput();
				
				requiredPropertiesEditorPage.editorInputChanged();
				optionalPropertiesEditorPage.editorInputChanged();
				otaPropertiesEditorPage.editorInputChanged();
				midletsEditorPage.editorInputChanged();
				userDefinedPropertiesEditorPage.editorInputChanged();
			}
		}

		super.setFocus();
	}

	/**
	 * @see org.eclipse.ui.part.EditorPart#setInput(org.eclipse.ui.IEditorInput)
	 */
	protected void setInput(IEditorInput input) {
		super.setInput(input);
		
		// Update the application descriptor instance
		if (input instanceof IStorageEditorInput) {
			IStorageEditorInput storageInput = (IStorageEditorInput) input;
			try {
				// Read the storage from the file system as 
				// a preference store
				IPath storagePath = storageInput.getStorage().getFullPath();
				if (storagePath != null) {
					IWorkspaceRoot root = MtjCorePlugin.getWorkspace().getRoot(); 
					jadFile = root.getFile(storagePath.makeAbsolute());
					
					if ((jadFile != null) && (jadFile.exists())) {
						updateEditorInput();
					}
				}
				
			} catch (Exception e) {
				MtjCorePlugin.log(IStatus.WARNING, e);
			}
		} 
	}

	/**
	 * Return a boolean indicating whether the specified property key is 
	 * an allowable user-defined property.
	 * 
	 * @param key
	 * @return
	 */
	boolean isUserDefinedPropertyKey(String key) {
		return
			!requiredPropertiesEditorPage.isManagingProperty(key) &&
			!optionalPropertiesEditorPage.isManagingProperty(key) &&
			!otaPropertiesEditorPage.isManagingProperty(key) &&
			!midletsEditorPage.isManagingProperty(key);
	}

	/**
	 * Return a boolean indicating whether the user wants
	 * to reload the updated file.
	 * 
	 * @return
	 */
	private boolean shouldReloadFile() {
		return MessageDialog.openQuestion(
				getSite().getShell(), 
				"File Updated", 
				"The file has been updated.  Would you like to reload?");
	}

	/**
	 * Update the editor input.
	 * 
	 * @throws IOException
	 */
	private void updateEditorInput() {
		modificationStamp = jadFile.getModificationStamp();
		String filename = jadFile.getLocation().toOSString();  
		preferenceStore = 
			new ManifestPreferenceStore(filename);
		
		try {
			preferenceStore.load();
		} catch (IOException e) {
			MtjCorePlugin.log(IStatus.WARNING, e);
		}
	}
}
