/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/

package org.eclipse.mtj.extension.dplm.impl;

import java.util.ArrayList;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.EList;
import org.eclipse.mtj.api.deployment.Deployment;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.enumerations.DeviceCommunicationProtocol;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.extension.DeploymentManagement;
import org.eclipse.mtj.api.extension.DeploymentProvider;
import org.eclipse.mtj.api.extension.MtjExtension;
import org.eclipse.mtj.api.extension.impl.MtjExtensionImpl;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.core.MtjServices;
import org.eclipse.mtj.core.util.MtjEmfUtil;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.dplm.DeploymentManagementPlugin;
import org.eclipse.mtj.extension.dplm.Messages;

public class DeploymentManagementImpl extends MtjExtensionImpl implements DeploymentManagement {

	public DeploymentManagementImpl() {
		super();
		
		setId(DeploymentManagementPlugin.getDefault().getBundle().getSymbolicName());
		setDescription(Messages.DeploymentManagementImpl_PluginDescription);
		setVendor(Messages.DeploymentManagementImpl_PluginVendor);
		setVersion(Messages.DeploymentManagementImpl_PluginVersion);
		setType(ExtensionType.DEPLOYMENT_MANAGEMENT_LITERAL);
	}

	/**
	 * This method returns the Devices that could be deployment targets and that support the RuntimePlatformDefinition
	 * 
	 * @param runtimePlatformDefinition
	 * @return
	 * @throws MtjException
	 */
	public Device[] getTargetDevices(RuntimePlatformDefinition runtimePlatformDefinition) throws MtjException {
        Device[] devices = null;
        ArrayList devs = new ArrayList();
        if ( runtimePlatformDefinition != null) {
            EList eList = runtimePlatformDefinition.getDevices();
            for (int i = 0; i < eList.size(); i++) {
            	Device d = (Device)eList.get(i);
                DeploymentProvider dp = getDeploymentProvider(d);
                if ( dp != null ) {
                	devs.add(d);
                }
            }                          
        }
        devices = new Device[devs.size()];
        for (int i = 0; i < devs.size(); i++) {
            devices[i] = (Device)devs.get(i);
        }                          
        return devices; 
	}

	/**
	 * Method deploys the deployment -object to the target devices by using the defined transfer protocols.
	 * If transfer protocol parameter has null value, default transfer protocol is used.
	 * 
	 * @param deployment
	 * @param devices
	 * @param transferProtocols
	 * @param monitor
	 */
	public void deploy(Deployment deployment, Device[] devices,
			DeviceCommunicationProtocol transferProtocol,
			IProgressMonitor monitor) throws MtjException {
        DeploymentProvider deploymentProvider = null;
        for (int i = 0; i < devices.length; i++) {
            deploymentProvider = getDeploymentProvider(devices[i]);
            if(deploymentProvider == null){
            	throw new MtjException(Messages.DeploymentManagementImpl_DeploymentProviderNotDefined + MtjEmfUtil.toString(devices[i]));
            }
            deploymentProvider.deploy(deployment, devices, transferProtocol, monitor);            
        }
	}

    /**
     * This method returns DeploymentProvider that owns the Device
     * 
     * @param device
     * @return
     */
    public DeploymentProvider getDeploymentProvider(Device device){
        Device[] devices;
        MtjExtension[] deploymentProviderList = MtjServices.getInstance()
                    .getImplementations(ExtensionType.DEPLOYMENT_PROVIDER_LITERAL, null, null);        
        if(deploymentProviderList.length == 0) return null;
        for (int i = 0; i < deploymentProviderList.length; i++) {  
            try {
                devices = ((DeploymentProvider)deploymentProviderList[i]).getDevices();
                if(devices.length != 0){
                    for (int j = 0; j < devices.length; j++) { 
                        if( MtjEmfUtil.equals(device, devices[j]) ){
                            return (DeploymentProvider)deploymentProviderList[i];
                        }
                    }
                }
            } catch (MtjException e) {
                // TODO investigate what to do in case of non existent devices for a certain provider
                e.printStackTrace();
            }
        }
        return null;
    }

}
