/*******************************************************************************
* Copyright (c) 2006 Nokia Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.extension.pp;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.mtj.api.deployment.Deployment;
import org.eclipse.mtj.api.enumerations.DeploymentType;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.extension.PackagingProvider;
import org.eclipse.mtj.api.extension.impl.BuildExtensionImpl;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.api.model.packaging.MidpPackagingResources;
import org.eclipse.mtj.api.project.Project;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.devide.project.MtjProject;
import org.eclipse.mtj.extension.devide.utils.FilteringClasspathEntryVisitor;
import org.eclipse.mtj.internal.utils.EntryTrackingJarOutputStream;
import org.eclipse.mtj.internal.utils.Utils;

public class JavaMePackagingProviderImpl extends BuildExtensionImpl implements
		PackagingProvider {

	public static final String EXT_JAR_NAME = "packager.jar"; //$NON-NLS-1$
	private Properties midletProperties;
	
	public JavaMePackagingProviderImpl() {
		super();
		
		setId("org.eclipse.mtj.extension.pp.CDC"); //$NON-NLS-1$
		setDescription(Messages.JavaMePackagingProviderImpl_PluginDescription);
		setVendor(Messages.JavaMePackagingProviderImpl_PluginVendor);
		setVersion(Messages.JavaMePackagingProviderImpl_PluginVersion);
		setType(ExtensionType.PACKAGING_PROVIDER_LITERAL);
		
		setExtJar(EXT_JAR_NAME);
	}

	public DeploymentType[] getSupportedTypes() throws MtjException {
		DeploymentType[] ret = { DeploymentType.DEPLOYMENT_TYPE_CDC_LITERAL };
		return ret;
	}

	public void setMidletProperties(Properties midletProperties) {
		this.midletProperties = midletProperties;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.mtj.api.extension.PackagingProvider#createDeployment(java.lang.String, org.eclipse.mtj.api.project.Project, org.eclipse.core.resources.IResource[], org.eclipse.core.resources.IFolder, org.eclipse.mtj.api.enumerations.DeploymentType, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public Deployment createDeployment(String projectName, Project projectData, IResource[] resources,
			IFolder deploymentFolder, DeploymentType type, String natureId, IProgressMonitor monitor) throws MtjException {

		try {
			IFolder projectFolder = ResourcesPlugin.getWorkspace().getRoot().getFolder(
					ResourcesPlugin.getWorkspace().getRoot().getProject(projectName).getLocation());
			
			IFile finalJarFile = getJarFile(projectName, deploymentFolder, ""); //$NON-NLS-1$
			if (finalJarFile.exists()) finalJarFile.delete(true, monitor);
	
//			// If the user wants auto versioning to occur, do it now
//			boolean autoVersion = projectData.getPackagingDetails() != null ?
//				projectData.getPackagingDetails().isIncrementVersionAutomatically() : false;
//			if (autoVersion) {
//				updateJADVersion(projectName, projectFolder, monitor);
//			}
			
			MidpPackagingResources midpPackagingResources = new MidpPackagingResources(resources);
			IFolder verifiedClassesOutputFolder = midpPackagingResources.getVerifiedClassesOutputFolder();
			IFolder verifiedLibrariesOutputFolder = midpPackagingResources.getVerifiedLibrariesOutputFolder();
				
			// Do the actual build
			File deployedJar = createDeployedJarFile(projectName, projectData, 
					verifiedClassesOutputFolder, verifiedLibrariesOutputFolder,
					monitor, deploymentFolder, projectFolder, natureId);
		
			// Let Eclipse know about changes and mark
			// everything as derived.
			deploymentFolder.refreshLocal(IResource.DEPTH_ONE, monitor);
	
			
			return null;
		}
		catch (CoreException ex) {
			throw new MtjException(ex);
		}
		catch (IOException ex) {
			throw new MtjException(ex);
		}
	}
	
	/**
	 * Get the IFile instance into which the JAR will be
	 * written.
	 * 
	 * @return
	 */
	private IFile getJarFile(String projectName, IFolder deploymentFolder, String extension) {
		String jarFileName = projectName.replace(' ', '_') + extension + ".jar"; //$NON-NLS-1$

		return deploymentFolder.getFile(jarFileName);
	}

	/**
	 * Create the deployed JAR file.
	 * 
	 * @param deployedJarFile
	 * @throws IOException
	 */
	private File createDeployedJarFile(
			String projectName, 
			Project projectData, 
			IFolder verifiedLibrariesOutputFolder,
			IFolder verifiedClassesOutputFolder,
			IProgressMonitor monitor, 
			IFolder deploymentFolder,
			IFolder projectFolder,
			String natureId)
		throws CoreException, IOException 
	{
		// The jar file
		IFile jarIFile = getJarFile(projectName, deploymentFolder, ""); //$NON-NLS-1$
		
		File jarFile = resourceToFile(jarIFile);
		
		// Open up the JAR output stream to which the contents
		// will be added.
		FileOutputStream fos = new FileOutputStream(jarFile);
		EntryTrackingJarOutputStream jarOutputStream = 
			new EntryTrackingJarOutputStream(fos);
		
		try {
			IMtjProject mtjProject = MtjProject.getMtjProject(projectName);
			// Add the contents
			addVerifiedClasspathContentsToJar(jarOutputStream, projectName,
					verifiedLibrariesOutputFolder, verifiedClassesOutputFolder,
					natureId, monitor);
		} catch (MtjException e) {
			// TODO: handle exception
		}

		// All done with the initial jar.
		jarOutputStream.close();
		
		return resourceToFile(getJarFile(projectName, deploymentFolder, "")); //$NON-NLS-1$
	}

	/**
	 * Add the contents of the verified classpath to the jar.
	 * 
	 * @param jarOutputStream
	 * @param monitor
	 */
	private void addVerifiedClasspathContentsToJar(
		EntryTrackingJarOutputStream jarOutputStream, 
		String projectName,
		IFolder verifiedLibrariesOutputFolder,
		IFolder verifiedClassesOutputFolder,
		String natureId,
		IProgressMonitor monitor) 
			throws CoreException, IOException
	{   IJavaProject javaProject = getJavaProject(projectName);
		PackagerClasspathEntryVisitor visitor =
			new PackagerClasspathEntryVisitor(jarOutputStream, verifiedLibrariesOutputFolder,
					verifiedClassesOutputFolder);
		visitor.getRunner().run(javaProject, visitor, natureId, monitor);
	}

	private IJavaProject getJavaProject(String projectName) {
		return JavaCore.create(getProject(projectName));
	}
	
	private IProject getProject(String projectName) {
		return ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
	}
	
	/**
	 * Convert the specified resource handle to a File handle.
	 * 
	 * @param resource
	 * @return
	 */
	private File resourceToFile(IResource resource) {
		return new File(resource.getLocation().toOSString());
	}
	
	/**
	 * Add the resources to the output jar stream after filtering based on the
	 * specified resource filter.
	 * 
	 * @param jarOutputStream
	 * @param resourceFilter
	 * @param rootPath
	 * @param container
	 * @throws CoreException
	 * @throws IOException
	 */
	private void addFilteredResourcesToJar(
		EntryTrackingJarOutputStream jarOutputStream,
		IResourceFilter resourceFilter,
		IPath rootPath,
		IContainer container)
			throws CoreException, IOException
	{
		IResource[] members = container.members();
		for (int i = 0; i < members.length; i++) {
			IResource resource = members[i];
			if (resource instanceof IContainer) {
				IContainer cont = (IContainer) resource;
				if (resourceFilter.shouldTraverseContainer(cont)) {
					addFilteredResourcesToJar(jarOutputStream, resourceFilter, rootPath, cont);
				}
			} else if (resource.getType() == IResource.FILE) {
				if (resourceFilter.shouldBeIncluded((IFile) resource)) {
					addResourceToJar(jarOutputStream, rootPath, resource);
				}
			}
		}
	}

	/**
	 * 
	 * Add the specified resources to the JAR file.
	 * 
	 * @param jarOutputStream
	 * @param rootPath
	 * @param resource
	 * @throws IOException
	 */
	private void addResourceToJar(
		EntryTrackingJarOutputStream jarOutputStream, 
		IPath rootPath, 
		IResource resource) 
			throws IOException
	{
		// Figure the path of the JAR file entry
		IPath resourcePath = resource.getFullPath();
		int commonSegments = resourcePath.matchingFirstSegments(rootPath);
		IPath entryPath = resourcePath.removeFirstSegments(commonSegments);
		
		// Add the new entry
		File file = new File(resource.getLocation().toOSString());
		addFileToJar(jarOutputStream, entryPath.toString(), file);
	}

	/**
	 * Add the specified file contents to the jar file.
	 * 
	 * @param jarOutputStream
	 * @param string
	 * @param file
	 * @throws IOException
	 */
	private void addFileToJar(
		EntryTrackingJarOutputStream jarOutputStream, 
		String entryName, 
		File file) 
			throws IOException
	{
		// Create the ZipEntry to represent the file
		ZipEntry entry = new ZipEntry(entryName);
		entry.setSize(file.length());
		entry.setTime(file.lastModified());
		createJarEntry(jarOutputStream, entry, new FileInputStream(file));
	}

	/**
	 * Create a new Jar file entry given the specified information.
	 * 
	 * @param jarOutputStream
	 * @param zipEntry
	 * @param is
	 * @throws IOException
	 */
	private void createJarEntry(
		EntryTrackingJarOutputStream jarOutputStream, 
		ZipEntry zipEntry, 
		InputStream is)
			throws IOException
	{
		// Copy the zip entry before adding it to the output stream
		ZipEntry newEntry = new ZipEntry(zipEntry.getName());
		newEntry.setTime(zipEntry.getTime());
		if (zipEntry.getComment() != null) {
			newEntry.setComment(zipEntry.getComment());
		}
		if (zipEntry.getExtra() != null) {
			newEntry.setExtra(zipEntry.getExtra());
		}
		
		if (!jarOutputStream.alreadyAdded(zipEntry)) {
			// Add the new ZipEntry to the stream
			jarOutputStream.putNextEntry(newEntry);
			Utils.copyInputToOutput(is, jarOutputStream);
			
			// Close this entry
			jarOutputStream.closeEntry();
		}
	}

	/**
	 * IClasspathEntryVisitor that does the work necessary to
	 * package up the information based on the classpath.
	 */
	private class PackagerClasspathEntryVisitor extends FilteringClasspathEntryVisitor {
		private EntryTrackingJarOutputStream jarStream;
		private boolean visitedSource;
		private IFolder verifiedLibrariesOutputFolder;
		private IFolder verifiedClassesOutputFolder;
		
		/** Constructor */
		private PackagerClasspathEntryVisitor(EntryTrackingJarOutputStream jarStream, 
				IFolder verifiedClassesOutputFolder,
				IFolder verifiedLibrariesOutputFolder) {
			this.jarStream = jarStream;
			this.verifiedLibrariesOutputFolder = verifiedLibrariesOutputFolder;
			this.verifiedClassesOutputFolder = verifiedClassesOutputFolder;
			visitedSource = false;
		}
		
		/**
		 * @see eclipseme.core.internal.utils.IClasspathEntryVisitor#visitLibraryEntry(org.eclipse.jdt.core.IClasspathEntry, org.eclipse.jdt.core.IJavaProject, org.eclipse.core.runtime.IProgressMonitor)
		 */
		public void visitLibraryEntry(
			IClasspathEntry entry,
			IJavaProject javaProject, 
			IProgressMonitor monitor)
				throws CoreException 
		{
			File entryFile = Utils.getResolvedClasspathEntryFile(entry);
			if ((entryFile != null) && entryFile.isFile()) {
				
				IFile lib = verifiedLibrariesOutputFolder.getFile(entry.getPath().lastSegment());
				if (lib.exists()) {
					try {
						copyLibContentsToJar(jarStream, lib, monitor);
					} catch (IOException e) {
						MtjCorePlugin.throwCoreException(IStatus.ERROR, -999, e);
					}
				}
			}
		}
		
		/**
		 * @see eclipseme.core.internal.utils.IClasspathEntryVisitor#visitSourceEntry(org.eclipse.jdt.core.IClasspathEntry, org.eclipse.jdt.core.IJavaProject, org.eclipse.core.runtime.IProgressMonitor)
		 */
		public void visitSourceEntry(
			IClasspathEntry entry,
			IJavaProject javaProject, 
			IProgressMonitor monitor)
				throws CoreException 
		{
			if (!visitedSource) {
				visitedSource = true;
				
				try {
					addFilteredResourcesToJar(
						jarStream, 
						new IncludeAllResourceFilter(), 
						verifiedClassesOutputFolder.getFullPath(), 
						verifiedClassesOutputFolder);
				} catch (IOException e) {
					MtjCorePlugin.throwCoreException(IStatus.ERROR, -999, e);
				}
			}
		}

		/**
		 * Copy the contents of a library file to the jar output.
		 * 
		 * @param jarOutputStream
		 * @param lib
		 * @param monitor
		 */
		private void copyLibContentsToJar(
			EntryTrackingJarOutputStream jarOutputStream, 
			IFile lib, 
			IProgressMonitor monitor) 
				throws IOException
		{
			File libFile = lib.getLocation().toFile();
			ZipInputStream zis = new ZipInputStream(new FileInputStream(libFile));
			
			ZipEntry zipEntry = null;
			do {
				zipEntry = zis.getNextEntry();
				if (zipEntry != null) {
					createJarEntry(jarOutputStream, zipEntry, zis);
				}
			} while (zipEntry != null);

			zis.close();
		}
	}

	/**
	 * Resource filter that always includes everything.
	 */
	private class IncludeAllResourceFilter implements IResourceFilter {
		/**
		 * @see eclipseme.core.model.impl.Packager.IResourceFilter#shouldTraverseContainer(org.eclipse.core.resources.IContainer)
		 */
		public boolean shouldTraverseContainer(IContainer container) {
			return true;
		}

		/**
		 * @see eclipseme.core.model.impl.Packager.IResourceFilter#shouldBeIncluded(org.eclipse.core.resources.IFile)
		 */
		public boolean shouldBeIncluded(IFile file) {
			return true;
		}
	}

	/**
	 * This interface represents a filter controlling
	 * inclusion of resources into the packaged output.
	 * <p>
	 * <b>Note:</b> This class/interface is part of an interim API that is still under development and expected to
	 * change before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost 
	 * certainly be broken as the API evolves.
	 * </p>
	 * Copyright (c) 2004 Craig Setera<br>
	 * All Rights Reserved.<br>
	 * Licensed under the Eclipse Public License - v 1.0<p/>
	 * <br>
	 * $Revision: 1.1.1.1 $
	 * <br>
	 * $Date: 2006/08/30 18:51:44 $
	 * <br>
	 * @author Craig Setera
	 */
	public interface IResourceFilter {
		/**
		 * Return a boolean indicating whether the specified
		 * container should be traversed.
		 * 
		 * @param container the container to be traversed
		 * @return whether the container should be traversed
		 */
		public boolean shouldTraverseContainer(IContainer container);
		
		/**
		 * Return a boolean indicating whether the specified
		 * file should be included.
		 * 
		 * @param file the file to be tested
		 * @return whether the file should be included
		 */
		public boolean shouldBeIncluded(IFile file);
  }
	
}
