/*******************************************************************************
* Copyright (c) 2006 IBM.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/

package org.eclipse.mtj.extension.bp.eclipse.midp;

import java.util.Properties;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.search.IJavaSearchConstants;
import org.eclipse.jdt.core.search.IJavaSearchScope;
import org.eclipse.jdt.core.search.SearchEngine;
import org.eclipse.jdt.core.search.SearchMatch;
import org.eclipse.jdt.core.search.SearchParticipant;
import org.eclipse.jdt.core.search.SearchPattern;
import org.eclipse.jdt.core.search.SearchRequestor;
import org.eclipse.jdt.internal.core.SourceType;
import org.eclipse.mtj.api.enumerations.DeploymentType;
import org.eclipse.mtj.api.extension.PackagingProvider;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.core.ui.util.MidletSearchScope;
import org.eclipse.mtj.core.util.EclipseUtil;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.bp.eclipse.BuildProviderUtil;

public class MidpPackaging {
	
	private IMtjProject mtjProject;
	private IFolder binaryFolder;
	private IFolder binaryLibFolder;
	private IFolder deploymentFolder;
	private IProgressMonitor monitor;
	
	/**
	 * @param monitor
	 */
	public MidpPackaging(IMtjProject mtjProject, IProgressMonitor monitor) {
		super();
		this.mtjProject = mtjProject;
		this.monitor = monitor;
	}

	public void performPackaging(String natureId) throws MtjException, CoreException {
		DeploymentType dtype = BuildProviderUtil.getDeploymentType (mtjProject);
		PackagingProvider provider = BuildProviderUtil.getPackagingProvider (mtjProject, dtype);
			
		IResource[] outputResources = { getBinaryFolder(), getBinaryLibFolder() };
				
		IJavaProject project = mtjProject.getJavaProject();
		IClasspathEntry[] e = project.getRawClasspath();
		Properties p = new Properties();
		for(int i = 0; i < e.length; i++) {
			if (e[i].getEntryKind() == IClasspathEntry.CPE_SOURCE
				&& e[i].getContentKind() == IPackageFragmentRoot.K_SOURCE)
				addMidlets(p, e[i].getPath()) ;
		}
		
		provider.setMidletProperties(p);
		provider.createDeployment(mtjProject.getProject().getName(), mtjProject.getProjectData(), 
				outputResources, getDeploymentFolder(), dtype, natureId, monitor);
		
		EclipseUtil.setResourcesAsDerived(getDeploymentFolder());
	}

	private void addMidlets(Properties p, IPath path) throws MtjException, CoreException {
		IJavaProject project = mtjProject.getJavaProject();
		class Requestor extends SearchRequestor {
			private Properties p;
			
			public Requestor(Properties p) {
				this.p = p;
			}
			
			public void acceptSearchMatch(SearchMatch match) {
				if(match.getElement() instanceof SourceType) {
					String className = ((SourceType)match.getElement()).getElementName();
					String classPath = ((SourceType)match.getElement()).getFullyQualifiedName();
					p.setProperty("MIDlet-", className+", , "+classPath); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}	        
		}; 
		
		SearchPattern pattern = SearchPattern.createPattern("*", IJavaSearchConstants.TYPE, IJavaSearchConstants.DECLARATIONS, SearchPattern.R_PATTERN_MATCH | SearchPattern.R_CASE_SENSITIVE); //$NON-NLS-1$
		IJavaSearchScope scope = new MidletSearchScope(project);
	    // Search
	    SearchEngine searchEngine = new SearchEngine();
	    searchEngine.search(pattern, new SearchParticipant[] {SearchEngine.getDefaultSearchParticipant()}, scope, new Requestor(p), null);		
	}
	
	/**
	 * @return Returns the deploymentFolder.
	 * @throws CoreException 
	 */
	public IFolder getDeploymentFolder() throws CoreException {
		if (deploymentFolder == null)
		{
			deploymentFolder = mtjProject.getDeploymentFolder(monitor);
			if (deploymentFolder.exists())
				deploymentFolder.delete( true, monitor);
			deploymentFolder.create(true, true, monitor);
		}
		if (!deploymentFolder.exists())
			deploymentFolder.create(true, true, monitor);
		return deploymentFolder;
	}

	/**
	 * @param deploymentFolder The deploymentFolder to set.
	 * @throws CoreException 
	 */
	public void setDeploymentFolder(IFolder deploymentFolder) throws CoreException {
		if (deploymentFolder.exists())
			deploymentFolder.delete( true, monitor);
		deploymentFolder.create(true, true, monitor);

		this.deploymentFolder = deploymentFolder;
	}

	/**
	 * @return Returns the binaryFolder.
	 * @throws CoreException 
	 */
	public IFolder getBinaryFolder() throws CoreException {
		if (binaryFolder == null)
			binaryFolder = mtjProject.getVerifiedClassesOutputFolder(monitor);

		return binaryFolder;
	}

	/**
	 * @param binaryFolder The binaryFolder to set.
	 * @throws CoreException 
	 */
	public void setBinaryFolder(IFolder outputFolder) {
		this.binaryFolder = outputFolder;
	}

	/**
	 * @return Returns the binaryLibFolder.
	 * @throws CoreException 
	 */
	public IFolder getBinaryLibFolder() throws CoreException {
		if (binaryLibFolder == null)
			binaryLibFolder = mtjProject.getVerifiedLibrariesOutputFolder(monitor);

		return binaryLibFolder;
	}

	/**
	 * @param binaryLibFolder The binaryLibFolder to set.
	 */
	public void setBinaryLibFolder(IFolder binaryLibFolder) {
		this.binaryLibFolder = binaryLibFolder;
	}
}
