/*******************************************************************************
* Copyright  2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.extension.dm.ui.preferences;

import java.util.Hashtable;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.preference.DirectoryFieldEditor;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.extension.DevicePlatformProvider;
import org.eclipse.mtj.core.toolkit.ToolkitSupport;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.dm.ui.Messages;
import org.eclipse.mtj.extension.dm.ui.util.DevicePlatformImportContentProvider;
import org.eclipse.mtj.extension.dm.ui.util.PreferenceUtils;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Shell;

public class AddDevicePlatformDialog2 extends TitleAreaDialog {	
	private final static String NOKIA_IMPORT_PROVIDER = "org.eclipse.mtj.extension.dpp.nokia"; //$NON-NLS-1$
	private final static String UEI_IMPORT_PROVIDER = "org.eclipse.mtj.extension.dpp.uei"; //$NON-NLS-1$

	private DirectoryFieldEditor rootEditor;
	private CheckboxTreeViewer dpTree;
	private Hashtable dppList;
	private Composite mainContainer;
	private Composite subContainer;
	private Composite dpContainer;
	private StyledText deviceDescription;
	private PreferenceUtils utils;
	private DevicePlatform[] myDps;
	private ToolkitSupport myToolkit;
	private String myTitle;
	private String mainMessage; //Select provider and enter root
	private String defaultMessage;
	private String rootMessage; //Enter root location
	private String dpRetrieveFailure; //DP could not be created
	private String errorInProperties; //TODO GUI Proto
	private Button checkAllButton;
	private Button unCheckAllButton;
	
	private String titleBarLabel = null;

	public AddDevicePlatformDialog2(Shell parentShell, String title) {		
		super(parentShell);
		setShellStyle(getShellStyle() | SWT.MAX | SWT.RESIZE);
		myTitle = title;
		utils = new PreferenceUtils();
	}
	
	protected Control createDialogArea(Composite parent) {
		mainContainer = utils.createDefaultComposite(parent, 1, false);
		GridData data = (GridData) mainContainer.getLayoutData();
		data.grabExcessVerticalSpace = true;

		createProviderSelectionArea(mainContainer);
		utils.createHorizontalSeparator(mainContainer);
		subContainer = utils.createDefaultComposite(mainContainer, 2, false);

		//Device Platforms tree view
		dpContainer = (Composite) createPlatformSelectionArea(subContainer);
		data = (GridData) subContainer.getLayoutData();
		data.grabExcessVerticalSpace = true;
		data.grabExcessVerticalSpace = true;
		
		initData();
		return mainContainer;
	}
	
	private Control createProviderSelectionArea(Composite parent){		
		Composite container = utils.createDefaultComposite(parent, 1, false);
		//Platform Provider
		createProviderSelection(container);
		//TODO  Root location checking
		createRootSelection(container);
		return container;
	}	
	
	private Control createProviderSelection(Composite parent){		
		Composite container = utils.createDefaultComposite(parent, 2, false);
		return container;
	}
	

	/** Directory selection editor for runtime platform root */
	private Control createRootSelection(Composite parent) {
		//Top most part panel with location field editor
		Composite container = utils.createDefaultComposite(parent, 1, false);
		//Root editor
		rootEditor = new DevicePlatformDirectoryEditor("rootLocation", "Location", container); //$NON-NLS-1$ //$NON-NLS-2$
		rootEditor.setPropertyChangeListener(new IPropertyChangeListener(){
			public void propertyChange(PropertyChangeEvent event) {
				checkRootLocationChange();
			}
		});

		return container;
	}
	
	private Control createPlatformSelectionArea(Composite parent){
		Group platformContainer = utils.createDefaultGroup(parent, 1, false);
		platformContainer.setText(Messages.AddDevicePlatformDialog2_DevicePlatforms);
		GridData data = (GridData) platformContainer.getLayoutData();
		data.grabExcessHorizontalSpace = true;
		data.grabExcessVerticalSpace = true;
		
		//DevicePlatform tree
		dpTree = utils.createDevicePlatformCheckTree(platformContainer);
		dpTree.addSelectionChangedListener(new ISelectionChangedListener(){
			public void selectionChanged(SelectionChangedEvent event) {
				// TODO
				changeDescriptionData();
			}			
		});
		
		dpTree.addCheckStateListener(new ICheckStateListener(){			
			/* If device platform is not checked -> set children unchecked and grayed.
			 * If all children are unchecked -> uncheck device platform and gray children.
			 */
			public void checkStateChanged(CheckStateChangedEvent event) {			
				Object element = event.getElement();
				ITreeContentProvider contentProvider = (DevicePlatformImportContentProvider)dpTree.getContentProvider();
				
				if(element instanceof DevicePlatform){				
					boolean checkAndClear = event.getChecked();					
					Object[] children = contentProvider.getChildren(element);
					
//					dpTree.setSubtreeChecked(element, checkAndClear);
					for(int i = 0; i < children.length; i++){
						dpTree.setChecked(children[i], checkAndClear);
						dpTree.setGrayed(children[i], !checkAndClear);
					}
				} else if(element instanceof Device){					
					//If this is grayed, disallow checking
					if(dpTree.getGrayed(element)){
						dpTree.setChecked(element, false);
						return;
					}
					
					/* Get parent and all siblings to check whether this is the 
				   	last checked. If yes -> uncheck the platform and gray children */
					DevicePlatform dp = (DevicePlatform) ((ITreeContentProvider)dpTree.getContentProvider()).getParent(element);
					Device[] children = (Device[]) dp.getDevices().toArray();
					
					if(event.getChecked()){						
					} else {
						boolean checkedFound = false;
						for(int i = 0; !checkedFound && i < children.length; i++){
							if(dpTree.getChecked(children[i])){
								checkedFound = true;
							}
						}
						if(!checkedFound){
							dpTree.setChecked(dp, false);
							for(int i = 0; i < children.length; i++){
								dpTree.setGrayed(children[i], true);
							}
						}
					}
				}
			}			
			});
		
		return platformContainer;
	}
	
	private void setAllElementsChecked(boolean checked){		
		//TODO check why first element always cheched when all are unckecked
		ITreeContentProvider contentProvider = (ITreeContentProvider)dpTree.getContentProvider();
		Object[] elements = contentProvider.getElements(dpTree.getInput());
		
		if(checked){	
			dpTree.setCheckedElements(elements);
			
			Object[] children = null;
			for(int i = 0; i < elements.length; i++){
				children = contentProvider.getChildren(elements[i]);
				
				for(int j = 0; j < children.length; j++){
					dpTree.setChecked(children[j], true);
					dpTree.setGrayed(children[j], false);
				}
			}
		}else{
			//Check only first temporarily and the uncheck
			Object[] tmp = new Object[]{elements[0]};
			dpTree.setCheckedElements(tmp);
			dpTree.setChecked(tmp, false);
		}
	}

	private Composite createDescriptionArea(Composite parent) {		
		Group container = utils.createDefaultGroup(parent, 1, false);
		container.setText(Messages.AddDevicePlatformDialog2_Description);
		
		deviceDescription = utils.createStyledText(container, 60);
		deviceDescription.setBackground(parent.getBackground());
		deviceDescription.setEditable(false);					
		return container;							
	}
	
	public void setTitleBarLabel (String label)
	{
		titleBarLabel = new String (label);
	}
	
	@Override
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		if (titleBarLabel != null)
			newShell.setText(titleBarLabel);
	}
	
	private void clearDetailsArea(){		
	}

	/** Ok and Cancel buttons. */
	protected Control createButtonBar(Composite parent) {
		Control tmp = super.createButtonBar(parent);
		//Disable OK until root location and contents validated
		getButton(IDialogConstants.OK_ID).setEnabled(false);
		return tmp;
	}
	
	private void changePlatform(String dppName){		
		//TODO DPP specific handling
	}

	/** Creates a dropdown list of Device Platform Providers */
	private void setProviderList(){		
		DevicePlatformProvider[] dpps = utils.getToolkitSupporters();
		dppList = new Hashtable();
		String dppName = null;
		for(int i = 0; i < dpps.length; i++){
			dppName = utils.getObjectName(dpps[i]);
			dppList.put(dppName, dpps[i]); //store dpps for selection change
		}
	}

	private void checkPlatformProviderChange(){		
		//TODO Check if there is a need to change something
		isPlatformValid();
		String errorMsg = rootEditor.getStringValue();
		if(errorMsg == null || errorMsg.trim().length() == 0){
			setMessage(rootMessage, IMessageProvider.INFORMATION);
		}
	}
	
	private void checkRootLocationChange() {	
		//If valid -> change contents else display error msg
		if (isPlatformValid()) {			
			//TODO retrieve the DevicePlatforms and create new objects
			createDevicePlatforms();
			
			if (myDps[0] == null) {
				setMessage(dpRetrieveFailure, IMessageProvider.ERROR);
			} else {				
				subContainer.setEnabled(true);
				
				//Set new platforms
				//Something fishy with tree impl
				dpTree.setInput(new DevicePlatform[] { myDps[0]});
				dpTree.setInput(myDps);
				dpTree.refresh();
				
				setAllElementsChecked(true);
				createDescriptionArea (subContainer);
				deviceDescription.setText(""); //$NON-NLS-1$
// Bug 159278: These buttons are not being created or enabled prior to this use, so I have commented them out
// They also have no labels, or listeners to do anything with them if they exist.
//				checkAllButton.setEnabled(true);
//				unCheckAllButton.setEnabled(true);
				dpContainer.layout();
				subContainer.layout();
				mainContainer.layout();
				getContents().pack();
				isPlatformValid();
			}
		}
	}
	
	/** Fill in the details data based on the device platform selection */
	private void changeDescriptionData() {
		Object  o = utils.getSingleSelection(dpTree.getSelection());
		deviceDescription.setText(utils.getObjectStyledName(o));
	}
	
	private void createDevicePlatforms() {		
		try {
			Shell activeShell = new Shell();
			IRunnableWithProgress op = new IRunnableWithProgress() {
				public void run(IProgressMonitor monitor) {
					monitor.beginTask(Messages.AddDevicePlatformDialog2_Reading_DevicePlatform_Properties, 100);
					monitor.worked(5);
					// DevicePlatformProvider dpp = getProvider(NOKIA_IMPORT_PROVIDER);
					DevicePlatformProvider dpp = getProvider(UEI_IMPORT_PROVIDER);
					DevicePlatform[] dps = utils.createDevicePlatformFromDirectory((ToolkitSupport)dpp, rootEditor.getStringValue());
					monitor.worked(40);
					if(dps == null){
						setMessage(errorInProperties, IMessageProvider.ERROR);
					} else{
						myDps = new DevicePlatform[dps.length];
						System.arraycopy(dps, 0, myDps, 0, dps.length);
						myToolkit = (ToolkitSupport)dpp;
					}
					monitor.worked(100);
					monitor.done();
				}
			};
			new ProgressMonitorDialog(this.getShell()).run(false, false, op);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}	

	public static DevicePlatformProvider getProvider(String id) {
		PreferenceUtils utils = new PreferenceUtils();
		DevicePlatformProvider[] dpps = utils.getToolkitSupporters();
		
		for(DevicePlatformProvider dpp: dpps) {
			if(dpp.getId().equals(id)) {
				return dpp;
			}		
		}
		return null;
	}

	
	private boolean isPlatformValid() {		
		if(!rootEditor.isValid()){
			getButton(IDialogConstants.OK_ID).setEnabled(false);
			setMessage(rootEditor.getErrorMessage(), IMessageProvider.ERROR);
			return false;
		}
		
		getButton(IDialogConstants.OK_ID).setEnabled(true);
		getButton(IDialogConstants.OK_ID).setFocus();
		setMessage(null);
		
		return true;
	}
	
	private void initData(){		
		defaultMessage = ""; //$NON-NLS-1$
		mainMessage = Messages.AddDevicePlatformDialog2_SelectPlatformProvdierRootLocation;
		rootMessage = Messages.AddDevicePlatformDialog2_EnterRootLocation;
		dpRetrieveFailure = Messages.AddDevicePlatformDialog2_DevicePlatformCouldNotBeRetreived;
	
		//TODO remove this, GUI proto
		errorInProperties = Messages.AddDevicePlatformDialog2_ErrorReadingEmulatorProperties;
		
		setTitle(myTitle);
		setMessage(mainMessage, IMessageProvider.INFORMATION);
	
		//Provider list
		setProviderList();
		subContainer.setEnabled(false);
	}	
	
	public DevicePlatform[] getDevicePlatforms() {	
		return myDps;
	}
	
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	@Override
	protected void okPressed() {		
		//TODO Select only those dps that are checked
		//DevicePlatform dpsToImport
		
		//TODO create default Runtimes for each device. 
		//NOTE: Check that addition should be ignored if user Cancels in the main page
		
		//TODO do not save additions in this dialog to psp -> only after main pref page OK selected	
		try {
			DevicePlatform[] dps = myToolkit.getDevicePlatforms();
			DevicePlatform[] r = new DevicePlatform[dps.length + myDps.length];
			int ix = 0;
			for (int i = 0; i< dps.length; i++) {
				r[ix] = dps[i];
				ix++;
			}
			for (int i = 0; i< myDps.length; i++) {
				//myDps[i].setName(nameText.getText());
				r[ix] = myDps[i];
				ix++;
			}
			myToolkit.setDevicePlatforms(r);
			
			super.okPressed();
		} catch (MtjException e) {
			try {
				myToolkit.setDevicePlatforms(new DevicePlatform[0]);				
			} catch (Exception ee) {
				setMessage(e.getLocalizedMessage(), IMessageProvider.ERROR);
			}
		}
	}
}


