/*******************************************************************************
* Copyright  2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/

package org.eclipse.mtj.extension.rpm.preferences;

import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.extension.DeviceManagement;
import org.eclipse.mtj.api.project.TargetDevice;
import org.eclipse.mtj.api.runtimes.DeviceConfiguration;
import org.eclipse.mtj.api.runtimes.DeviceProfile;
import org.eclipse.mtj.api.runtimes.RuntimePlatform;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.api.runtimes.ServiceApi;
import org.eclipse.mtj.core.MtjServices;
import org.eclipse.mtj.core.ui.util.DialogController;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.rpm.Messages;
import org.eclipse.mtj.extension.rpm.model.MtjServiceHandler;
import org.eclipse.mtj.extension.rpm.ui.preferences.ModifyDeviceComposite;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

interface ApiConstants {
	String IMP = "IMP";  //$NON-NLS-1$
	String IMP_NG = "IMP-NG";  //$NON-NLS-1$
	String MIDP = "MIDP";  //$NON-NLS-1$
	String CLDC = "CLDC";  //$NON-NLS-1$
	String CDC = "CDC"; 	 //$NON-NLS-1$
}

public class ModifyDeviceController implements ApiConstants, DialogController, ICheckStateListener {	
	private ModifyDeviceComposite composite;
	private RuntimePlatform selectedRuntime;
	private MtjServiceHandler mtjServiceHandler;
	private TargetDevice targetDevice;
	private RuntimePlatformDefinition rpd; 
	private RuntimePlatformDefinition dprpd;		
	private Device selectedDevice; 
	private DeviceConfiguration selectedConfiguration;
	private DeviceProfile selectedProfile;
	private CheckboxTableViewer optionalApiTableViewer;
			
	public ModifyDeviceController(ModifyDeviceComposite composite, MtjServiceHandler mtjServiceHandler, RuntimePlatform selectedRuntime) {
		this.composite = composite;
		this.selectedRuntime = selectedRuntime;
		this.mtjServiceHandler = mtjServiceHandler;
		
		this.optionalApiTableViewer = composite.getOptionalApiTableViewer();
		this.optionalApiTableViewer.addCheckStateListener(this);
		
		this.targetDevice = (TargetDevice)selectedRuntime.getTargetDevices().get(0);
		if(targetDevice != null) {
			this.rpd = targetDevice.getRuntimePlatform();
			this.dprpd = mtjServiceHandler.getRuntimePlatformDefinition(targetDevice.getDevicePlatformName(), targetDevice.getDeviceName());
		}
		
		setData(composite, selectedRuntime);
		createProfileAndConfiguration();
		createServiceApi();
	}
	
	private static void setData(ModifyDeviceComposite c, RuntimePlatform s) {
		c.getRPNameLabel().setText(s.getName());
		
		EList targetList = s.getTargetDevices();
		TargetDevice targetDevice = null;
		if (targetList.size() >= 1) {
			targetDevice = (TargetDevice) targetList.get(0);		
			c.getDeviceNameLabel().setText(targetDevice.getName());
			RuntimePlatformDefinition rpd = targetDevice.getRuntimePlatform();
			DeviceProfile defaultProfile = (DeviceProfile) rpd.getDeviceProfiles().get(0);
			DeviceConfiguration selectedConfiguration = (DeviceConfiguration) rpd.getDeviceConfigurations().get(0);

			DeviceManagement dm =  (DeviceManagement)MtjServices.getInstance().getImplementations(ExtensionType.DEVICE_MANAGEMENT_LITERAL, null, null)[0];
			try {
				DevicePlatform myDp = dm.getDevicePlatformByName(targetDevice.getDevicePlatformName());
			} catch (MtjException e) {
				e.printStackTrace();
			}
		}
	}
	
	public void checkStateChanged(CheckStateChangedEvent event) {
		boolean checkState = event.getChecked();
		Object o = event.getElement();
		if(o instanceof ServiceApi) {
			ServiceApi api = (ServiceApi) o;

			// "JSR-205" "JSR-120"
			if(checkState && api.getName().equals("JSR120")) { //$NON-NLS-1$
				uncheckApi(this.dprpd.getServiceApis(), "JSR205"); //$NON-NLS-1$
				this.optionalApiTableViewer.setChecked(api, true);
			}
			if(checkState && api.getName().equals("JSR205")) { //$NON-NLS-1$
				uncheckApi(this.dprpd.getServiceApis(), "JSR120"); //$NON-NLS-1$
				this.optionalApiTableViewer.setChecked(api, true);
			}
			
			/*
			else if(checkState && api.getName().equals(MMAPI)) {
				uncheckApi(this.dprpd.getServiceApis(), MMAPI);
				this.optionalApiTableViewer.setChecked(api, true);
			}
			*/
		}						
	}

	private void uncheckApi(EList l, String value) {
		for(int i = 0; i < l.size(); i++) {
			ServiceApi a = (ServiceApi)l.get(i);
			if(a.getName().equals(value)) {
				this.optionalApiTableViewer.setChecked(a, false);
			}
		}		
	}
	
	private boolean containsProfile(String profileName, String profileVersion, Iterator iterator) {		
		while(iterator.hasNext()) {	
			DeviceProfile dp = (DeviceProfile)iterator.next();
			if(dp.getName().equals(profileName) && dp.getVersion().equals(profileVersion)) {
				return true;
			}
		}
		return false;
	}
	
	private boolean containsConfiguration(String configurationName, String configurationVersion, Iterator iterator) {		
		while(iterator.hasNext()) {	
			DeviceConfiguration dc = (DeviceConfiguration)iterator.next();
			if(dc.getName().equals(configurationName) && dc.getVersion().equals(configurationVersion)) {
				return true;
			}
		}
		return false;
	}
	
	private void createProfileAndConfiguration() {		
		EList dpProfiles = dprpd.getDeviceProfiles();
		EList dpConfigs = dprpd.getDeviceConfigurations();
		EList profiles = rpd.getDeviceProfiles();
		EList configs = rpd.getDeviceConfigurations();

		Group pg = composite.getProfileGroup();
		Group cg = composite.getConfigGroup();
		
		if(profiles.size() >= 2) {
			Iterator i = profiles.iterator();
			DeviceProfile selectedProfile = null;
			while(i.hasNext()) {
				DeviceProfile dp = (DeviceProfile) i.next();
				if(selectedProfile == null) selectedProfile = dp;
				else {
					selectedProfile = Double.parseDouble(selectedProfile.getVersion()) > Double.parseDouble(dp.getVersion()) 
						? selectedProfile: dp; 				
				}
			}
			profiles.clear();
			profiles.add(selectedProfile);
		}
		if(configs.size() >= 2) {
			Iterator i = configs.iterator();
			DeviceConfiguration selectedConfiguration = null;
			while(i.hasNext()) {
				DeviceConfiguration dc = (DeviceConfiguration) i.next();
				if(selectedConfiguration == null) selectedConfiguration = dc;
				else {
					selectedConfiguration = Double.parseDouble(selectedConfiguration.getVersion()) > Double.parseDouble(dc.getVersion()) 
							? selectedConfiguration: dc;
				}
			}
			configs.clear();
			configs.add(selectedConfiguration);
		}
		
		if(!containsProfile(IMP,"1.0",dpProfiles.iterator())) { //$NON-NLS-1$
			Button button = createRadioButton(pg, IMP+" 1.0");			  //$NON-NLS-1$
			button.setEnabled(false);
		}
		if(!containsProfile(IMP_NG,"1.0",dpProfiles.iterator())) { //$NON-NLS-1$
			Button button = createRadioButton(pg, IMP_NG+" 1.0");			  //$NON-NLS-1$
			button.setEnabled(false);
		}
		if(!containsProfile(MIDP,"1.0",dpProfiles.iterator())) { //$NON-NLS-1$
			Button button = createRadioButton(pg, MIDP+" 1.0");			  //$NON-NLS-1$
			button.setEnabled(false);
		}
		if(!containsProfile(MIDP,"2.0",dpProfiles.iterator())) { //$NON-NLS-1$
			Button button = createRadioButton(pg, MIDP+" 2.0");			  //$NON-NLS-1$
			button.setEnabled(false);
		}
		
		if(!containsConfiguration(CDC,"1.0",dpConfigs.iterator())) {  //$NON-NLS-1$
			Button button = createRadioButton(cg, CDC+" 1.0"); //$NON-NLS-1$
			button.setEnabled(false);
		}
		if(!containsConfiguration(CDC,"1.1",dpConfigs.iterator())) {  //$NON-NLS-1$
			Button button = createRadioButton(cg, CDC+" 1.1"); //$NON-NLS-1$
			button.setEnabled(false);
		}
		if(!containsConfiguration(CLDC,"1.0",dpConfigs.iterator())) {  //$NON-NLS-1$
			Button button = createRadioButton(cg, CLDC+" 1.0"); //$NON-NLS-1$
			button.setEnabled(false);
		}
		if(!containsConfiguration(CLDC,"1.1",dpConfigs.iterator())) {  //$NON-NLS-1$
			Button button = createRadioButton(cg, CLDC+" 1.1"); //$NON-NLS-1$
			button.setEnabled(false);
		}

		// profiles
		for(int i = 0; i < dpProfiles.size(); i++) {
			DeviceProfile dp = (DeviceProfile) dpProfiles.get(i);
			String name = dp.getName();
			String version = dp.getVersion();
			Button button = createRadioButton(pg, name+" "+version); //$NON-NLS-1$
			button.setData(dp);
			button.addSelectionListener(new SelectionAdapter(){
				public void widgetSelected(SelectionEvent event) {
					selectedProfile = (DeviceProfile)((Button)event.getSource()).getData();
				}
			});

			if(containsProfile(name,version,profiles.iterator())) {
				button.setSelection(true);
			}
		}
		
		// configuration
		for(int i = 0; i < dpConfigs.size(); i++) {
			DeviceConfiguration dc = (DeviceConfiguration) dpConfigs.get(i);
			String name = dc.getName();
			String version = dc.getVersion();
			Button button = createRadioButton(cg, name+" "+version); //$NON-NLS-1$
			button.setData(dc);
			button.addSelectionListener(new SelectionAdapter(){
				public void widgetSelected(SelectionEvent event) {
					selectedConfiguration = (DeviceConfiguration)((Button)event.getSource()).getData();
				}
			});

			if(containsConfiguration(name,version,configs.iterator())) {
				button.setSelection(true);				
			}
		}		
	}
	
	private static void createDeviceList(CheckboxTableViewer deviceTableViewer, TargetDevice targetDevice, MtjServiceHandler mtjServiceHandler) {
		class tlp extends LabelProvider implements ITableLabelProvider {
			public Image getColumnImage(Object element, int columnIndex) {
				  return null;
			}
			
			public String getColumnText(Object element, int columnIndex) {
				if(element instanceof Device) {
					Device device = (Device) element;
					if(columnIndex == 0) { return ""; } //$NON-NLS-1$
					else if(columnIndex == 1) { return device.getName(); }
					else if(columnIndex == 2) { return device.getDescription(); }
				}
				return null;
			}			
		}
		
		class EListContentProvider implements IStructuredContentProvider 
		{
			public Object[] getElements(Object o) {
				if(o instanceof EList) {
					return ((EList)o).toArray();
				}
				return null;
			}
			
			public void dispose() {
			}
			
			public void inputChanged(Viewer viewer,
	                Object oldInput,
	                Object newInput) {
			}
		}

		deviceTableViewer.setContentProvider(new EListContentProvider());
		deviceTableViewer.setLabelProvider(new tlp());

		createTableColumn(deviceTableViewer.getTable(),0,6,""); //$NON-NLS-1$
		createTableColumn(deviceTableViewer.getTable(),1,20,Messages.ModifyDeviceController_DeviceTableViewerName);
		createTableColumn(deviceTableViewer.getTable(),2,40,Messages.ModifyDeviceController_DeviceTableViewerDescription);

		if(targetDevice != null) {
			EList dl = mtjServiceHandler.getDeviceList(targetDevice.getDevicePlatformName());
			deviceTableViewer.setInput(dl);		
			
			Iterator iterator = dl.iterator();
			while (iterator.hasNext()) {
				Device d = (Device) iterator.next();
				if(d.getName().equals(targetDevice.getDeviceName())) {
					deviceTableViewer.setChecked(d,true);
					break;
				}
			}
		}
	}
	
	private void createServiceApi() {
		class tlp extends LabelProvider implements ITableLabelProvider {
			public Image getColumnImage(Object element, int columnIndex) {
				  return null;
			}
			
			public String getColumnText(Object element, int columnIndex) {
				if(element instanceof ServiceApi) {
					ServiceApi api = (ServiceApi) element;
					if(columnIndex == 0) { return ""; } //$NON-NLS-1$
					else if(columnIndex == 1) { return api.getName(); }
					else if(columnIndex == 2) { return api.getVersion(); }
					else if(columnIndex == 3) { return api.getDescription(); }
				}
				return null;
			}			
		}
		
		this.optionalApiTableViewer.setContentProvider(new EListContentProvider());
		this.optionalApiTableViewer.setLabelProvider(new tlp());
		
		createTableColumn(this.optionalApiTableViewer.getTable(),0,2,""); //$NON-NLS-1$
		createTableColumn(this.optionalApiTableViewer.getTable(),1,10,Messages.ModifyDeviceController_OptionalApiTableViewerName);
		createTableColumn(this.optionalApiTableViewer.getTable(),2,6,Messages.ModifyDeviceController_OptionalApiTableViewerVersion);
		createTableColumn(this.optionalApiTableViewer.getTable(),3,17,Messages.ModifyDeviceController_OptionalApiTableViewerDescription);

		removeDuplicates(rpd);
		
		if(dprpd != null) {
			setApis(rpd, dprpd);
		}
		else {
			EList rpApiList = new BasicEList(rpd.getServiceApis());
			removeSubItems(rpApiList);
			
			this.optionalApiTableViewer.setInput(rpApiList);
			this.optionalApiTableViewer.setAllChecked(true);
		}
	}

	private static void removeSubItems(EList rpApiList) {
		if(checkIfContainsApi(rpApiList, "JSR177")) { //$NON-NLS-1$
			removeApi(rpApiList, "JSR177/APDU"); //$NON-NLS-1$
			removeApi(rpApiList, "JSR177/CRYPTO"); //$NON-NLS-1$
			removeApi(rpApiList, "JSR177/JCRMI"); //$NON-NLS-1$
			removeApi(rpApiList, "JSR177/PKI"); //$NON-NLS-1$
		}
		if(checkIfContainsApi(rpApiList, "JSR82")) { //$NON-NLS-1$
			removeApi(rpApiList, "JSR82/BT"); //$NON-NLS-1$
			removeApi(rpApiList, "JSR82/OBEX"); //$NON-NLS-1$
		}
		if(checkIfContainsApi(rpApiList, "JSR75")) { //$NON-NLS-1$
			removeApi(rpApiList, "JSR75/FC"); //$NON-NLS-1$
			removeApi(rpApiList, "JSR75/PIM");				 //$NON-NLS-1$
		}		
	}
	
	private static void removeApi(EList list,String apiName) {
		for(Object o: list) {
			ServiceApi api = (ServiceApi) o;
			if(api.getName().equals(apiName)) {
				list.remove(api);
				break;
			}
		}
	}
	
	private static boolean checkIfContainsApi(EList list, String apiName) {
		for(Object o: list) {
			ServiceApi api = (ServiceApi) o;
			if(api.getName().equals(apiName))
				return true;
		}
		
		return false;
	}
	
	private static void removeDuplicates(RuntimePlatformDefinition rpd) {
		EList l = rpd.getServiceApis();
		EList removeList = new BasicEList();
		
		ServiceApi wmaApi = null; 
		ServiceApi mmapiApi = null; 
		
		for(int i = 0; i < l.size(); i++) {
			ServiceApi api = (ServiceApi) l.get(i);
			
			// WMA
			if(api.getName().equals("JSR205") || api.getName().equals("JSR120")) { //$NON-NLS-1$ //$NON-NLS-2$
				if(wmaApi == null) {
					wmaApi = api;
				} 
				else {
					if( Double.parseDouble(api.getVersion()) > Double.parseDouble(wmaApi.getVersion())) {
						removeList.add(wmaApi);
						wmaApi = api;						
					} 
					else {
						removeList.add(api);						
					}
				}
			}

			// MMAPI
			/*
			else if(api.getName().equals("JSR-135")) {
				if(mmapiApi == null) {
					mmapiApi = api;
				} 
				else {
					if( Double.parseDouble(api.getVersion()) > Double.parseDouble(mmapiApi.getVersion())) {
						removeList.add(mmapiApi);
						mmapiApi = api;						
					} 
					else {
						removeList.add(api);						
					}				
				}
			}
			*/
		}
		
		for(int i = 0; i < removeList.size(); i++) {
			l.remove(removeList.get(i));
		}
	}
	
	private void setApis(RuntimePlatformDefinition rpd, RuntimePlatformDefinition dprpd) {
		EList rpApiList = new BasicEList(rpd.getServiceApis());
		EList dpApiList = new BasicEList(dprpd.getServiceApis());
		removeSubItems(rpApiList);
		removeSubItems(dpApiList);

		this.optionalApiTableViewer.setInput(dpApiList);
		this.optionalApiTableViewer.setAllChecked(false);

		for(int i = 0; i < dpApiList.size(); i++) {
			ServiceApi api = (ServiceApi) dpApiList.get(i);			
			if(getServiceApi(api.getName(), api.getVersion(), rpApiList.iterator()) != null) {
				this.optionalApiTableViewer.setChecked(api, true);
			}
		}		
	}
	
	private ServiceApi getServiceApi(String name, String version, Iterator apiIterator) {
		while(apiIterator.hasNext()) {
			ServiceApi api = (ServiceApi) apiIterator.next();
			if(name.equals(api.getName()) && version.equals(api.getVersion())) {
				return api;
			}
		}
		return null;
	}
	
	private static void createTableColumn(Table table, int colIndex, int weight, String header) {
		ColumnWeightData cwd = new ColumnWeightData(weight);
		TableColumn tableColumn = new TableColumn(table, SWT.NONE, colIndex);
		tableColumn.setResizable(cwd.resizable);
		tableColumn.setText(header);				

		TableLayout tableLayout = (TableLayout) table.getLayout();
		tableLayout.addColumnData(cwd);
	}
	
	private static ServiceApi[] createServiceApiList(DeviceProfile selectedProfile, RuntimePlatformDefinition rpd, ModifyDeviceComposite c){		
		//Service API selection
		ServiceApi[] services = getCleanedServiceApis(rpd.getServiceApis(), selectedProfile.getServiceApis());
		int count = services.length;
		Button[] serviceButtons = new Button[count];
		StringBuffer sb = null;
		String label = null;
		for(int i = 0; i < count; i++){
			sb = new StringBuffer();
			sb.append(services[i].getName());
			sb.append("-"); //$NON-NLS-1$
			sb.append(services[i].getVersion());
			sb.append(" ("); //$NON-NLS-1$
			sb.append(services[i].getDescription());
			sb.append(")"); //$NON-NLS-1$
			label = sb.toString();
			serviceButtons[i] = createCheckButton(c.getServiceGroup(), label);
			serviceButtons[i].setSelection(true);
			serviceButtons[i].setData(services[i]);
		}
		
		return services;
	}
	
	public static Button createCheckButton(Composite parent, String label){		
		Button button = new Button(parent, SWT.CHECK);
		button.setFont(parent.getFont());
		if (label != null) {
			button.setText(label);
		}
		
		return button;
	}

	public static ServiceApi[] getCleanedServiceApis(EList serviceApis, EList profileApis){		
		Vector v = new Vector<ServiceApi>();
		
		ServiceApi tmpApi = null;
		for(int i = 0; i < serviceApis.size(); i++){			
			tmpApi = (ServiceApi)serviceApis.get(i);			
			v.add(tmpApi);
		}
		
		Object[] o = v.toArray();
		ServiceApi[] sapis = new ServiceApi[o.length];
		for(int i = 0; i < o.length; i++){			
			sapis[i] = (ServiceApi)o[i];
		}
		
		return sapis;
	}
	
	private static String getPlatformTypeName(DevicePlatform dp){		
		if(dp != null){			
			StringTokenizer st = new StringTokenizer(dp.getType().getName(), "_"); //$NON-NLS-1$
			StringBuffer sb = new StringBuffer();
			String s = null;
			while(st.hasMoreTokens()){
				
				s = st.nextToken();
				
				sb.append(s.charAt(0));
				sb.append(s.substring(1).toLowerCase());
				sb.append(" "); //$NON-NLS-1$
			}
			
			return sb.toString().trim();
		}
		else{
			return new String();
		}
	}
	
	public void handleOK() {			
		if(this.rpd != null) {
			// configuration
			EList l = this.rpd.getDeviceConfigurations();
			if(selectedConfiguration != null) {
				l.clear();
				l.add(selectedConfiguration);
			}
			// profile
			l = this.rpd.getDeviceProfiles();
			if(selectedProfile != null) {
				l.clear();			
				l.add(selectedProfile);
			}
			// api
			l = this.rpd.getServiceApis();
			l.clear();
			Object[] o = this.optionalApiTableViewer.getCheckedElements();
			for(int i = 0; i < o.length; i++) {
				l.add(o[i]);
			}
		}
	}
	
	public static Button createRadioButton(Group parent, String label) {		
		Button button = new Button(parent, SWT.RADIO);
		button.setFont(parent.getFont());
		if (label != null){
			button.setText(label);
		}
		return button;	
	}

	class EListContentProvider implements IStructuredContentProvider 
	{
		public Object[] getElements(Object o) {
			if(o instanceof EList) {
				return ((EList)o).toArray();
			}
			return null;
		}
		
		public void dispose() {
		}
		
		public void inputChanged(Viewer viewer,
                Object oldInput,
                Object newInput) {
		}
	}
}
