/*******************************************************************************
* Copyright  2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/

package org.eclipse.mtj.extension.rpm.preferences;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.project.TargetDevice;
import org.eclipse.mtj.api.runtimes.RuntimePlatform;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.extension.rpm.Messages;
import org.eclipse.mtj.extension.rpm.model.MtjServiceHandler;
import org.eclipse.mtj.extension.rpm.model.RpmUIManager;
import org.eclipse.mtj.extension.rpm.ui.preferences.RuntimeListComposite;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

public class RpmController implements SelectionListener, ISelectionChangedListener, ICheckStateListener {
	private static Logger log = Logger.getLogger(RpmController.class.getName());
	private RuntimeListComposite runtimeListComposite;
	private MtjServiceHandler mtjServiceHandler = new MtjServiceHandler();
	private RuntimePlatform selectedRuntime;	
	
	private Shell containerShell = new Shell();	
	private Button addButton;
	private Button editRuntimeButton;
	private Button editDeviceButton;
	private Button removeButton;
	private TableViewer runtimeTableViewer;
	private CheckboxTableViewer deviceTableViewer;
	
	public RpmController(RuntimeListComposite runtimeListComposite) {		
		this.runtimeListComposite = runtimeListComposite;

		addButton = runtimeListComposite.getAddButton();
		editRuntimeButton = runtimeListComposite.getEditRuntimeButton();
		editDeviceButton = runtimeListComposite.getEditDeviceButton();
		removeButton = runtimeListComposite.getRemoveButton();
		runtimeTableViewer = runtimeListComposite.getRuntimeTableViewer();
		deviceTableViewer = runtimeListComposite.getDeviceTableViewer();
		
		setData();
		addListener();

		selectFirstRuntime();
	}
	
	/**
	 * Cleans values from ui 
	 */
	public void setDefaults() {
		// TODO
		runtimeTableViewer.setInput(new RuntimePlatform[0]);
		deviceTableViewer.setInput(new BasicEList());
	}
	
	private void selectFirstRuntime() {
		if(runtimeTableViewer.getTable().getItemCount() > 0) {
			runtimeTableViewer.getTable().setSelection(0);			
			runtimeSelected();
		}		
	}
	
	private static TableColumn createTableColumn(String title, Table table, int index, int widthHint) {
		TableColumn tableColumn = new TableColumn(table, SWT.NONE, index);
		tableColumn.setResizable(true);
		tableColumn.setWidth(widthHint);
		tableColumn.setText(title);
		return tableColumn;
	}
	
	private void setData() {
		runtimeTableViewer.setContentProvider(new RuntimePlatformContentProvider(runtimeTableViewer));
		runtimeTableViewer.setSorter(new RuntimePlatformSorter(1));		
		runtimeTableViewer.setLabelProvider(new RuntimePlatformLabelProvider());

		createTableColumn(Messages.RpmController_RuntimeTableViewerName, runtimeTableViewer.getTable(), 0, 100);
		createTableColumn(Messages.RpmController_RuntimeTableViewerDescription, runtimeTableViewer.getTable(), 1, 200);

		class tlp extends LabelProvider implements ITableLabelProvider {
			public Image getColumnImage(Object element, int columnIndex) {
				  return null;
			}
			
			public String getColumnText(Object element, int columnIndex) {
				if(element instanceof Device) {
					Device device = (Device) element;
					if(columnIndex == 0) { return ""; } //$NON-NLS-1$
					else if(columnIndex == 1) { return device.getName(); }
					else if(columnIndex == 2) { return device.getDescription(); }
				}
				return null;
			}			
		}
		
		class EListContentProvider implements IStructuredContentProvider 
		{
			public Object[] getElements(Object o) {
				if(o instanceof EList) {
					return ((EList)o).toArray();
				}
				return null;
			}
			
			public void dispose() {
			}
			
			public void inputChanged(Viewer viewer,
	                Object oldInput,
	                Object newInput) {
			}
		}

		deviceTableViewer.setContentProvider(new EListContentProvider());
		deviceTableViewer.setLabelProvider(new tlp());
		
		createTableColumn("",deviceTableViewer.getTable(),0,20); //$NON-NLS-1$
		createTableColumn(Messages.RpmController_DeviceTableViewerName,deviceTableViewer.getTable(),1,160);
		createTableColumn(Messages.RpmController_DeviceTableViewerDescription,deviceTableViewer.getTable(),2,160);
		
		// Data
		runtimeTableViewer.setInput(mtjServiceHandler.getRuntimePlatforms());		
		
		setButtonsEnabled(false);
	}
	
	private void addListener() {
		addButton.addSelectionListener(this);
		editRuntimeButton.addSelectionListener(this);
		editDeviceButton.addSelectionListener(this);
		removeButton.addSelectionListener(this);
		this.runtimeTableViewer.addSelectionChangedListener(this);
		this.deviceTableViewer.addCheckStateListener(this);
	}

	public void widgetDefaultSelected(SelectionEvent e) {}
	public void widgetSelected(SelectionEvent e) {
		if(e.widget == addButton) {
			addRuntimePlatform();
		} else if(e.widget == editRuntimeButton) {
			editRuntimePlatform(selectedRuntime);
		} else if(e.widget == editDeviceButton) {
			editDevicePlatform(selectedRuntime);
		} else if(e.widget == removeButton) {
			removeRuntimePlatform();
		}
	}

	private void runtimeSelected() {
		ISelection selection = runtimeTableViewer.getSelection();
		if (!selection.isEmpty()) {			
			log.info("RpmController TableItemSelected"); //$NON-NLS-1$			
			log.finest(Messages.RpmController_TableItemSelected);
			IStructuredSelection structured = (IStructuredSelection) selection;
			Object element = structured.getFirstElement();
			if(element instanceof RuntimePlatform){
				log.info("RuntimePlatform: isnull "+ (element == null)); //$NON-NLS-1$
				log.info("RuntimePlatform: "+((RuntimePlatform)element).getName()); //$NON-NLS-1$
				selectedRuntime = (RuntimePlatform)element;
				setDevices();
			}
			setButtonsEnabled(true);
		} else {
			log.finest(Messages.RpmController_TableItemDeselected);
			setButtonsEnabled(false);
		}		
	}
	
	private void setButtonsEnabled(boolean enabled) {
		editRuntimeButton.setEnabled(enabled);
		editDeviceButton.setEnabled(enabled);
		removeButton.setEnabled(enabled);				
	}
	
	public void selectionChanged(SelectionChangedEvent event) {
		if(event.getSource() == runtimeTableViewer) {
			runtimeSelected();
		}		
	}
	
	private void setDevices() {
		TargetDevice targetDevice = (TargetDevice)selectedRuntime.getTargetDevices().get(0);
		createDeviceList(deviceTableViewer, targetDevice, mtjServiceHandler);		
	}
	
	private void addRuntimePlatform() {
		if(RpmUIManager.addRuntimePlatform(this.runtimeListComposite.getShell(), (RuntimePlatform[])runtimeTableViewer.getInput(), mtjServiceHandler, false) == Dialog.OK) {
			RuntimePlatform newRp = mtjServiceHandler.getAddedRuntime();
			((RuntimePlatformContentProvider) runtimeTableViewer.getContentProvider()).addVariable(runtimeTableViewer.getInput(), newRp);
			selectFirstRuntime();
		}
	}

	private void editRuntimePlatform(RuntimePlatform rp) {
		if(RpmUIManager.editRuntimePlatform(selectedRuntime, (RuntimePlatform[])runtimeTableViewer.getInput(), runtimeListComposite) == Dialog.OK) {
			this.runtimeTableViewer.refresh();
			this.setDevices();	
		}		
	}
	
	private void editDevicePlatform(RuntimePlatform rp) {
		RpmUIManager.editDevicePlatform(rp, runtimeListComposite);
	}
	
	private void removeRuntimePlatform() {	
		if(selectedRuntime == null) 
			return;

		MessageDialog dialog = new MessageDialog(
				runtimeListComposite.getShell(),
				Messages.RpmController_Confirm,null,
				Messages.RpmController_ConfirmDescription,
				MessageDialog.QUESTION, 
				new String[] {Messages.RpmController_Ok,Messages.RpmController_Cancel},0);
		
		int value = dialog.open();
		runtimeTableViewer.getTable().setFocus();
		if(value == 1) 
			return;

		Object oldInput = runtimeTableViewer.getInput();
		Object newInput = ((RuntimePlatformContentProvider)runtimeTableViewer.getContentProvider()).removeVariable(oldInput, selectedRuntime);
		runtimeTableViewer.remove(selectedRuntime);			
		runtimeTableViewer.setInput(newInput);
		runtimeTableViewer.refresh();
		
		// cleaning devices away
		deviceTableViewer.setInput(new BasicEList());				
		
		RuntimePlatform rp = (RuntimePlatform) runtimeTableViewer.getElementAt(0);
		if (rp != null) {
			runtimeTableViewer.getTable().setSelection(0);
			selectedRuntime = rp;
			setDevices();
			editDeviceButton.setEnabled(true);
		}
	}
	
	public RuntimePlatform[] getRuntimePlatforms() {
		RuntimePlatform[] runtimes = (RuntimePlatform[]) runtimeTableViewer.getInput();
		 return runtimes;
	}
	
	private static void createDeviceList(CheckboxTableViewer deviceTableViewer, TargetDevice targetDevice, MtjServiceHandler mtjServiceHandler) {		
		if(targetDevice != null) {
			EList dl = mtjServiceHandler.getDeviceList(targetDevice.getDevicePlatformName());
			deviceTableViewer.setInput(dl);		
			
			Iterator iterator = dl.iterator();
			while (iterator.hasNext()) {
				Device d = (Device) iterator.next();
				if(d.getName().equals(targetDevice.getDeviceName())) {
					deviceTableViewer.setChecked(d,true);
					break;
				}
			}
		}
	}	
	
	public void checkStateChanged(CheckStateChangedEvent event) {
		boolean checkState = event.getChecked();
		Object o = event.getElement();
		if(o instanceof Device) {
			Device d = (Device) o;
			this.deviceTableViewer.setAllChecked(false);
			this.deviceTableViewer.setChecked(d, true);	
			if(checkState)	
				setRuntimeDevice(d);			
		}						
	}

	private void setRuntimeDevice(Device device) {
		String dpString = ((TargetDevice) selectedRuntime.getTargetDevices().get(0)).getDevicePlatformName();
		DevicePlatform dp = mtjServiceHandler.getDevicePlatform(dpString);
		TargetDevice td = mtjServiceHandler.createTargetDevice(dp, (RuntimePlatformDefinition) device.getRuntimePlatformDefinitions().get(0), device); 
		EList l = selectedRuntime.getTargetDevices();
		l.clear();
		l.add(td);
	}
}

class RuntimePlatformContentProvider implements IStructuredContentProvider {	
	private TableViewer myViewer;
	private RuntimePlatform[] myInput;
	
	public RuntimePlatformContentProvider(TableViewer viewer){		
		myViewer = viewer;
	}

	public Object[] getElements(Object inputElement) {
		if(inputElement instanceof RuntimePlatform[]){
			return (RuntimePlatform[])inputElement;
		}
		else if(inputElement instanceof EList){
			return (RuntimePlatform[])((EList)inputElement).toArray();
		}
		
		return null;
	}
	
	
	public Object addVariable(Object inputElement, RuntimePlatform newRp) {		
		Object newInput = null;
		
		EList eList = null;
		
		// Create list if no input
		if (inputElement == null) {
			
			inputElement = new RuntimePlatform[1];
			((RuntimePlatform[])inputElement)[0] = newRp;
			
			newInput = inputElement;
		}
		else{
			if(inputElement instanceof EList){
				eList = (EList)inputElement;
				
				((EList)inputElement).add(newRp);
				
				newInput = inputElement;
			}
			else if(inputElement instanceof RuntimePlatform[]){
				
				RuntimePlatform[] oldInput = (RuntimePlatform[])inputElement;
				
				RuntimePlatform[] newRpInput = new RuntimePlatform[oldInput.length + 1];
				for(int i = 0; i < oldInput.length; i++){
					newRpInput[i] = oldInput[i];
				}
				newRpInput[oldInput.length] = newRp;
				
				newInput = newRpInput;
			}
			
		}
		
		myViewer.setInput(newInput);
		
		return newInput;
	}
	
	/**
	 * Remove the selected runtime platform from this model and return new input.
	 * 
	 * @param oldInputElement
	 * @param selectedRp
	 * @return new input
	 */
	public RuntimePlatform[] removeVariable(Object oldInputElement, RuntimePlatform selectedRp){		
		if(oldInputElement instanceof RuntimePlatform[]){

			RuntimePlatform[] oldInput = (RuntimePlatform[])oldInputElement;
			if(oldInput.length == 1){
				return null;
			}
			
			List list = new ArrayList();
			for(int i = 0; i < oldInput.length; i++){
				if(oldInput[i].getName().intern() != selectedRp.getName().intern()){
					list.add(oldInput[i]);
				}
			}
			
			RuntimePlatform[] newInput = new RuntimePlatform[list.size()];
			
			return (RuntimePlatform[])list.toArray(newInput);
		}
		else if(oldInputElement instanceof EList){
			
			EList list = (EList)oldInputElement;
			
			if(list.size() == 1){
				return null;
			}
			
			list.remove(selectedRp);
			return (RuntimePlatform[]) list.toArray();
		}
		
		return null;
	}

	public void dispose() {
		// TODO Auto-generated method stub
	}

	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {		
		myViewer = (TableViewer) viewer;
		
		if(newInput instanceof EList){
			myInput = (RuntimePlatform[])((EList)newInput).toArray();
		}
		else if(newInput instanceof RuntimePlatform[]){
			myInput = (RuntimePlatform[]) newInput;
		}
		else if(newInput instanceof RuntimePlatform){
			myInput = new RuntimePlatform[1];
			myInput[0] = (RuntimePlatform)newInput;
		}
		else{
			myInput = null;
		}
	}
}

class RuntimePlatformSorter extends ViewerSorter {	
	private boolean reversed = false;
	private int column;
	
	public RuntimePlatformSorter(int column) {
		super();
		this.column = column;
	}
	
	public int compare(Viewer viewer, Object o1, Object o2) {

		RuntimePlatform rp1 = (RuntimePlatform)o1;
		RuntimePlatform rp2 = (RuntimePlatform)o2;
		
		return compare(rp1.getName(), rp2.getName());
	}
	
	protected int compare(String task1, String task2) {
		int result = collator.compare(task1, task2);
		if (reversed){
			result = -result;
		}
		return result;
	}
	
	public boolean isReversed() {
		return reversed;
	}

	public void setReversed(boolean b) {
		reversed = b;
	}
}

class RuntimePlatformLabelProvider extends LabelProvider implements ITableLabelProvider {
	
	public RuntimePlatformLabelProvider(){
		super();
	}

	public Image getColumnImage(Object element, int columnIndex) {
		// TODO Auto-generated method stub
		return null;
	}

	public String getColumnText(Object element, int index) {
		
		if(element instanceof RuntimePlatform){
			
			RuntimePlatform rp = (RuntimePlatform)element;
			
			switch (index) {
			case 0:
				return rp.getName();
			case 1:
				return rp.getDescription();
			default:
				return new String();
			}
		}
		
		return null;
	}


}

