/*******************************************************************************
* Copyright  2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/

package org.eclipse.mtj.extension.dm.preferences;

import java.util.Iterator;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.runtimes.DeviceConfiguration;
import org.eclipse.mtj.api.runtimes.DeviceProfile;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.api.runtimes.ServiceApi;
import org.eclipse.mtj.core.ui.util.DialogController;
import org.eclipse.mtj.extension.dm.Messages;
import org.eclipse.mtj.extension.dm.ui.preferences.DeviceInfoComposite;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;

public class DeviceInfoController implements DialogController {
	private Device selectedDevice;
	private DeviceInfoComposite composite;
	private Tree apiTree; 
	
	public DeviceInfoController(Device selectedDevice, DeviceInfoComposite composite) {
		this.composite = composite;
		this.selectedDevice = selectedDevice;
		
		this.apiTree = composite.getApiTree();
		setData();
	}
	
	private void setData() {
		composite.getDeviceNameLabel().setText(selectedDevice.getName());
		composite.getDeviceDescriptionLabel().setText(selectedDevice.getDescription());

		EList l = selectedDevice.getRuntimePlatformDefinitions();
		RuntimePlatformDefinition rpd = (RuntimePlatformDefinition)l.get(0);
		
		createProfile(rpd);
		createConfiguration(rpd);
		createServiceApi(rpd);
	}

	private static void removeSubItems(EList rpApiList) {
		if(checkIfContainsApi(rpApiList, "JSR177")) { //$NON-NLS-1$
			removeApi(rpApiList, "JSR177/APDU"); //$NON-NLS-1$
			removeApi(rpApiList, "JSR177/CRYPTO"); //$NON-NLS-1$
			removeApi(rpApiList, "JSR177/JCRMI"); //$NON-NLS-1$
			removeApi(rpApiList, "JSR177/PKI"); //$NON-NLS-1$
		}
		if(checkIfContainsApi(rpApiList, "JSR82")) { //$NON-NLS-1$
			removeApi(rpApiList, "JSR82/BT"); //$NON-NLS-1$
			removeApi(rpApiList, "JSR82/OBEX"); //$NON-NLS-1$
		}
		if(checkIfContainsApi(rpApiList, "JSR75")) { //$NON-NLS-1$
			removeApi(rpApiList, "JSR75/FC"); //$NON-NLS-1$
			removeApi(rpApiList, "JSR75/PIM");				 //$NON-NLS-1$
		}		
	}
	
	private static void removeApi(EList list,String apiName) {
		for(Object o: list) {
			ServiceApi api = (ServiceApi) o;
			if(api.getName().equals(apiName)) {
				list.remove(api);
				break;
			}
		}
	}
	
	private static boolean checkIfContainsApi(EList list, String apiName) {
		for(Object o: list) {
			ServiceApi api = (ServiceApi) o;
			if(api.getName().equals(apiName))
				return true;
		}
		
		return false;
	}

	private void createServiceApi(RuntimePlatformDefinition rpd) {
		apiTree.setHeaderVisible(true);
		apiTree.setLinesVisible(true);
		
		EList rpApiList = new BasicEList(rpd.getServiceApis());
		removeSubItems(rpApiList);
		
		TreeColumn nameColumn = new TreeColumn(apiTree, SWT.LEFT);
		TreeColumn versionColumn = new TreeColumn(apiTree, SWT.CENTER);
		TreeColumn descriptionColumn = new TreeColumn(apiTree, SWT.LEFT);

		nameColumn.setText(Messages.DeviceInfoController_Name);
		nameColumn.setWidth(110);
		versionColumn.setText(Messages.DeviceInfoController_Version);
		versionColumn.setWidth(50);
		descriptionColumn.setText(Messages.DeviceInfoController_Description);
		descriptionColumn.setWidth(150);
		
		Iterator<ServiceApi> iterator = rpApiList.iterator();
		while(iterator.hasNext()) {
			ServiceApi api = iterator.next();
			TreeItem item = new TreeItem(apiTree, SWT.NONE);
			item.setData(api);
			item.setText(new String [] {api.getName(),api.getVersion(),api.getDescription()});
		}
		
		apiTree.setSize(100, 100);
	}
	
	public static Button createRadioButton(Group parent, String label) {		
		Button button = new Button(parent, SWT.CHECK);
		button.setFont(parent.getFont());
		if (label != null){
			button.setText(label);
		}
		return button;	
	}
	
	private boolean containsProfile(String profileName, String profileVersion, Iterator iterator) {		
		while(iterator.hasNext()) {	
			DeviceProfile dp = (DeviceProfile)iterator.next();
			if(dp.getName().equals(profileName) && dp.getVersion().equals(profileVersion)) {
				return true;
			}
		}
		return false;
	}
	
	private boolean containsConfiguration(String configurationName, String configurationVersion, Iterator iterator) {		
		while(iterator.hasNext()) {	
			DeviceConfiguration dc = (DeviceConfiguration)iterator.next();
			if(dc.getName().equals(configurationName) && dc.getVersion().equals(configurationVersion)) {
				return true;
			}
		}
		return false;
	}
	
	private void createProfile(RuntimePlatformDefinition rpd) {
		EList l = rpd.getDeviceProfiles();
		Group group = composite.getProfileGroup();

		createProfileIfNeeded("IMP","1.0",l,group); //$NON-NLS-1$ //$NON-NLS-2$
		createProfileIfNeeded("IMP-NG","1.0",l,group); //$NON-NLS-1$ //$NON-NLS-2$
		createProfileIfNeeded("MIDP","1.0",l,group); //$NON-NLS-1$ //$NON-NLS-2$
		createProfileIfNeeded("MIDP","2.0",l,group); //$NON-NLS-1$ //$NON-NLS-2$
				
		for(int i = 0; i < l.size(); i++) {
			DeviceProfile dp = (DeviceProfile) l.get(i);
			String dps = dp.getName()+" "+dp.getVersion(); //$NON-NLS-1$
			Button button = createRadioButton(group, dps);
			addCheckButtonSelectionListener(button);
		}	
	}
	
	private static void addCheckButtonSelectionListener(Button button) {
		button.setSelection(true);
		button.addSelectionListener(new SelectionListener() {
			public void widgetDefaultSelected(SelectionEvent e) {}
			public void widgetSelected(SelectionEvent e) {
				((Button)e.getSource()).setSelection(true);
			}
		});		
	}
	
	private void createConfiguration(RuntimePlatformDefinition rpd) {
		EList l = rpd.getDeviceConfigurations();
		Group group = composite.getConfigGroup();
		
		createConfigurationIfNeeded("CLDC","1.0",l,group); //$NON-NLS-1$ //$NON-NLS-2$
		createConfigurationIfNeeded("CLDC","1.1",l,group); //$NON-NLS-1$ //$NON-NLS-2$
		createConfigurationIfNeeded("CDC","1.0",l,group); //$NON-NLS-1$ //$NON-NLS-2$
		createConfigurationIfNeeded("CDC","1.1",l,group); //$NON-NLS-1$ //$NON-NLS-2$
		
		for(int i = 0; i < l.size(); i++) {
			DeviceConfiguration dc = (DeviceConfiguration) l.get(i);
			String dcs = dc.getName()+" "+dc.getVersion(); //$NON-NLS-1$
			Button button = createRadioButton(group, dcs);
			addCheckButtonSelectionListener(button);
		}
	}

	private void createProfileIfNeeded(String api, String version, EList l, Group group) {
		if(!containsProfile(api,version,l.iterator())) {
			Button button = createRadioButton(group, api+" "+version); //$NON-NLS-1$
			button.setEnabled(false);
		}		
	}

	private void createConfigurationIfNeeded(String api, String version, EList l, Group group) {
		if(!containsConfiguration(api,version,l.iterator())) {
			Button button = createRadioButton(group, api+" "+version); //$NON-NLS-1$
			button.setEnabled(false);
		}		
	}
	
	public void handleOK() {
		// TODO
	}		
}

