/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 

package org.eclipse.mtj.core;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IStatusHandler;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

/**
 * The main plugin class to be used in the desktop.
 */
public class MtjCorePlugin extends AbstractUIPlugin implements IEclipseMtjCoreConstants {

	//The shared instance.
	private static MtjCorePlugin plugin;
	
	/** Status code for which a UI prompter is registered. */
	private static final IStatus PROMPTER_STATUS = new Status(
			IStatus.INFO,
			"org.eclipse.debug.ui",	// TODO This should probably not be done.... //$NON-NLS-1$
			200, "", null); //$NON-NLS-1$
	
	/**
	 * The constructor.
	 */
	public MtjCorePlugin() {
		plugin = this;
	}

	/**
	 * This method is called upon plug-in activation
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
	}

	/**
	 * This method is called when the plug-in is stopped
	 */
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
		plugin = null;
	}

	/**
	 * Returns the shared instance.
	 */
	public static MtjCorePlugin getDefault() {
		return plugin;
	}

	/**
	 * Returns an image descriptor for the image file at the given
	 * plug-in relative path.
	 *
	 * @param path the path
	 * @return the image descriptor
	 */
	public static ImageDescriptor getImageDescriptor(String path) {
		return AbstractUIPlugin.imageDescriptorFromPlugin("Mtj_Core", path); //$NON-NLS-1$
	}
	
	public static String getId() {
		return "org.eclipse.mtj.core"; //$NON-NLS-1$
	}
	
	/**
	 * Throw a new CoreException wrapped around the specified
	 * exception.
	 * 
	 * @param severity
	 * @param code
	 * @param exception
	 * @throws CoreException
	 */
	public static void throwCoreException(
		int severity,
		int code,
		Throwable exception)
			throws CoreException 
	{
		// Make sure we create a valid status object
		String message = null;
		if (exception != null) { 
			message = exception.getMessage();
		}
		if (message == null) message = Messages.MtjCorePlugin_NoMessage;
		
		IStatus status = new Status(
			severity,
			IEclipseMtjCoreConstants.PLUGIN_ID,
			code,
			message,
			exception);
		throw new CoreException(status);		
	}
	
	/**
	 * Log the specified message.
	 */
	public static void log(int severity, String message) {
		log(severity, message, null);
	}
	
	/**
	 * Log the specified exception.
	 */
	public static void log(int severity, Throwable throwable) {
		log(severity, throwable.getMessage(), throwable);
	}
	
	/**
	 * Log the specified message and exception
	 */
	public static void log(int severity, String message, Throwable throwable) {
		MtjCorePlugin plugin = MtjCorePlugin.getDefault();
		String id = IEclipseMtjCoreConstants.PLUGIN_ID;
		Status status =
			new Status(severity, id, IStatus.OK, message, throwable);
		plugin.getLog().log(status);
	}

	/**
	 * Returns the workspace instance.
	 */
	public static IWorkspace getWorkspace() {
		return ResourcesPlugin.getWorkspace();
	}

	/**
	 * Throw a new CoreException wrapped around the specified
	 * String.
	 * 
	 * @param severity
	 * @param code
	 * @param message
	 * @throws CoreException
	 */
	public static void throwCoreException(
		int severity,
		int code,
		String message)
			throws CoreException 
	{
		if (message == null) message = Messages.MtjCorePlugin_NoMessage;

		IStatus status = new Status(
			severity,
			IEclipseMtjCoreConstants.PLUGIN_ID,
			code,
			message,
			null);
		throw new CoreException(status);		
	}

	/**
	 * Attempt to prompt on a status object.  If prompting fails,
	 * a CoreException will be thrown.
	 * 
	 * @param status
	 * @param source
	 * @return
	 * @throws CoreException
	 */
	public static Object statusPrompt(IStatus status, Object source)
		throws CoreException
	{
		Object result = null;
		
		IStatusHandler prompterStatus = 
			DebugPlugin.getDefault().getStatusHandler(PROMPTER_STATUS);
				
		if (prompterStatus == null) {
			// if there is no handler, throw the exception
			throw new CoreException(status);
		} else {
			result = prompterStatus.handleStatus(status, source);
		} 
	
		return result;
	}

	/**
	 * This method returns active shell
	 * @return active shell
	 */
	public static Shell getActiveShell()
	{
		if(plugin == null)
			return null;
		IWorkbench workBench = plugin.getWorkbench();
		if(workBench == null)
			return null;
		IWorkbenchWindow workBenchWindow = workBench.getActiveWorkbenchWindow();
		if(workBenchWindow == null)
			return null;
		else
			return workBenchWindow.getShell();
	}
	
	
	
	/**
	 * Creates a new status object for our plugin. The created status has no
	 * children.
	 * 
	 * @param severity
	 *            the severity; one of <code>OK</code>, <code>ERROR</code>,
	 *            <code>INFO</code>, or <code>WARNING</code>
	 * @param code
	 *            the plug-in-specific status code, or <code>OK</code>
	 * @param message
	 *            a human-readable message, localized to the current locale
	 */
	public static IStatus newStatus(int severity, int code, String message) {
		return newStatus(severity, code, message, null);
	}

	/**
	 * Creates a new status object for our plugin. The created status has no
	 * children.
	 * 
	 * @param severity
	 *            the severity; one of <code>OK</code>, <code>ERROR</code>,
	 *            <code>INFO</code>, or <code>WARNING</code>
	 * @param code
	 *            the plug-in-specific status code, or <code>OK</code>
	 * @param message
	 *            a human-readable message, localized to the current locale
	 * @param exception
	 *            a low-level exception, or <code>null</code> if not
	 *            applicable
	 */
	public static IStatus newStatus(int severity, int code, String message,
			Throwable exception) {
		return new Status(severity, PLUGIN_ID, code, message, exception);
	}
	/**
	 * Recursively set the resources in the specified container and all
	 * resources within that container as derived.
	 * 
	 * @param container
	 * @throws CoreException
	 */
	public static void setResourcesAsDerived(IContainer container)
			throws CoreException {
		if (container.exists()) {
			// Mark this folder first...
			container.setDerived(true);

			// Recursively handle the members of the directory
			IResource[] resources = container.members();
			for (int i = 0; i < resources.length; i++) {
				IResource resource = resources[i];
				if (resource instanceof IContainer) {
					setResourcesAsDerived((IContainer) resource);
				} else {
					resource.setDerived(true);
				}
			}
		}
	}
	public static Object getConfigurationSpecification(
			String identifier) throws CoreException {
		return null;
	}	
	
}
