/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.api.extension;

import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DeviceDescription;
import org.eclipse.mtj.api.devices.DeviceGroup;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.enumerations.PlatformType;
import org.eclipse.mtj.api.extension.MtjExtension;
import org.eclipse.mtj.api.runtimes.DeviceConfiguration;
import org.eclipse.mtj.api.runtimes.DeviceProfile;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.api.runtimes.ServiceApi;
import org.eclipse.mtj.exception.MtjException;

/**
 * Device Management is responsible to offer services to fetch Device Platforms, Devices and Device Descriptions. 
 * It combines data from Device Platform Providers, Device Description Providers and Deployment Providers.
 * </p>
 * $Revision: 1.1.1.1 $
 * <br>
 * $Date: 2006/08/30 18:50:20 $
 * <br>
 */
public interface DeviceManagement extends MtjExtension {

	/**
	 * Method returns all existing device platforms.
	 * 
	 * @return
	 */
	public DevicePlatform[] getDevicePlatforms() throws MtjException;

	/**
	 * Method returns existing device platform by name.
	 * 
	 * @return
	 */
	public DevicePlatform getDevicePlatformByName(String name) throws MtjException;

	/**
	 * Method returns the device platform provider that owns the device platform.
	 * 
	 * @param devicePlatform
	 * @return
	 * @throws MtjException
	 */
	public DevicePlatformProvider getDevicePlatformProvider(DevicePlatform devicePlatform) throws MtjException;

	/**
	 * Method returns all devices of the device platform.
	 * 
	 * @param devicePlatformName
	 * @return
	 */
	public Device[] getDevices(String devicePlatformName) throws MtjException;

	/**
	 * Method returns DeviceDescription -object of the vendor/model.
	 * 
	 * @param vendor
	 * @param model
	 * @return
	 */
	public DeviceDescription getDeviceDescription(String vendor, String model) throws MtjException;

	/**
	 * Method returns DeviceDescription -objects that match to the device group.
	 * 
	 * @param group
	 * @return
	 */
	public DeviceDescription[] getMatchingDeviceDescriptions(DeviceGroup group) throws MtjException;

	/**
	 * Method returns all defined device groups.
	 * 
	 * @return
	 */
	public DeviceGroup[] getDeviceGroups() throws MtjException;

	/**
	 * Method returns all DeviceDescription -objects that match to the deviceConfig,
	 * deviceProfile and service api definitions.
	 * 
	 * @param deviceConfig
	 * @param deviceProfile
	 * @param apis
	 * @return
	 */
	public DeviceDescription[] getDevices(DeviceConfiguration deviceConfig,
			DeviceProfile deviceProfile, ServiceApi[] apis) throws MtjException;

	/**
	 * Method returns all Device -objects that match to the deviceConfig,
	 * deviceProfile and service api definitions and device platform type.
	 * 
	 * @param deviceConfig
	 * @param deviceProfile
	 * @param apis
	 * @param devicePlatformType
	 * @return
	 */
	public Device[] getTargetDevices(DeviceConfiguration deviceConfig,
			DeviceProfile deviceProfile, ServiceApi[] apis,
			PlatformType devicePlatformType) throws MtjException;

	/**
	 * Method returns all Runtime Platform Definition of the device platform's device.
	 * 
	 * @param devicePlatformName
	 * @param deviceName
	 * @return
	 */
	public RuntimePlatformDefinition[] getRuntimePlatforms(
			String devicePlatformName, String deviceName) throws MtjException;

	/**
	 * Method returns all deployment target Device -objects that match to the deviceConfig,
	 * deviceProfile and service api definitions and device platform type.
	 * 
	 * @param deviceConfig
	 * @param deviceProfile
	 * @param apis
	 * @param devicePlatformType
	 * @return
	 */
	public Device[] getDeployableDevices(DeviceConfiguration deviceConfig,
			DeviceProfile deviceProfile, ServiceApi[] apis,
			PlatformType devicePlatformType) throws MtjException;

	/**
	 * Method returns DevicePlatform's Device that name corresponds the parameter value
	 * 
	 * @param devicePlatformName
	 * @param deviceName
	 * @return
	 * @throws MtjException
	 */
	public Device getDevice(String devicePlatformName, String deviceName) throws MtjException;
	
	/**
	 * Method removes the DevicePlatform from the owner DevicePlatformProdiver if the provider 
	 * implements ToolkitSupport -interface.
	 * 
	 * @param devicePlatform
	 * @throws MtjException
	 */
	public void removeDevicePlatform(DevicePlatform devicePlatform) throws MtjException;
	
}
