/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/

package org.eclipse.mtj.extension.devide.core;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jdt.core.IClasspathContainer;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.api.project.Project;
import org.eclipse.mtj.api.project.TargetDevice;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.core.util.MtjEmfUtil;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.devide.Messages;
import org.eclipse.mtj.extension.devide.project.MtjProject;

/**
 * A classpath container implementation that reflects the classpath
 * provided by the platform definition currently associated with
 * an IJavaProject.
 * <p>
 * <b>Note:</b> This class/interface is part of an interim API that is still under development and expected to
 * change before reaching stability. It is being made available at this early stage to solicit feedback
 * from pioneering adopters on the understanding that any code that uses this API will almost 
 * certainly be broken as the API evolves.
 * </p>
 * Copyright (c) 2003-2005 Craig Setera<br>
 * All Rights Reserved.<br>
 * Licensed under the Eclipse Public License - v 1.0<p/>
 * <br>
 * $Revision: 1.1.1.1 $
 * <br>
 * $Date: 2006/08/30 18:51:15 $
 * <br>
 * @author Yang Liu
 * @author Craig Setera
 */

public class J2MEClasspathContainer implements IClasspathContainer {
	/** Name of the J2ME Classpath Container */
	public static final String J2ME_CONTAINER = Messages.JavaMEClasspathContainer_Container;
	public static final String J2ME_LIBRARY_LABEL = Messages.JavaMEClasspathContainer_Library;

	// The classpath container path and project
	private IJavaProject project;
	private IPath path;

	// The description of this classpath container.
	private String description;

	// The classpath entries cache.
	private IClasspathEntry[] cachedEntries;

	/**
	 * Construct a new classpath container for the specified project.  
	 * 
	 * @param project
	 * @param path
	 * @throws CoreException
	 */
	public J2MEClasspathContainer(IJavaProject project, IPath path)
			throws CoreException 
	{
		this.project = project;
		this.path = path;
	}

	public static IClasspathEntry[] getClasspathEntries(RuntimePlatformDefinition rpd) {

		if (rpd != null) {
			// Stash the classpath entries specified by the device
			return MtjEmfUtil.getClasspathEntries(rpd);
		} else {
			return new IClasspathEntry[0];
		}
	}
	
	public static IClasspathEntry[] getClasspathEntries(IJavaProject project) {
			RuntimePlatformDefinition rpd = null;
			try {
				rpd = getProjectRPD(project);
			} catch (MtjException e) {
				e.printStackTrace();
				return null;
			}

			if (rpd != null) {
				// Stash the classpath entries specified by the device
				return MtjEmfUtil.getClasspathEntries(rpd);
			} else {
				return new IClasspathEntry[0];
			}
	}

	/**
	 * 
	 * @throws MtjException 
	 * @see org.eclipse.jdt.core.IClasspathContainer#getClasspathEntries()
	 */
	public IClasspathEntry[] getClasspathEntries() {
		if (cachedEntries == null) {
			if (!project.exists())
			{
				// assume project is still being created, or we wouldn't be trying to get the classpath
				// so try to look at the temporary file created during project creation.
				try {
					Project openProject  = MtjProject.getTemporaryProjectData(project);  
																				
					TargetDevice targetDevice = openProject.getDefaultTargetDevice();
					RuntimePlatformDefinition rpd = targetDevice.getRuntimePlatform();
					return getClasspathEntries (rpd);
				} catch (MtjException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
					return new IClasspathEntry[0];
				}
			}
			cachedEntries = getClasspathEntries(project);
		}
		
		return cachedEntries;
	}

	/**
	 * @see org.eclipse.jdt.core.IClasspathContainer#getDescription()
	 */
	public String getDescription() {
		if (description == null) {
			this.description = J2ME_LIBRARY_LABEL + Messages.JavaMEClasspathContainer_Failed_get_library_information;

			RuntimePlatformDefinition rpd;
			try {
				rpd = getProjectRPD(project);
				if (rpd != null) {
					// Stash the description
					this.description = new String (
							J2ME_LIBRARY_LABEL + " [ " + rpd.getName() + " ]"); //$NON-NLS-1$ //$NON-NLS-2$
				}
			} catch (MtjException e) {
//				e.printStackTrace();
			}
		}

		return description;
	}

	/**
	 * @see org.eclipse.jdt.core.IClasspathContainer#getKind()
	 */
	public int getKind() {
		return K_APPLICATION;
	}

	/**
	 * @see org.eclipse.jdt.core.IClasspathContainer#getPath()
	 */
	public IPath getPath() {
		return path;
	}
	
	/**
	 * Return the device for the specified project or <code>null</code>
	 * if the project cannot be determined.
	 * 
	 * @param project
	 * @return
	 * @throws MtjException 
	 */
	private static RuntimePlatformDefinition getProjectRPD(IJavaProject project) throws MtjException {
		RuntimePlatformDefinition rpd = null;

		IMtjProject mtjproject = MtjProject.getMtjProject(project);
		if (mtjproject != null) {
			Project proj = mtjproject.getProjectData();
			if ( proj != null && proj.getDefaultTargetDevice() != null && 
				 proj.getDefaultTargetDevice().getRuntimePlatform() != null ) {
				rpd = proj.getDefaultTargetDevice().getRuntimePlatform();
			}
		}
		if (rpd == null)
		{
			try {
				Project openProject  = MtjProject.getTemporaryProjectData(project);
				TargetDevice targetDevice = openProject.getDefaultTargetDevice();
				rpd = targetDevice.getRuntimePlatform();
			} catch (MtjException e) {
			}
		}
		
		return rpd;
	}
}
