/**
 * Copyright (c) 2003-2004 Craig Setera
 * All Rights Reserved.
 * Licensed under the Eclipse Public License - v 1.0
 * For more information see http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.mtj.extension.preverify;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.List;

import org.eclipse.mtj.api.model.preverifier.IPreverificationPolicy;
import org.eclipse.mtj.api.model.preverifier.PreverificationError;
import org.eclipse.mtj.extension.preverify.internal.PreverificationClassNode;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassWriter;

/**
 * A class preverifier implementation
 * <p />
 * Copyright (c) 2003-2004 Craig Setera<br>
 * All Rights Reserved.<br>
 * Licensed under the Eclipse Public License - v 1.0<p/>
 * <br>
 * $Revision: 1.1.1.1 $
 * <br>
 * $Date: 2006/08/30 18:51:59 $
 * <br>
 * @author Craig Setera
 */
public class ClassPreverifier {
	private IPreverificationPolicy preverificationPolicy;

	/**
	 * Construct a new preverifier that uses the specified
	 * policy during preverification processing.
	 * 
	 * @param preverificationPolicy
	 */
	public ClassPreverifier(IPreverificationPolicy preverificationPolicy) {
		this.preverificationPolicy = preverificationPolicy;
	}

	/**
	 * Preverify the specified class file.
	 * 
	 * @param classStream
	 * @param classloader
	 * @return
	 * @throws IOException
	 */
	public PreverificationResults preverify(InputStream classStream, URL[] classpath)
			throws IOException
	{
		ClassLoader classLoader =
			new URLClassLoader(classpath, Thread.currentThread().getContextClassLoader());
		return preverify(classStream, classLoader);
	}
	
	/**
	 * Preverify the specified class file.
	 * 
	 * @param classStream
	 * @param classloader
	 * @return
	 * @throws IOException
	 */
	public PreverificationResults preverify(InputStream classStream, ClassLoader classloader)
			throws IOException
	{
		if (classStream == null) {
			throw new IllegalArgumentException(Messages.ClassPreverifier_ClassByteStreamNotNull);
		}

		PreverificationClassNode classNode = 
			new PreverificationClassNode(preverificationPolicy, classloader);

		// Do the visitation
		ClassReader classReader = new ClassReader(classStream);
		classReader.accept(classNode, false);
		classStream.close();
		
		// Collect the errors
		List errorList = classNode.getErrorList();
		PreverificationError[] errorArray =
			(PreverificationError[]) errorList.toArray(new PreverificationError[errorList.size()]);

		// Generate the results object
		PreverificationResults results = null;
		if (errorArray.length > 0) {
			results = new PreverificationResults(null, null, errorArray);
		} else {
			// The preverification writer does the real work
			ClassWriter classWriter = new ClassWriter(false);
			classNode.accept(classWriter);
			
			byte[] bytecode = classWriter.toByteArray();
			results = new PreverificationResults(classNode, bytecode, errorArray);
		}
		
		return results;
	}
}

