/*******************************************************************************
* Copyright (c) 2006 IBM Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
* Contributors:
*IBM - initial API and implementation
*******************************************************************************/
package org.eclipse.mtj.jad.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;

/**
 * A JadFileDescriptor represents the data found
 * in a JAD file.
 **/
public class JadFileDescriptor
		extends PropertyFile
{
	public ArrayList midletList;

	public String midpVersion;

	public static final String MIDP10 = "MIDP-1.0"; //$NON-NLS-1$
	public static final String MIDP20 = "MIDP-2.0"; //$NON-NLS-1$
	public static final String[] MIDP_VALID_VALUES = {MIDP10, MIDP20};
	public static final String MIDP_DEFAULT_VALUE = MIDP20;
	
	public static final String CLDC10 = "CLDC-1.0"; //$NON-NLS-1$
	public static final String CLDC11 = "CLDC-1.1"; //$NON-NLS-1$
	public static final String[] CLDC_VALID_VALUES = {CLDC10, CLDC11};
	public static final String CLDC_DEFAULT_VALUE = CLDC11;

	
	
	public final static String MIDLET_PREFIX = "MIDlet-"; //$NON-NLS-1$
	public final static String MICROEDITION_PREFIX = "MicroEdition-"; //$NON-NLS-1$	

	public static final int
		NAME = 0, /*+1.0, +2.0*/
		VERSION = 1, /*+1.0, +2.0*/
		VENDOR = 2, /*+1.0, +2.0*/
		DESCRIPTION = 3, /*1.0, 2.0*/
		JAR_SIZE = 4, /*1.0, 2.0*/
		JAR_URL = 5, /*1.0, 2.0*/
		ICON = 6, /*2.0*/
		INFO_URL = 7, /*2.0*/
		DATA_SIZE = 8, /*2.0*/
		PROFILE_NAME = 9, /*1.0, 2.0*/
		CONFIGURATION_NAME = 10, /*1.0, 2.0*/		
		PERMISSIONS = 11, /*2.0*/
		PERMISSIONS_OPT = 12, /*2.0*/
		INSTALL_NOTIFY = 13, /*2.0*/
		DELETE_NOTIFY = 14, /*2.0*/
		DELETE_CONFIRM = 15, /*2.0*/
		JAR_RSA_SHA1 = 16, /*2.0*/
		CERTIFICATE1= 17; /*2.0*/

    
    //To get the labels from properties, if new values added here update the MidpJad.xxx
	//in MidpResources.properties file - Sri!
	public static final String S_NAME = "MIDlet-Name"; //$NON-NLS-1$
	public static final String S_VERSION = "MIDlet-Version"; //$NON-NLS-1$
	public static final String S_VENDOR = "MIDlet-Vendor"; //$NON-NLS-1$
	public static final String S_DESCRIPTION = "MIDlet-Description"; //$NON-NLS-1$
	public static final String S_JAR_SIZE = "MIDlet-Jar-Size"; //$NON-NLS-1$
	public static final String S_JAR_URL = "MIDlet-Jar-URL"; //$NON-NLS-1$
	public static final String S_PROFILE_NAME = "MicroEdition-Profile"; //$NON-NLS-1$
	public static final String S_CONFIGURATION_NAME = "MicroEdition-Configuration"; //$NON-NLS-1$
	public static final String S_ICON = "MIDlet-Icon"; //$NON-NLS-1$
	public static final String S_INFO_URL = "MIDlet-Info-URL"; //$NON-NLS-1$			
	public static final String S_DATA_SIZE = "MIDlet-Data-Size"; //$NON-NLS-1$
	public static final String S_PERMISSIONS = "MIDlet-Permissions"; //$NON-NLS-1$			
	public static final String S_PERMISSIONS_OPT = "MIDlet-Permissions-Opt"; //$NON-NLS-1$			
	public static final String S_INSTALL_NOTIFY = "MIDlet-Install-Notify"; //$NON-NLS-1$			
	public static final String S_DELETE_NOTIFY = "MIDlet-Delete-Notify"; //$NON-NLS-1$	 
	public static final String S_DELETE_CONFIRM = "MIDlet-Delete-Confirm";//$NON-NLS-1$		
	public static final String S_JAR_RSA_SHA1 = "MIDlet-Jar-RSA-SHA1";	//$NON-NLS-1$	
	public static final String S_CERTIFICATE1_1 = "MIDlet-Certificate-1-1";	//$NON-NLS-1$	
	
	
	final static String[] midp10JadFields;
	final static String[] midp20JadFields;
	
	static {
		midp10JadFields = new String [] {
				S_NAME, 
				S_VERSION, 
				S_VENDOR, 
				S_DESCRIPTION, 
				S_JAR_SIZE, 
				S_JAR_URL, 
				S_ICON,
				S_INFO_URL,
				S_DATA_SIZE,				
				S_PROFILE_NAME, 
				S_CONFIGURATION_NAME				
			};		
		
		midp20JadFields = new String [] {
			S_NAME, 
			S_VERSION, 
			S_VENDOR, 
			S_DESCRIPTION, 
			S_JAR_SIZE, 
			S_JAR_URL, 
			S_ICON,
			S_INFO_URL,
			S_DATA_SIZE,
			S_PROFILE_NAME, 
			S_CONFIGURATION_NAME,
			S_PERMISSIONS,
			S_PERMISSIONS_OPT,
			S_INSTALL_NOTIFY,
			S_DELETE_NOTIFY,
			S_DELETE_CONFIRM,
			S_JAR_RSA_SHA1,
			S_CERTIFICATE1_1
		};
	}
	
	/**
	 * Returns the field names that are known by this version of MIDP
	 * 
	 * @return the field names
	 */
	public String [] getFieldNames() {
		return getFieldNames(midpVersion);
	}

	/**
	 * Returns the field names that are known by this version of MIDP
	 * 
	 * @param midpVersion the version.  Must be MidpVersionConstants.MIDP1 or 
	 * MidpVersionConstants.MIDP10 
	 * @return the field names
	 */	
	public static String [] getFieldNames(String midpVersion) {
		if (midpVersion.compareTo (MIDP20) == 0)
			return midp20JadFields;
		else
			return midp10JadFields;
	}
	
	private void initializeRequiredFields() {
		String [] fieldNames = getFieldNames();
		// Just put an empty string in the required field of a jad
		setProperty(fieldNames[NAME],""); //$NON-NLS-1$
		setProperty(fieldNames[VERSION],""); //$NON-NLS-1$
		setProperty(fieldNames[VENDOR],""); //$NON-NLS-1$
		setProperty(fieldNames[JAR_SIZE],""); //$NON-NLS-1$
		setProperty(fieldNames[JAR_URL],""); //$NON-NLS-1$
	}
		
	protected JadFileDescriptor(String midpVersion) {
		super();
		this.midpVersion = midpVersion;
		midletList = new ArrayList();
		initializeRequiredFields();
	}
	
	/**
	 * <code>index</code> is the MIDlet number, 1-based.
	 **/
	public MIDletDescriptor getMIDlet(int index) {
		return (MIDletDescriptor) midletList.get(index - 1);
	}

	/**
	 * Return number of MIDlets.
	 **/
	public int getMIDletCount() {
		return midletList.size();
	}

	/**
	 * <code>index</code> is the MIDlet number, 1-based.
	 **/
	public void setMIDlet(int index,MIDletDescriptor d) {
		String key = getMIDletKey(index);
		d.setKey(key);
		setPropertyLine(d);
		setupMIDlets();
	}

	public void addMIDlet(String name, String icon, String classname) {
		MIDletDescriptor d = new MIDletDescriptor(name, icon, classname);
		addMIDlet(d);
	}

	public void addMIDlet(MIDletDescriptor d) {
		String key = getMIDletKey(midletList.size() + 1); 
		d.setKey(key);
		setPropertyLine(d);
		midletList.add(d);
	}
	
	public static String getMIDletKey(int index) {
		return MIDLET_PREFIX + index;
	}
	
	public static boolean isMidletNrKey(String key) {
		try {
            String numberString =
                key.substring(
                    MIDLET_PREFIX.length(),
                    key.length());
            
			Integer.parseInt(numberString);
		}
		catch (Exception e) {
			// not a midlet entry
			return false;
		}
		return true;
	}
	
	
	/**
	 * Removes MIDlet # <code>index</code>.
	 **/
	public void removeMIDlet(int index) {
		int max = midletList.size();
				
		super.removeProperty(getMIDletKey(index));

		// Need to put some order in the Hashtable		
		super.removeProperty(getMIDletKey(max));
		for (int i = index; i < max; i++) {
			super.setProperty(getMIDletKey(i), ((MIDletDescriptor)midletList.get(i)).getValue());
		}

		midletList.remove(index - 1);
	}

	public void setProperty(int index, String value) {
		setProperty(getFieldNames()[index], value);
	}

	/**
	 * @see PropertyFile.getProperty
	 * 
	 * @param index
	 * @return
	 */
	public String getProperty(int index) {
		return getProperty(getFieldNames()[index]);
	}
	
	/**
	 * Returns all keys that do not start with "Midlet-"
	 */
	public String [] getCustomKeys() {
		
		Set keySet = keySet();
		ArrayList list = new ArrayList(keySet.size());
		
		for (Iterator iter = keySet.iterator(); iter.hasNext();) {
			String key = (String) iter.next();
			if (isCustomKey(key)) {
				list.add(key);
			}		
		}
		String [] keys = new String[list.size()];
		return (String[])list.toArray(keys);
	}
	
	public boolean isCustomKey(String key) {
		if (key.startsWith(MIDLET_PREFIX) || key.startsWith(MICROEDITION_PREFIX)) {
			return false;
		}
		return true;
	}
	
	/**
	 * Swaps MIDlets #<code>index1</code> and #<code>index2</index>,
	 * the indices being 1-based.
	 **/
	public void swap(int index1, int index2) {
		int length = midletList.size();

		if (index1 - 1 > length || index2 - 1 > length) {
			return;
		}
		String mid1 = getMIDletKey(index1), mid2 = getMIDletKey(index2);

		MIDletDescriptor d1 = (MIDletDescriptor) getPropertyLine(mid1);
		MIDletDescriptor d2 = (MIDletDescriptor) getPropertyLine(mid2);
		
		if (d1 == null || d2 == null) {
			return;
		}

		//Swap values
		String value1 = d1.getValue();
		d1.setValue(d2.getValue());
		d2.setValue(value1);
		
		// Update vector
		setupMIDlets();
	}

	/**
	 * Convenience method that returns the set of
	 * all the classes referenced by this descriptor.
	 * The vector does not contain duplicate entries.
	 **/
	public Set getMIDletClassnames() {
		Set set = new HashSet();
		for (int i=0,max=midletList.size();i<max;i++) {
			set.add(((MIDletDescriptor)midletList.get(i)).classname);
		}
		return set;
	}

	/**
	 * Get a descriptor that is appropriate for the given project (is configured
	 * with the right MIDP version.
	 * 
	 * @param project the project
	 * @return an appropriate JadFileDescriptor
	 */
	public static JadFileDescriptor getDescriptor(IProject project) {
		JadFileDescriptor descriptor = new JadFileDescriptor(MIDP20);
		return descriptor;
	}
	
	public static JadFileDescriptor getDescriptor() {
		JadFileDescriptor descriptor = new JadFileDescriptor(MIDP20);
		return descriptor;
	}

	/**
	 * @return JadFileDescriptor an instance describing
	 * <code>contents</code>
	 */
	public static JadFileDescriptor getDescriptor(IFile file) throws CoreException, IOException {
		JadFileDescriptor instance = new JadFileDescriptor(MIDP20);		
		// dont retain comments until I'm sure that jad files actually support them!
		instance.loadInto(file, false);
		return instance;
	}
	
	/**
	 * Sets up the MIDlet vector.
	 */
	public void setupMIDlets() {
		midletList.clear();
		for (int i = 0; ; i++) {
			String key = getMIDletKey(i + 1);
			PropertyLine line = getPropertyLine(key);

			if (line != null) {
				MIDletDescriptor d = MIDletDescriptor.convertFrom(line);
				if (line != d) { // if it's already a MIDletDescriptor, do nothing
					d.setKey(key);  
					setPropertyLine(d);
				}
				midletList.add(i, d);				
			} 
			else {
				break;
			}
		}		
	}
	
	
	private boolean isMidpX (String version)
	{
		String configuration = getProperty(PROFILE_NAME);
		if (configuration == null)
			return false;
		if (configuration.compareTo (version) == 0)
			return true;
		else
			return false;
	}
	
	public boolean isMidp20 ()
	{
		return isMidpX (MIDP20);
	}
	
	public boolean isMidp10 ()
	{
		return isMidpX (MIDP10);
	}
	
	private boolean isCldcX (String version)
	{
		String configuration = getProperty(CONFIGURATION_NAME);
		if (configuration == null)
			return false;
		if (configuration.compareTo (version) == 0)
			return true;
		else
			return false;
	}
	
	public boolean isCldc10 ()
	{
		return isCldcX (CLDC10);
	}

	public boolean isCldc11 ()
	{
		return isCldcX (CLDC11);
	}
	/**
	 * Use getDescriptor(IFile) instead.
	 * 
	 * @see com.ibm.ive.j9.util.PropertyFile#loadInto(BufferedReader, boolean)
	 */
//	public void loadInto(BufferedReader reader, boolean retainComments)
//		throws IOException {
//		super.loadInto(reader, retainComments);
//		setupMIDlets();
//	}

	/**
	 * Use getDescriptor(IFile) instead.
	 * 
	 * @see com.ibm.ive.j9.util.PropertyFile#loadInto(IFile, boolean)
	 */
//	public void loadInto(IFile file, boolean retainComments)
//		throws CoreException, IOException {
//		super.loadInto(file, retainComments);
//		setupMIDlets();		
//	}

	/**
	 * Use getDescriptor(IFile) instead.
	 * 
	 * 
	 */
//	public void loadInto(InputStream stream, boolean retainComments)
//		throws IOException {
//		super.loadInto(stream, retainComments);
//		setupMIDlets();
//	}
	
	/**
	 * 
	 */
	public Object clone() {
		JadFileDescriptor clone = new JadFileDescriptor(midpVersion);
		for (Iterator i = lineIterator(); i.hasNext();) {
			PropertyLine line = (PropertyLine) i.next();
			clone.setPropertyLine((PropertyLine)line.clone());			
		}
		clone.setupMIDlets();
		return clone;						
	}

	/**
	 * 
	 */
	public void removeProperty(String key) {
		super.removeProperty(key);
		if (isMidletNrKey(key)) {
			setupMIDlets();
		}
	}

	/**
	 * @
	 */
	public void setProperty(String key, String value) {
		super.setProperty(key, value);
		if (isMidletNrKey(key)) {
			setupMIDlets();
		}
	}
}