/*******************************************************************************
* Copyright  2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/

package org.eclipse.mtj.extension.rpm.preferences;

import java.util.logging.Logger;

import org.eclipse.emf.common.util.EList;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.project.TargetDevice;
import org.eclipse.mtj.api.runtimes.RuntimePlatform;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.core.ui.util.DialogController;
import org.eclipse.mtj.extension.dm.ui.preferences.AddDevicePlatformDialog2;
import org.eclipse.mtj.extension.rpm.Messages;
import org.eclipse.mtj.extension.rpm.model.MtjServiceHandler;
import org.eclipse.mtj.extension.rpm.ui.preferences.ModifyRuntimeComposite;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;

public class ModifyRuntimeController implements DialogController, ICheckStateListener, SelectionListener {
	private static final Logger log = Logger.getLogger(ModifyRuntimeController.class.getName());
	
	private MtjServiceHandler mtjServiceHandler;	
	private RuntimePlatform selectedRuntime;
	private DevicePlatform[] devicePlatforms;
	private RuntimePlatform[] runtimes;
	private DevicePlatform selectedDevicePlatform;	
	private ModifyRuntimeComposite composite;
	
	private CheckboxTableViewer sdkTableViewer;
	private Text runtimeNameText;
	private Text runtimeDescriptionText;
	private Button importButton;

	private boolean write = false;
	
	public ModifyRuntimeController(ModifyRuntimeComposite composite, RuntimePlatform[] runtimes ,MtjServiceHandler mtjServiceHandler) {
		this(composite, runtimes, mtjServiceHandler, null, false);
	}
	
	public ModifyRuntimeController(ModifyRuntimeComposite composite, RuntimePlatform[] runtimes ,MtjServiceHandler mtjServiceHandler, boolean write) {
		this(composite, runtimes, mtjServiceHandler, null, write);
	}
	
	public ModifyRuntimeController(ModifyRuntimeComposite composite, RuntimePlatform[] runtimes , MtjServiceHandler mtjServiceHandler, RuntimePlatform selectedRuntime, boolean write) {
		this.mtjServiceHandler = mtjServiceHandler;
		this.selectedRuntime = selectedRuntime;
		this.runtimes = runtimes;
		
		this.composite = composite;
		this.sdkTableViewer = composite.getSdkTableViewer();		
		this.runtimeNameText = composite.getNameText();
		this.runtimeDescriptionText = composite.getDescriptionText();
		this.importButton = composite.getImportButton();
		
		this.sdkTableViewer.addCheckStateListener(this);
		this.importButton.addSelectionListener(this);
		
		this.write = write;
		
		setData();
		createSdkList();	

		Object[] o = this.sdkTableViewer.getCheckedElements();

		if(selectedRuntime == null) 
			for(Object dp: o) {
				runtimeNameText.setText(createRuntimeName((DevicePlatform)dp,runtimes));
				runtimeDescriptionText.setText(((DevicePlatform)dp).getName());
			}
	}
	
	private void setData() {		
		devicePlatforms = this.mtjServiceHandler.getDevicePlatforms();

		if(selectedRuntime != null) {
			runtimeNameText.setText(selectedRuntime.getName());
			runtimeDescriptionText.setText(selectedRuntime.getDescription());
		} 
	}
	
	private static boolean containsName(String  name, RuntimePlatform[] runtimes) {
		for(RuntimePlatform rp: runtimes) {
			if(rp.getName().equals(name)) {
				return true;
			}
		}
		return false;
	}
	
	public void widgetDefaultSelected(SelectionEvent e) {}
	public void widgetSelected(SelectionEvent e) {
		if(e.widget == this.importButton) {
			AddDevicePlatformDialog2 dialog = new AddDevicePlatformDialog2(composite.getShell(), Messages.ModifyRuntimeController_AddMobileDevicePlatform);
			dialog.setTitleBarLabel(Messages.ModifyRuntimeController_DevicePlatformTitle);

			dialog.setBlockOnOpen(true);

			if (dialog.open() == Dialog.OK) {				
				devicePlatforms = this.mtjServiceHandler.getDevicePlatforms();
				this.sdkTableViewer.setInput(devicePlatforms);
				if(devicePlatforms != null)
					if(devicePlatforms.length > 0)
						this.sdkTableViewer.setChecked(devicePlatforms[0], true);
				this.sdkTableViewer.refresh();
			}
		}
	}
	
	public void handleOK() {
		if(selectedRuntime != null) {
			selectedRuntime.setName(this.runtimeNameText.getText());
			selectedRuntime.setDescription(this.runtimeDescriptionText.getText());
			
			Object[] o = this.sdkTableViewer.getCheckedElements();
			if(o != null) {
				if(o.length > 0) {
					DevicePlatform dp = (DevicePlatform) o[0];
					String dpString = ((TargetDevice) selectedRuntime.getTargetDevices().get(0)).getDevicePlatformName();
					if(!dpString.equals(dp.getName())) {
						Device d = (Device)dp.getDevices().get(0);
						RuntimePlatformDefinition rpd = (RuntimePlatformDefinition) d.getRuntimePlatformDefinitions().get(0);
						TargetDevice td = mtjServiceHandler.createTargetDevice( dp, rpd, d);
						selectedRuntime.getTargetDevices().clear();
						selectedRuntime.getTargetDevices().add(td);
					}					
				}
			}			
		} else {
			Object[] o = this.sdkTableViewer.getCheckedElements();
			if(o.length > 0)
				if(o[0] instanceof DevicePlatform) {
					DevicePlatform d = (DevicePlatform)o[0];
					EList l = d.getDevices();
					Device device = (Device) l.get(0);
					EList rtl = device.getRuntimePlatformDefinitions();
					RuntimePlatformDefinition rpd = (RuntimePlatformDefinition)rtl.get(0);
					RuntimePlatform[] prevRuntimes = mtjServiceHandler.getRuntimePlatforms();
					this.mtjServiceHandler.setSelectedDP(d);
					this.mtjServiceHandler.setSelectedRPD(rpd);

					this.mtjServiceHandler.addRuntime(this.runtimeNameText.getText(), this.runtimeDescriptionText.getText());					

					if(write) {
						RuntimePlatform rp = this.mtjServiceHandler.getAddedRuntime();
	
						if(prevRuntimes == null) 
							prevRuntimes = new RuntimePlatform[0]; 
						RuntimePlatform[] newRuntimes = new RuntimePlatform[prevRuntimes.length+1]; 
						
						for (int i = 0; i < prevRuntimes.length; i++) {
							newRuntimes[i] = prevRuntimes[i];
						}
						
						newRuntimes[prevRuntimes.length] = rp;				
						MtjServiceHandler.storeRuntimes(newRuntimes);
					}
				}
		}
	}
		
	public void checkStateChanged(CheckStateChangedEvent event) {
		//boolean checkState = event.getChecked();
		Object o = event.getElement();
		if(o instanceof DevicePlatform) {
			if(o == selectedDevicePlatform && o != null) {
				this.sdkTableViewer.setChecked(o, true);
			} else {
				DevicePlatform dp = (DevicePlatform) o;
				this.sdkTableViewer.setAllChecked(false);
				this.sdkTableViewer.setChecked(dp, true);				
				selectedDevicePlatform = dp;
				
				runtimeNameText.setText(createRuntimeName(dp,runtimes));
				this.runtimeDescriptionText.setText(dp.getName());
			} 
		}						
	}

	private static String createRuntimeName(DevicePlatform dp, RuntimePlatform[] runtimes) {
		int counter = 0;
		String name = createRuntimeName(dp);
		String runtimeName = name;
		
		while(containsName(runtimeName,runtimes)) {
			counter++;
			runtimeName = name+counter;
		}

		return runtimeName;
	}
	
	private static String createRuntimeName(DevicePlatform dp) {
		String name = dp.getName();
		String rtName = null;
		
		if(name.contains("Sony Ericsson SDK 2.2.3")) {  //$NON-NLS-1$
			return "SonyEricsson_2.2.3_Runtime";  //$NON-NLS-1$
		}
		if(name.contains("Sony Ericsson J2ME SDK 1.2")) { //$NON-NLS-1$
			return "SonyEricsson_1.2_Runtime";  //$NON-NLS-1$
		}
		if(name.contains("Sun Java(TM) Wireless Toolkit 2.5")) { //$NON-NLS-1$
			return "WTK_2.5_Runtime"; //$NON-NLS-1$
		}
		if(name.contains("Sun Java Wireless Toolkit  2.3")) { //$NON-NLS-1$
			return "WTK_2.3_Runtime"; //$NON-NLS-1$
		}
		if(name.contains("Sun") || name.contains("SUN") || name.contains("sun")) {  //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			return "WTK_Runtime"; //$NON-NLS-1$
		}
		if(name.contains("Nokia Prototype SDK 4.0")) { //$NON-NLS-1$
			return "Nokia_Prototype_4.0_Runtime"; //$NON-NLS-1$
		}
		if(name.contains("S60") || name.contains("s60")) { //$NON-NLS-1$ //$NON-NLS-2$
			return "S60_Runtime"; //$NON-NLS-1$
		}
		if(name.contains("S40") || name.contains("s40")) { //$NON-NLS-1$ //$NON-NLS-2$
			return "S40_Runtime"; //$NON-NLS-1$
		}
		if(name.contains("Nokia") || name.contains("NOKIA") || name.contains("nokia")) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			return "Nokia_Runtime"; //$NON-NLS-1$
		}
		
		return "Runtime"; //$NON-NLS-1$
	}
	
	private void createSdkList() {
		class tlp extends LabelProvider implements ITableLabelProvider {
			public Image getColumnImage(Object element, int columnIndex) {
				  return null;
			}
			
			public String getColumnText(Object element, int columnIndex) {
				if(element instanceof DevicePlatform) {
					DevicePlatform dp = (DevicePlatform) element;
					if(columnIndex == 0) { return ""; } //$NON-NLS-1$
					else if(columnIndex == 1) { return dp.getName(); }
					else if(columnIndex == 2) { return dp.getVersion(); }
					else if(columnIndex == 3) { return dp.getVendor(); }
				}
				return null;
			}			
		}
		
		class Provider implements IStructuredContentProvider 
		{
			public Object[] getElements(Object o) {
				if(o instanceof DevicePlatform[]) {
					return (DevicePlatform[])o;
				}
				return null;
			}
			
			public void dispose() {
			}
			
			public void inputChanged(Viewer viewer,
	                Object oldInput,
	                Object newInput) {
			}
		}

		createTableColumn(sdkTableViewer.getTable(),0,20,""); //$NON-NLS-1$
		createTableColumn(sdkTableViewer.getTable(),1,170,Messages.ModifyRuntimeController_Name);
		createTableColumn(sdkTableViewer.getTable(),2,40,Messages.ModifyRuntimeController_Version);
		createTableColumn(sdkTableViewer.getTable(),3,80,Messages.ModifyRuntimeController_Vendor);

		this.sdkTableViewer.setLabelProvider(new tlp());
		this.sdkTableViewer.setContentProvider(new Provider());
		this.sdkTableViewer.setInput(this.devicePlatforms);

		if(selectedRuntime != null)
			this.setSelectedDevicePlatform();
		else
			selectFirst();		
	}
	
	private void selectFirst() {
		DevicePlatform dp = (DevicePlatform) this.sdkTableViewer.getElementAt(0);
		
		if(dp != null) {
			this.sdkTableViewer.setChecked(dp,true);
		}
	}
	
	private void setSelectedDevicePlatform() {
		TargetDevice td = (TargetDevice) this.selectedRuntime.getTargetDevices().get(0);
		String dp = td.getDevicePlatformName();
		
		for(int i = 0; i < devicePlatforms.length; i++) {
			if(devicePlatforms[i].getName().equals(dp)) {
				this.sdkTableViewer.setChecked(devicePlatforms[i], true);
			}
		}		
	}
	
	private static void createTableColumn(Table table, int colIndex, int weight, String header) {
		TableColumn tableColumn = new TableColumn(table, SWT.NONE, colIndex);
		tableColumn.setResizable(true);
		tableColumn.setText(header);				
		tableColumn.setWidth(weight);
	}
}


