/*******************************************************************************
* Copyright (c) 2005,2006 Nokia Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
 * Additional Contributors -
 * 		Kevin Horowitz (IBM Corp) - Update javadoc
 *******************************************************************************/
package org.eclipse.mtj.api.extension;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.mtj.exception.MtjException;

/**
 * The Security Managment Provider provides the interface to interact with Key Stores.
 */
public interface SecurityManagement extends MtjExtension {
	
	/**
	 * Accesses an existing keystore, opens the keystore and 
	 * returns a string array containing the key aliases within the 
	 * keystore.
	 * 
	 * @param keyStore
	 * @param storePass
	 * @param monitor
	 * @return String[]
	 * @throws MtjException
	 */
	public String [] openKeyStore( String keyStore, String storePass, IProgressMonitor monitor) throws MtjException;		
	
	/**
	 * Returns the keystore type used by keystore.
	 * @return String
	 * @throws MtjException
	 */
	public String getStoreType() throws MtjException;

	/**
	 * Returns the current key alias.
	 * Calling classes will set this value based on user selection from the keystore.
	 * @return key alias that is currently being used.
	 * @throws MtjException
	 */
	public String getAliaskey() throws MtjException;

	/**
	 * Returns the current keystore password.
	 * @return
	 * @throws MtjException
	 */
	public String getPassWrd() throws MtjException;

	/**
	 * Returns the location of the current keystore.
	 * @return
	 * @throws MtjException
	 */
	public String getKeyStoreNameLoc() throws MtjException; 

	/**
	 * Returns the validity of the current key pair.
	 * @return
	 * @throws MtjException
	 */
	public String getValidity() throws MtjException;
	
	/**
	 * Sets the type of keystore to generate.
	 * @param storeType
	 * @throws MtjException
	 */
	public void setStoreType(String storeType) throws MtjException;

	/**
	 * Sets the alias for the current key pair.
	 * @param aliasKey
	 * @throws MtjException
	 */
	public void setAliaskey(String aliasKey) throws MtjException;

	/**
	 * Sets the password for the current keystore.
	 * @param passWrd
	 * @throws MtjException
	 */
	public void setPassWrd(String passWrd) throws MtjException;

	/**
	 * Sets the Key Store name location.
	 * @param keyStoreNameLoc
	 * @throws MtjException
	 */
	public void setKeyStoreNameLoc(String keyStoreNameLoc) throws MtjException; 

	/**
	 * Sets the validity period for the certificate.
	 * @param validity
	 * @throws MtjException
	 */
	public void setValidity(String validity) throws MtjException;
	

	/**
	 * Allows users of this class to set the keystore values.
	 * @param loc
	 * @param alias
	 * @param psswd
	 * @param strtype
	 * @throws MtjException
	 */
	public void setValues (String loc, String alias, String psswd,String strtype) throws MtjException;
		
	/**
	 * Resets all values including keystore, password, alias key, etc
	 * @throws MtjException
	 */
	public void resetValues () throws MtjException;	  
	
	/**
	 * Return true if the keystore name and location were set during this session.
	 * @return true if a keystore name and location were set during this session, false otherwise.
	 */
	public boolean isKeyStoreSelected () throws MtjException;
	
	/**
	 * Generates a new CSR.
	 * @param certFile - location and name of file to generate
	 * @param monitor
	 * @return - True if success, otherwise false.
	 * @throws MtjException
	 */
	public boolean generateCSR( String certFile, IProgressMonitor monitor) throws MtjException;
	
	/**
	 * Imports signed certificate to current keystore.
	 * 
	 * @param certFile - location of signed certificate to import
	 * @param monitor
	 * @return - True if success, otherwise false.
	 * @throws MtjException
	 */
	public boolean importSignedCert( String certFile, IProgressMonitor monitor) throws MtjException;

	/**
	 * Deletes the current key pair, from the current keystore
	 * 
	 * @param monitor
	 * @return - True if success, otherwise false.
	 * @throws MtjException
	 */
	public boolean deleteKey(IProgressMonitor monitor ) throws MtjException;
	
	/**
	 * Changes the password of the current keystore.
	 * 
	 * @param newStorePass - Changes the keystore password to newStorePass.
	 * @param storePass - Previous keystore password.
	 * @param monitor
	 * @return - True if success, otherwise false. 
	 * @throws MtjException
	 */
	public boolean changeStorePassword(String newStorePass, String storePass, IProgressMonitor monitor) throws MtjException;
	
	/**
	 * Creates a new key pair with the information passed in. Attaches the new key pair to the current keystore.
	 * 
	 * @param alias - New key pair alias.
	 * @param commonName - New key pair common name.
	 * @param orgUnit - New key pair organization unit name.
	 * @param orgName - New key pair organization name.
	 * @param localityName - New key pair locality name.
	 * @param stateName - New key pair state name.
	 * @param country - New key pair country name.
	 * @param monitor - Progress monitor. 
	 * @return - true is success, otherwise false.
	 * @throws MtjException
	 */
	public boolean createNewKey(String alias, String commonName, String orgUnit, String orgName, String localityName, String stateName, String country, IProgressMonitor monitor) throws MtjException;

	/**
	 * Get the certificates associated with the alias key/keystore.
	 * @param monitor
	 * @return
	 * @throws MtjException
	 */
	public String getCertificateInfo(IProgressMonitor monitor) throws MtjException;
	
	/**
	 * Implementations that rely on an external security tool,
	 * are responsible for persistant storage of the tool location value.
	 * This method should return the location of the tool. Should never return null.
	 * @param monitor
	 * @return - Tool location.
	 * @throws MtjException
	 */
	public String getToolLocation(IProgressMonitor monitor) throws MtjException;
	
	/**
	 * Implementations that rely on an external security tool,
	 * are responsible for persistant storage of the tool location value. 
	 * This method should contain the code to store the location.
	 * @param loc - Directory where tool resides.
	 * @param monitor 
	 * @throws MtjException
	 */
	public void storeToolLocation(String loc, IProgressMonitor monitor) throws MtjException;
}
