/*******************************************************************************
 * Copyright (c) 2003,2006 Craig Setera and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM - Minor changes
 *    Nokia - Minor changes
 *******************************************************************************/

package org.eclipse.mtj.api.model.preverifier;

import org.eclipse.mtj.core.Messages;

/**
 * The description of an error that occurred during preverification.
 * <p />
 * Copyright (c) 2003-2004 Craig Setera<br>
 * All Rights Reserved.<br>
 * Licensed under the Eclipse Public License - v 1.0<p/>
 * <br>
 * $Revision: 1.2 $
 * <br>
 * $Date: 2006/09/07 13:56:26 $
 * <br>
 * @author Craig Setera
 */

// TODO : This needs to be restructured using MessageFormat for internationalization
public class PreverificationError {
	// The error information
	private PreverificationErrorType errorType;
	private PreverificationErrorLocation errorLocation;
	private String detail;
	
	/**
	 * Construct a new error.
	 * 
	 * @param type
	 * @param location
	 * @param detailMessage
	 */
	public PreverificationError(
		PreverificationErrorType type, 
		PreverificationErrorLocation location, 
		String detailMessage) 
	{
		super();

		this.detail = detailMessage;
		this.errorLocation = location;
		this.errorType = type;
	}

	/**
	 * Returns the detailed message concerning this error or <code>null</code>
	 * if no detail message has been specified.
	 * 
	 * @return Returns the detail.
	 */
	public String getDetail() {
		return detail;
	}

	/**
	 * Returns the location where the error occurred.
	 * 
	 * @return Returns the error location.
	 */
	public PreverificationErrorLocation getLocation() {
		return errorLocation;
	}

	/**
	 * Returns the type of the error.
	 * 
	 * @return Returns the error type.
	 */
	public PreverificationErrorType getType() {
		return errorType;
	}
	
	/**
	 * Returns preverification error's description
	 * 
	 * @return
	 */
	public String getErrorText() {
		StringBuffer errorText = new StringBuffer();
		
		// If details were passed in, we just use that outright
		String detail = getDetail();
		if ((detail != null) && (detail.length() > 0)) {
			errorText.append(detail);
		} else {
			appendErrorInformation(errorText);
		}
		
		return errorText.toString();
	}
	
	/**
	 * Append the error information specific to the
	 * preverification error.
	 * 
	 * @param errorText
	 */
	private void appendErrorInformation(
		StringBuffer errorText) 
	{
		appendErrorTypeInformation(errorText);
		
		if (getType() != PreverificationErrorType.FINALIZERS) {
			appendErrorLocationInformation(errorText);
		}
	}

	/**
	 * Append the error type information to the text.
	 * 
	 * @param errorText
	 */
	private void appendErrorTypeInformation(
		StringBuffer errorText) 
	{
		// No detailed message was provided, so we will build one
		// up
		switch (getType().getErrorCode()) {
			case PreverificationErrorType.FINALIZERS_CODE:
				errorText.append(Messages.PreverificationError_0);
				break;
			
			case PreverificationErrorType.FLOATING_POINT_CODE:
				errorText.append(Messages.PreverificationError_1);
				break;
				
			case PreverificationErrorType.MISSING_TYPE_CODE:
				errorText.append(Messages.PreverificationError_2);
				break;
				
			case PreverificationErrorType.NATIVE_CODE:
				errorText.append(Messages.PreverificationError_3);
				break;
				
			case PreverificationErrorType.UNKNOWN_ERROR_CODE:
				errorText.append(Messages.PreverificationError_4);
				break;
		}
	}

	/**
	 * Append the error type information to the text.
	 * 
	 * @param errorText
	 */
	private void appendErrorLocationInformation(
		StringBuffer errorText) 
	{
		IClassErrorInformation classInfo = getLocation().getClassInformation();
		IMethodErrorInformation methodInfo = getLocation().getMethodInformation();
		IFieldErrorInformation fieldInfo = getLocation().getFieldInformation();
		
		switch (getLocation().getLocationType().getTypeCode()) {
			case PreverificationErrorLocationType.CLASS_DEFINITION_CODE:
				errorText
					.append(Messages.PreverificationError_5)
					.append(classInfo.getName());
				break;

			case PreverificationErrorLocationType.CLASS_FIELD_CODE:
				errorText
					.append(Messages.PreverificationError_6)
					.append(fieldInfo.getName());
				break;

			case PreverificationErrorLocationType.METHOD_SIGNATURE_CODE:
				errorText
					.append(Messages.PreverificationError_7)
					.append(methodInfo.getName());
				break;

			case PreverificationErrorLocationType.METHOD_FIELD_CODE:
				errorText
					.append(Messages.PreverificationError_8)
					.append(methodInfo.getName())
					.append(Messages.PreverificationError_9)
					.append(classInfo.getName());
				break;

			case PreverificationErrorLocationType.METHOD_INSTRUCTION_CODE:
				errorText
					.append(Messages.PreverificationError_10)
					.append(methodInfo.getName());
				break;
		}
	}

}
